#pragma once

#include "favourite_address_advisor.h"
#include "user_context.h"

#include <drive/backend/processors/common_app/fetcher.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/additional_service_session.h>
#include <drive/backend/device_snapshot/snapshot.h>

class TUserSessionsContext {
public:
    class TSessionContext {
    private:
        const NDrive::IServer* Server = nullptr;

        mutable TMaybe<TBillingSession::TBillingCompilation> BillingCompilation;

        NFavouriteAddressAdvisor::IAdvisor::TPtr FavouriteAddressAdvisorPtr;

    private:
        R_READONLY(TVector<TCommonImageData>, Images);
        R_READONLY(TAtomicSharedPtr<const ISession>, Session);
        R_READONLY(TAtomicSharedPtr<IOffer>, SharedSessionOffer);
        R_READONLY(TVector<TDBTag>, SessionTags);
        R_READONLY(TRTDeviceSnapshot, Snapshot);
        R_READONLY(TMaybe<TTaggedObject>, TagsSnapshot);
        R_READONLY(TDBTag, ModelTag);
        R_READONLY(TDBTag, DelegationTag);
        R_READONLY(bool, ContextClosed, false);
        R_READONLY(bool, FuelingAbility, false);
        R_READONLY(bool, TankerFuelingAbility, false);
        R_READONLY(bool, ServicingEnabled, false);
        R_READONLY(bool, CarReplaceAbility, false);
        R_OPTIONAL(TInstant, AcceptanceFinished);
        R_OPTIONAL(TInstant, AcceptanceStarted);
        R_OPTIONAL(TPaymentsData, PaymentsData);
        R_OPTIONAL(bool, ReportObject);

    private:
        bool GetCurrentMultiBill(TMultiBill& mBill) const;

    public:
        const ISession* operator->() const {
            return Session.Get();
        }

        const TBillingSession::TBillingCompilation* GetBillingCompilation() const;
        IOffer::TPtr GetOffer() const;

        static void BuildCarReport(NJson::TJsonWriter& carReport, const bool closed, const TString& objectId, const bool fuelingAbility, bool tankerFuelingAbility, const TCarsFetcher& fetcher, TUserPermissions::TPtr permissions, const TSet<TString>& supportAllowedPhotos = {});

        void BuildCarReport(NJson::TJsonWriter& carReport, const TCarsFetcher& fetcher, TUserPermissions::TPtr permissions, const TSet<TString>& supportAllowedPhotos = {}) const {
            const auto& objectId = Session->GetObjectId();
            if (objectId) {
                BuildCarReport(carReport, ContextClosed, objectId, FuelingAbility, TankerFuelingAbility, fetcher, permissions, supportAllowedPhotos);
            }
        }

        TSessionContext(TAtomicSharedPtr<const ISession> session, TMaybe<TPaymentsData>&& payments, const NDrive::IServer* server)
            : Server(server)
            , Session(session)
            , Snapshot(Server->GetSnapshotsManager().GetSnapshot(session->GetObjectId()))
            , ContextClosed(Session->GetClosed())
            , PaymentsData(payments)
        {
        }

        bool Initialize(NDrive::TEntitySession& session, const TCarsFetcher& fetcher, const TMap<TString, TDBTag>& delegationCarTags, NFavouriteAddressAdvisor::IAdvisor::TPtr favouriteAddressAdvisorPtr, bool fetchPhotos);

        bool IsServiceFuelingPossible(const TUserCurrentContext* userCurrentContext) const;

        IEventsSession<TCarTagHistoryEvent>::TConstPtr GetEventsSession() const;

        TConstDBTag GetCurrentSessionTag() const {
            auto eventsSession = GetEventsSession();
            auto optionalLastEvent = eventsSession ? eventsSession->GetLastEvent() : Nothing();
            return optionalLastEvent && optionalLastEvent->HasData() ? *optionalLastEvent : TConstDBTag();
        }

        bool IsAccepted() const {
            return !!AcceptanceFinished;
        }
        bool ShouldReportObject() const {
            return ReportObject.GetOrElse(true);
        }

        NJson::TJsonValue GetReport(ELocalization locale, const TUserCurrentContext* userCurrentContext, const TUserSessionsContext& userSessionsContext, bool needBill, NDriveSession::TReportTraits reportTraits, IRequestProcessor::EApp app) const;
        NJson::TJsonValue GetReportDelegation() const;
        NJson::TJsonValue GetReportSession(ELocalization locale, bool needBill, NDriveSession::TReportTraits reportTraits) const;
        NJson::TJsonValue GetReportObjectDiff(ELocalization locale) const;
        NJson::TJsonValue GetReportFeedback(ELocalization locale, const TUserCurrentContext* userCurrentContext, bool needBill, NDriveSession::TReportTraits reportTraits) const;
        NJson::TJsonValue GetReportPayments(ELocalization locale, const TUserCurrentContext* userCurrentContext) const;
        NJson::TJsonValue GetReportPhotos(const THostByImageSourceMapping& hostBySourceMapping) const;
        NJson::TJsonValue GetReportPoi(ELocalization locale) const;
        NJson::TJsonValue GetReportTransportation(const TGeoCoord* userLocation, IRequestProcessor::EApp app, ELocalization locale) const;
        NJson::TJsonValue GetReportNotification(ELocalization locale) const;
        NJson::TJsonValue GetAdditionalServiceOffersReport(ELocalization locale, const TUserCurrentContext* userCurrentContext, const TUserSessionsContext& userSessionsContext) const;
        NJson::TJsonValue GetServiceFuelingReport(ELocalization locale) const;
    };

private:
    const NDrive::IServer* Server;
    TUserPermissions::TPtr Permissions;
    NFavouriteAddressAdvisor::IAdvisor::TPtr FavouriteAddressAdvisorPtr;
    TCarsFetcher CurrentFetcher;
    TCarsFetcher HistoryFetcher;
    TSet<TString> CurrentObjectIds;
    TSet<TString> CurrentObjectModels;
    TSet<TString> HistoryObjectIds;
    TSet<TString> HistoryObjectModels;
    R_READONLY(TVector<TDBTag>, DelegationsInProgress);
    R_READONLY(TVector<TDBTag>, DelegationsFinished);
    R_READONLY(TVector<TDBTag>, Futures);
    R_READONLY(TVector<TDBTag>, States);
    R_READONLY(TVector<TSessionContext>, Sessions);
    R_READONLY(TVector<TSessionContext>, SharedSessions);
    R_READONLY(TVector<TAtomicSharedPtr<const IEventsSession<TTagHistoryEvent>>>, AdditionalServiceSessions);

private:
    TVector<TSessionContext> FetchCurrentSessions(const TVector<TDBTag>& tags, const TString& userId, NDrive::TEntitySession& tx) const;

public:
    TUserSessionsContext(const NDrive::IServer* server, TUserPermissions::TPtr permissions, ELocalization locale, NDeviceReport::TReportTraits traits)
        : Server(server)
        , Permissions(permissions)
        , CurrentFetcher(*server, NDeviceReport::ReportSession & traits)
        , HistoryFetcher(*server, NDeviceReport::ReportSession & traits)
    {
        CurrentFetcher.SetLocale(locale);
        HistoryFetcher.SetLocale(locale);
        FavouriteAddressAdvisorPtr = NFavouriteAddressAdvisor::TAdvisor::Construct(*Server, permissions);
    }

    NJson::TJsonValue GetDelegationsReport() const;
    NJson::TJsonValue GetFuturesReport(ELocalization locale, const TUserCurrentContext* userCurrentContext) const;
    NJson::TJsonValue GetSessionsReport(ELocalization locale, bool needBill, NDriveSession::TReportTraits reportTraits, IRequestProcessor::EApp app, const TUserCurrentContext* userCurrentContext) const;
    NJson::TJsonValue GetSharedSessionsReport(ELocalization locale, bool needBill, NDriveSession::TReportTraits reportTraits, IRequestProcessor::EApp app, const TUserCurrentContext* userCurrentContext) const;
    NJson::TJsonValue GetAdditionalServiceSessionsReport(ELocalization locale, bool needBill, NDriveSession::TReportTraits reportTraits, IRequestProcessor::EApp app, const TUserCurrentContext* userCurrentContext) const;
    NJson::TJsonValue GetStatesReport(ELocalization locale, const TUserCurrentContext* userCurrentContext) const;
    NJson::TJsonValue GetCarsMetaReport() const;
    TString GetCarsReport(const TSet<TString>& supportAllowedPhotos = {}) const;

    bool HasSharedSessions() const {
        return !SharedSessions.empty();
    }
    bool HasAdditionalServiceSessions() const {
        return !AdditionalServiceSessions.empty();
    }
    bool HasStates() const {
        return !States.empty();
    }

    ui64 GetSessionCount() const;
    bool IsExtraSessionsAvailable() const;

    const TCarsFetcher& GetFetcher(const bool isCurrent) const {
        return isCurrent ? CurrentFetcher : HistoryFetcher;
    }

    bool Initialize(NDrive::TEntitySession& session, TJsonReport& report, const TInstant reqActuality, const bool needClosedSession, const TString& sessionId, const TUserCurrentContext* userCurrentContext);
};
