#include "user_context.h"

#include <drive/backend/processors/car_scanner/processor.h>

#include <drive/backend/actions/session_photo_screen.h>
#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/accounts/yandex_account.h>
#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/data/achievement.h>
#include <drive/backend/data/account_tags.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/delegation.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/fueling.h>
#include <drive/backend/data/long_term.h>
#include <drive/backend/data/offer.h>
#include <drive/backend/data/radar.h>
#include <drive/backend/data/scoring/scoring.h>
#include <drive/backend/data/sharing.h>
#include <drive/backend/data/state.h>
#include <drive/backend/data/user_origin.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/rental/rental_offer_holder_tag.h>
#include <drive/backend/images/database.h>

#include <drive/library/cpp/mds/client.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>

void TExternalPromoContext::Initialize(const TVector<TDBTag>& userTags) {
    for (auto&& tag : userTags) {
        if (tag.Is<TExternalPromoTag>()) {
            ExternalPromoTags.push_back(tag);
        }
    }
}

TFeedbackButtons TExternalPromoContext::GetButtons(ELocalization locale) const {
    TFeedbackButtons result;
    for (const auto& tag : ExternalPromoTags) {
        auto description = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
        if (!description) {
            continue;
        }
        const TExternalPromoTag::TDescription* descriptionImpl = description->GetAs<TExternalPromoTag::TDescription>();
        const TExternalPromoTag* tagImpl = tag.GetTagAs<TExternalPromoTag>();
        if (!descriptionImpl || !tagImpl || descriptionImpl->GetDeadline() < ModelingNow()) {
            continue;
        }

        const TString& code = tagImpl->GetCode();
        const TString& icon = descriptionImpl->GetIcon();
        auto title = Server->GetLocalization()->ApplyResources(descriptionImpl->GetTitle(), locale);

        if (icon && title && code) {
            TFeedbackButton button;
            button.Code = code;
            button.Icon = icon;
            button.Title = title;
            button.Deeplink = tagImpl->GetDeeplink();
            button.Description = Server->GetLocalization()->ApplyResources(descriptionImpl->GetDescription(), locale);
            button.DetailedDescription = Server->GetLocalization()->ApplyResources(descriptionImpl->GetDetailedDescription(), locale);
            button.Priority = descriptionImpl->GetPriority();
            result.push_back(std::move(button));
        }
    }
    return result;
}

TOrganizationContext::TOrganizationContext(const NDrive::IServer* server)
    : Server(server)
    , B2BApplicationTagNames(StringSplitter(Server->GetSettings().GetValue<TString>(NDrive::B2BApplicationTagNameSetting).GetOrElse(TString())).Split(',').SkipEmpty())
{
}

bool TOrganizationContext::Initialize(const TVector<TDBTag>& userTags, const TMap<TString, TDBTag>& accountTags) {
    TSet<ui64> organizationIds;
    for (auto&& tag : userTags) {
        const TWalletAccessTag* tagImpl = tag.GetTagAs<TWalletAccessTag>();
        if (tagImpl) {
            OrganizationTags.push_back(tag);
            organizationIds.insert(tagImpl->GetParentId());
            if (auto it = accountTags.find(::ToString(tagImpl->GetParentId())); it != accountTags.end()) {
                ExpiredDebtTags.emplace(tagImpl->GetParentId(), it->second);
            }
        }
        if (B2BApplicationTagNames.contains(tag->GetName())) {
            ApplicationFormTags.push_back(tag);
        }
    }
    if (Server->GetDriveAPI()->HasBillingManager()) {
        auto wallets = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetAccountsChildren(organizationIds);
        for (const auto& [wallet, parentId] : wallets) {
            auto accounts = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetAccountsByName(wallet);
            if (!accounts) {
                return false;
            }
            OrganizationWallets[parentId][wallet] = {wallet, accounts->size()};
            auto countIt = AccountsCount.find(parentId);
            if (countIt == AccountsCount.end()) {
                countIt = AccountsCount.emplace(parentId, 0).first;
            }
            countIt->second += accounts->size();
        }
    }
    return true;
}

NJson::TJsonValue TOrganizationContext::GetReport(ELocalization locale) const {
    NJson::TJsonValue organizationsJson(NJson::JSON_ARRAY);
    if (Server->GetDriveAPI()->HasBillingManager()) {
        TSet<ui32> organizationIds;
        TMap<ui32, TString> organizationTags;
        for (const auto& tag : OrganizationTags) {
            const TWalletAccessTag* tagImpl = tag.GetTagAs<TWalletAccessTag>();
            if (tagImpl) {
                organizationIds.insert(tagImpl->GetParentId());
                organizationTags[tagImpl->GetParentId()] = tag.GetTagId();
            }
        }
        auto organizations = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetAccountsByIds(organizationIds);
        if (organizations) {
            for (const auto& organization : *organizations) {
                if (organization) {
                    NJson::TJsonValue report = organization->GetReport();
                    auto countIt = AccountsCount.find(organization->GetId());
                    if (countIt != AccountsCount.end()) {
                        report["accounts_count"] = countIt->second;
                    } else {
                        report["accounts_count"] = 0;
                    }
                    if (auto tagIt = organizationTags.find(organization->GetId()); tagIt != organizationTags.end()) {
                        report["tag_id"] = tagIt->second;
                    }
                    auto walletsIt = OrganizationWallets.find(organization->GetId());
                    if (walletsIt != OrganizationWallets.end()) {
                        auto& walletsJson = report["wallets"];
                        for (const auto& [wallet, info] : walletsIt->second) {
                            walletsJson.InsertValue(wallet, info.GetReport());
                        }
                    }
                    if (auto it = ExpiredDebtTags.find(organization->GetId()); it != ExpiredDebtTags.end()) {
                        if (auto impl = std::dynamic_pointer_cast<const TExpiredDebtAccountTag>(it->second.GetData())) {
                            NJson::MergeJson(impl->GetReport(), report);
                        }
                    }
                    organizationsJson.AppendValue(report);
                }

            }
        }
    }
    NJson::TJsonValue applicationFormsJson(NJson::JSON_ARRAY);
    NJson::TJsonValue leadsJson(NJson::JSON_ARRAY);
    for (const auto& tag : ApplicationFormTags) {
        const TUserDictionaryTag* tagImpl = tag.GetTagAs<TUserDictionaryTag>();
        if (tagImpl) {
            auto report = tagImpl->GetPublicReport(locale, Server, nullptr);
            applicationFormsJson.AppendValue(report);
            leadsJson.AppendValue(NJson::TMapBuilder("fields", report)("tag_id", tag.GetTagId()));
        }
    }
    return NJson::TMapBuilder("organizations", organizationsJson)("application_forms", applicationFormsJson)("leads", leadsJson);
}

TMaybe<TUserChatShow> TUserCurrentContext::GetDefaultChat(ui64 sessionCount) const {
    auto id = TString();
    auto deeplink =  TString();
    auto phoneVerificationChatEnabled = GetSetting<bool>("user_registration.phone_verification.enabled").GetOrElse(false);
    auto phoneVerificationChatId = GetSetting<TString>("user_registration.phone_verification.chat").GetOrElse("registration_verify_phone");
    auto idleChatId = GetSetting<TString>("session.idle_chat_id").GetOrElse("");
    const auto& status = UserData->GetStatus();
    if (status == NDrive::UserStatusActive) {
        if (!UserData->IsPhoneVerified() && phoneVerificationChatEnabled) {
            id = phoneVerificationChatId;
        } else if (idleChatId && sessionCount == 0) {
            id = idleChatId;
        } else {
            return {};
        }
    }
    auto enableFastReg = GetSetting<bool>("user_registration.enable_fast_reg").GetOrElse(true);
    auto requiredOrigin = GetSetting<TString>("user_registration.fast_registration.required_origin");
    bool fast = enableFastReg && (!requiredOrigin || requiredOrigin == Origin);
    if (status == NDrive::UserStatusFastRegistered && UserData->IsFirstRiding() && fast) {
        return {};
    }
    if (id.empty() && status == NDrive::UserStatusBlocked) {
        if (auto problemTag = LeadProblemTag.GetTagAs<TUserProblemTag>()) {
            const auto& tagsMeta = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta();
            id = problemTag->GetBanChat(tagsMeta);
            deeplink = problemTag->GetBanDeeplink(tagsMeta);
        }
    }
    if (id.empty() && status == NDrive::UserStatusBlocked) {
        id = "ban";
    }
    if (id.empty() && status == NDrive::UserStatusOnboarding && FastRegistrationTag && fast) {
        id = GetSetting<TString>("user_registration.fast_registration.chat").GetOrElse("registration_light");
    }
    if (id.empty()) {
        auto useExistingChatRobot = GetSetting<bool>("user_registration.registation.use_existing_chat").GetOrElse(true);
        auto chatRobot = useExistingChatRobot ? Server->GetChatRobot("registration") : nullptr;
        if (chatRobot && chatRobot->HasChat(UserData->GetUserId(), "")) {
            id = "registration";
        } else {
            id = GetSetting<TString>("user_registration.registration.chat").GetOrElse("registration");
        }
    }
    TUserChatShow result;
    result.Id = std::move(id);
    result.Deeplink = std::move(deeplink);
    result.IsClosable = false;
    result.ShowInRiding = false;
    return result;
}

NJson::TJsonValue TUserCurrentContext::GetIncomingDelegationsReport() const {
    NJson::TJsonValue report = NJson::JSON_ARRAY;
    for (auto&& d : IncomingDelegations) {
        report.AppendValue(d.GetTagAs<TIncomingDelegationUserTag>()->GetReport(d));
    }
    return report;
}

TSet<TString> TUserCurrentContext::GetDelegationObjectIds() const {
    TSet<TString> result;
    for (auto&& i : Delegations) {
        result.emplace(i.GetTagAs<TDelegationUserTag>()->GetObjectId());
    }
    return result;
}

NJson::TJsonValue TUserCurrentContext::GetAreaUserReport() const {
    NJson::TJsonValue infoForUserJson = NJson::JSON_MAP;

    ui32 processed = 0;
    TMap<TString, TString> infos;
    const auto actor = [&infos, &processed](const TAreaInfoForUser* info) {
        if (info) {
            processed += 1;
            for (auto&& kv : info->GetInfos()) {
                infos.emplace(kv.first, kv.second);
            }
        }
        return true;
    };

    if (UserLocation) {
        Server->GetDriveAPI()->GetAreasDB()->ProcessHardTagsInPoint<TAreaInfoForUser>(*UserLocation, actor, TInstant::Zero());
    }
    if (processed == 0) {
        auto defaultLocationString = Server->GetSettings().GetValue<TString>("default_location");

        TGeoCoord defaultLocation;
        if (defaultLocationString && TryFromString(*defaultLocationString, defaultLocation)) {
            Server->GetDriveAPI()->GetAreasDB()->ProcessHardTagsInPoint<TAreaInfoForUser>(defaultLocation, actor, TInstant::Zero());
        }
    }

    for (auto&& i : infos) {
        infoForUserJson.InsertValue(i.first, i.second);
    }
    return infoForUserJson;
}

NJson::TJsonValue TUserCurrentContext::GetFlagsReport() const {
    return Permissions->GetFlagsReport();
}

NJson::TJsonValue TUserCurrentContext::GetPaymentReport(ELocalization locale, TInstant timestamp, const TMaybe<TSet<TString>>& filtredAccounts, const TString& creditCard) const {
    NJson::TJsonValue paymentMethodsReport = TBillingManager::GetPaymentMethodsReport(
        Permissions->GetSetting<bool>(Server->GetSettings(), "show_bonus_payment_method").GetOrElse(false),
        locale,
        timestamp,
        UserAccounts,
        filtredAccounts,
        PaymentMethods,
        Permissions->UseYandexPaymentMethod(Server->GetSettings()) ? YandexPaymentMethod : Nothing(),
        *Server,
        Permissions->GetMobilePaymentMethod(Server->GetSettings()),
        creditCard,
        { SelectedAccount }
    );
    if (!filtredAccounts && !Permissions->GetUserFeatures().GetIsPlusUser() && Server->GetDriveAPI()->HasBillingManager() && YandexPaymentMethod && YandexPaymentMethod->GetBalance() > 0 && YandexAccountDescription) {
        NDrive::NBilling::TFakeYandexAccount fakeAcc(Permissions->GetUserId(), *YandexAccountDescription, new NDrive::NBilling::TYandexAccountRecord(), nullptr, Server->GetDriveAPI()->GetBillingManager().GetAccountsManager());
        NDrive::NBilling::IBillingAccount::TReportContext context;
        context.ExternalBalance = YandexPaymentMethod->GetBalance();
        NJson::TJsonValue accountReport = fakeAcc.GetNewUserReport(locale, *Server, UserAccounts, context);
        accountReport.InsertValue("link", Server->GetSettings().GetValueDef<TString>("plus_wallet_landing", "https://plus.yandex.ru/"));
        paymentMethodsReport.AppendValue(accountReport);
    }
    return paymentMethodsReport;
}

NJson::TJsonValue TUserCurrentContext::GetReport(ELocalization locale, TInstant timestamp, ui64 sessionCount) const {
    if (UserData) {
        NJson::TJsonValue userJson = NJson::JSON_MAP;
        userJson.InsertValue("is_first_riding", UserData->IsFirstRiding());
        userJson.InsertValue("is_registred", Permissions->IsRegistered());
        userJson.InsertValue("user_id", Permissions->GetUserId());
        {
            auto& jsonPlus = userJson.InsertValue("plus", NJson::JSON_MAP);
            jsonPlus.InsertValue("is_activated", Permissions->GetUserFeatures().GetIsPlusUser());
        }
        if (Radars.size()) {
            auto data = Yensured(Radars.front().GetTagAs<TRadarUserTag>())->GetPublicReport(locale, *Server);
            userJson.InsertValue("scanner", data);
        }
        if (Fueling.size()) {
            auto data = Yensured(Fueling.front().GetTagAs<TUserFuelingTag>())->GetPublicReport(Server->GetLocalization(), Server->GetFuelingManager());
            userJson.InsertValue("fueling", data);
        }
        auto originTag = OriginTag.GetTagAs<TUserOriginTag>();
        if (originTag && originTag->GetOrigin()) {
            userJson.InsertValue("origin", originTag->GetOrigin());
        }

        bool registrationStarted = originTag ||
            Permissions->IsRegistered() ||
            UserData->GetDrivingLicenseDatasyncRevision() ||
            UserData->GetPassportDatasyncRevision();
        userJson.InsertValue("is_registration_started", registrationStarted);

        auto settings = NJson::TJsonValue();
        auto serviceSettingsEnabled = GetSetting<bool>("session.service_settings.enabled").GetOrElse(false);
        if (serviceSettingsEnabled) {
            if (ServiceSettingsTag) {
                settings = Yensured(ServiceSettingsTag.GetTagAs<TUserDictionaryTag>())->GetPublicReport(locale, Server, Permissions);
            } else {
                TUserDictionaryTag defaultSettings;
                defaultSettings.SetName(ServiceSettingsTagName);
                settings = defaultSettings.GetPublicReport(locale, Server, Permissions);
            }
        } else {
            if (SettingsTag) {
                settings = Yensured(SettingsTag.GetTagAs<TUserDictionaryTag>())->GetPublicReport(locale, Server, Permissions);
            } else {
                TUserDictionaryTag defaultSettings;
                defaultSettings.SetName(UserSettingsTagName);
                settings = defaultSettings.GetPublicReport(locale, Server, Permissions);
            }
        }
        userJson.InsertValue("settings", std::move(settings));

        if (Server->GetDriveAPI()->HasBillingManager()) {
            auto& jsonBilling = userJson.InsertValue("billing", NJson::JSON_MAP);
            {
                bool inProc = false;
                ui32 debt = Server->GetDriveAPI()->GetBillingManager().GetDebt(ActivePayments, nullptr, {}, &inProc);
                if (debt > 0) {
                    auto& debtJson = jsonBilling.InsertValue("debt", NJson::JSON_MAP);
                    debtJson.InsertValue("amount", debt);
                    if (inProc) {
                        debtJson.InsertValue("status", "waiting");
                    } else {
                        debtJson.InsertValue("status", "no_funds");
                    }
                }
            }

            if (YandexPaymentMethod) {
                auto& yandexAccount = jsonBilling["yandex_account"];
                yandexAccount.InsertValue("balance", (YandexPaymentMethod->GetBalance() / 100) * 100);
            }

            auto trustAccount = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetTrustAccount(UserAccounts);
            if (trustAccount) {
                userJson.InsertValue("credit_card", trustAccount->GetUserReport());
            }

            ui32 bonuses = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetBonuses(UserAccounts);
            auto& jsonBonuses = jsonBilling.InsertValue("bonuses", NJson::JSON_MAP);
            jsonBonuses.InsertValue("amount", bonuses);

            if (ReportTraits & NDriveSession::ReportB2BOrganization) {
                jsonBilling.InsertValue("b2b", OrganizationContext.GetReport(locale));
            }
            if (ReportTraits & NDriveSession::ReportPaymentMethods) {
                jsonBilling.InsertValue("payment_methods", GetPaymentReport(locale, timestamp));
            }
        }

        TMaybe<TUserChatShow> showChat;
        if (HasChatShowTag()) {
            auto tag = GetChatShowTagUnsafe().GetTagAs<TUserChatShowTag>();
            showChat = tag ? tag->Get(UserData->GetStatus()) : showChat;
        } else {
            showChat = GetDefaultChat(sessionCount);
        }
        if (showChat) {
            userJson.InsertValue("show_chat", NJson::ToJson(showChat));
        }

        if (Permissions) {
            if (!Permissions->GetUserFeatures().GetDefaultEmails().empty()) {
                TSet<TString> visibleEmails;
                if (Permissions->GetEmail()) {
                    visibleEmails.insert(Permissions->GetEmail());
                }
                for (const auto& mail : Permissions->GetUserFeatures().GetDefaultEmails()) {
                    if (!mail.Contains("@yandex.") || mail.EndsWith("ru") || mail.EndsWith("com")) {
                        visibleEmails.insert(mail);
                    }
                }
                userJson["validated_emails"] = NJson::ToJson(visibleEmails);
            }
            userJson["passport"]["display_name"] = Permissions->GetUserFeatures().GetPassportName();
            TString avatarId = Permissions->GetUserFeatures().GetPassportDefaultAvatar();
            TString avatarUrl = GetSetting<TString>("passport.avatar_url_macro").GetOrElse("");
            SubstGlobal(avatarUrl, "_ID_", avatarId);
            userJson["passport"]["avatar_url"] = avatarUrl;
        }

        userJson.InsertValue("details", UserData->GetPublicReport());
        if (Permissions->GetSetting<bool>("use_chat_email_specification", false)) {
            userJson["details"]["setup"]["email"]["verified"] = true;
        }
        auto aggressionReport = GetAggressionReport(locale);
        if (aggressionReport.IsDefined()) {
            userJson["driving_style"] = aggressionReport;
        }
        NJson::TJsonValue subscription;
        for (const auto& subscriptionTag : SubscriptionTags) {
            auto tag = subscriptionTag.GetTagAs<TUserSubscriptionTag>();
            if (!tag) {
                continue;
            }
            if (tag->GetStatus() == TUserSubscriptionTag::ESubscriptionStatus::Active) {
                auto description = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag->GetName());
                auto tagDescription = Yensured(description)->GetAs<TUserSubscriptionTag::TDescription>();

                NJson::TJsonValue status;
                status = NJson::ToJson(tagDescription->GetStatus());
                status.InsertValue("type", description->GetName());
                status.InsertValue("expire_ts", tag->GetSLAInstant().Seconds());
                subscription["status"] = status;

                NJson::TJsonValue filters;
                auto rolesManager = Server->GetDriveAPI()->GetRolesManager();
                const auto& tagsManager = Server->GetDriveAPI()->GetTagsManager();
                auto actions = tag->GetActions(subscriptionTag, tagsManager, *rolesManager, true);
                    for (const auto& action : actions) {
                        auto filter = action.GetAs<TFilterAction>();
                        if (filter) {
                            filters.AppendValue(filter->GetName());
                        }
                }
                subscription["filters"] = filters;
                break;
            }
        }
        userJson["subscription"] = subscription;

        return userJson;
    } else {
        return NJson::JSON_NULL;
    }
}

NJson::TJsonValue TUserCurrentContext::GetAggressionReport(ELocalization locale) const {
    return DrivingProfile.GetPublicReport(*Yensured(Server), GetPermissions(), locale);
}

TUserCurrentContext::TUserCurrentContext(
    const NDrive::IServer* server,
    TUserPermissions::TPtr permissions,
    NDriveSession::TReportTraits reportTraits,
    const TString& origin,
    TMaybe<TGeoCoord> userLocation
)
    : Server(server)
    , Permissions(permissions)
    , ReportTraits(reportTraits)
    , Origin(origin)
    , ServiceSettingsTagName(Server->GetSettings().GetValue<TString>(NDrive::ServiceSettingsTagNameSetting).GetOrElse(TString()))
    , UserSettingsTagName(Server->GetSettings().GetValue<TString>(NDrive::UserSettingsTagNameSetting).GetOrElse(TString()))
    , UserFlagsTagName(Server->GetSettings().GetValue<TString>(NDrive::UserFlagsTagNameSetting).GetOrElse(TString()))
    , UserLocation(userLocation)
    , ExternalPromoContext(Server)
    , OrganizationContext(Server)
    , ObservableTags(permissions->GetTagNamesByAction(TTagAction::ETagAction::Observe))
{
}

bool TUserCurrentContext::Initialize(NDrive::TEntitySession& session, TJsonReport& report) {
    auto driveApi = Yensured(Server->GetDriveAPI());
    auto fallbackUser = Permissions->GetUserFeatures().GetIsFallbackUser();
    const auto& tagsManager = driveApi->GetTagsManager();

    TEventsGuard eg1(report, "TUserCurrentContext::Initialize");
    TVector<TString> tagsForFetch = {
        TDelegationUserTag::TypeName,
        TRadarUserTag::TypeName,
        TSessionSharingTag::Type(),
        TUserFuelingTag::GetTypeName(),
        TTagReservationFutures::TypeName,
        TUserChatShowTag::TypeName,
        TUserOriginTag::Type(),
        TIncomingDelegationUserTag::TypeName,
    };
    auto descriptions = tagsManager.GetTagsMeta().GetTagsByType(TExternalPromoTag::TypeName);
    if (ReportTraits & NDriveSession::ReportB2BOrganization) {
        auto walletAccessDescriptions = tagsManager.GetTagsMeta().GetTagsByType(TWalletAccessTag::TypeName);
        descriptions.insert(descriptions.end(), walletAccessDescriptions.begin(), walletAccessDescriptions.end());
    }
    {
        auto offerHolderTagDescriptions = tagsManager.GetTagsMeta().GetTagsByType(TOfferHolderUserTag::Type());
        descriptions.insert(descriptions.end(), offerHolderTagDescriptions.begin(), offerHolderTagDescriptions.end());
    }
    {
        auto offerHolderTagDescriptions = tagsManager.GetTagsMeta().GetTagsByType(TLongTermOfferHolderTag::Type());
        descriptions.insert(descriptions.end(), offerHolderTagDescriptions.begin(), offerHolderTagDescriptions.end());
    }
    {
        auto offerHolderTagDescriptions = tagsManager.GetTagsMeta().GetTagsByType(TRentalOfferHolderTag::Type());
        descriptions.insert(descriptions.end(), offerHolderTagDescriptions.begin(), offerHolderTagDescriptions.end());
    }
    {
        auto stateTagDescriptions = tagsManager.GetTagsMeta().GetTagsByType(TGenericUserStateTag::Type());
        descriptions.insert(descriptions.end(), stateTagDescriptions.begin(), stateTagDescriptions.end());
    }
    for (auto&& description : descriptions) {
        if (!description) {
            continue;
        }
        const TString& name = description->GetName();
        if (!ObservableTags.contains(name)) {
            continue;
        }
        tagsForFetch.push_back(name);
    }

    {
        auto userProblemTagNames = tagsManager.GetTagsMeta().GetRegisteredTagNames({ TUserProblemTag::TypeName });
        tagsForFetch.insert(tagsForFetch.end(), userProblemTagNames.begin(), userProblemTagNames.end());
    }
    auto fastRegistrationTagName = GetSetting<TString>("user_registration.fast_registration.tag").GetOrElse("fast_registration_available");
    if (fastRegistrationTagName) {
        tagsForFetch.push_back(fastRegistrationTagName);
    }
    auto aggressiveTagName = GetSetting<TString>("aggression.user_tag").GetOrElse("");
    if (aggressiveTagName) {
        tagsForFetch.push_back(aggressiveTagName);
    }
    auto aggressiveTrialTagName = GetSetting<TString>("aggression.trial_tag").GetOrElse("");
    if (aggressiveTrialTagName) {
        tagsForFetch.push_back(aggressiveTrialTagName);
    }
    auto aggressivePriceUpTagName = GetSetting<TString>("aggression.price_up_tag").GetOrElse("");
    if (aggressivePriceUpTagName) {
        tagsForFetch.push_back(aggressivePriceUpTagName);
    }
    TSet<TString> aggressiveBlockTagNames = StringSplitter(GetSetting<TString>("aggression.block_tags").GetOrElse("")).Split(',').SkipEmpty();
    tagsForFetch.insert(tagsForFetch.end(), aggressiveBlockTagNames.begin(), aggressiveBlockTagNames.end());
    TSet<TString> drivingProfileCashbackNotificationTagNames = StringSplitter(GetSetting<TString>("driving_profile.cashback_notification_tag_names").GetOrElse("")).Split(',').SkipEmpty();
    tagsForFetch.insert(tagsForFetch.end(), drivingProfileCashbackNotificationTagNames.begin(), drivingProfileCashbackNotificationTagNames.end());

    if (ServiceSettingsTagName) {
        tagsForFetch.push_back(ServiceSettingsTagName);
    }
    if (UserFlagsTagName) {
        tagsForFetch.emplace_back(UserFlagsTagName);
    }
    if (!!UserSettingsTagName) {
        tagsForFetch.emplace_back(UserSettingsTagName);
    }
    if (const auto& tags = OrganizationContext.GetB2BApplicationTagNames(); ReportTraits & NDriveSession::ReportB2BOrganization && tags) {
        tagsForFetch.insert(tagsForFetch.end(), tags.begin(), tags.end());
    }
    {
        AchievementTagNames = tagsManager.GetTagsMeta().GetRegisteredTagNames({TAchievementUserTag::TypeName});
        tagsForFetch.insert(tagsForFetch.end(), AchievementTagNames.begin(), AchievementTagNames.end());
    }
    TSet<TString> subscriptionTagNames = StringSplitter(GetSetting<TString>("subscription.available_tags").GetOrElse("")).Split(',').SkipEmpty();
    tagsForFetch.insert(tagsForFetch.end(), subscriptionTagNames.begin(), subscriptionTagNames.end());

    TVector<TDBTag> userTags;
    {
        auto eg = report.BuildEventGuard("RestoreUserTags");
        if (!tagsManager.GetUserTags().RestoreEntityTags(Permissions->GetUserId(), tagsForFetch, userTags, session)) {
            return false;
        }
    }
    {
        auto eg = report.BuildEventGuard("InitializeExternalPromoContext");
        ExternalPromoContext.Initialize(userTags);
    }
    if (ReportTraits & NDriveSession::ReportB2BOrganization) {
        auto eg = report.BuildEventGuard("InitializeOrganizationContext");
        const auto& accountTagsManager = Server->GetDriveAPI()->GetTagsManager().GetAccountTags();
        const auto& tagsMeta = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta();
        auto tagNames = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Account, {TExpiredDebtAccountTag::TypeName});
        TVector<TDBTag> tags;
        if (!accountTagsManager.RestoreTags({}, ::MakeVector(NContainer::Keys(tagNames)), tags, session)) {
            return false;
        }

        TMap<TString, TDBTag> expiredDebtTags;
        for (auto&& tag : tags) {
            if (auto impl = std::dynamic_pointer_cast<TExpiredDebtAccountTag>(tag.GetData())) {
                expiredDebtTags.emplace(tag.GetObjectId(), std::move(tag));
            }
        }

        if (!OrganizationContext.Initialize(userTags, expiredDebtTags)) {
            return false;
        }
    }
    for (auto&& tag : userTags) {
        if (!tag) {
            continue;
        }
        const TString& name = tag->GetName();
        if (tag.Is<TDelegationUserTag>()) {
            Delegations.emplace_back(tag);
        }
        if (tag.Is<TIncomingDelegationUserTag>()) {
            IncomingDelegations.emplace_back(tag);
        }
        if (tag.Is<TTagReservationFutures>()) {
            Futures.emplace_back(tag);
        }
        if (tag.Is<TRadarUserTag>()) {
            Radars.emplace_back(tag);
        }
        if (tag.Is<TUserFuelingTag>()) {
            Fueling.emplace_back(tag);
        }
        if (tag.Is<TUserDictionaryTag>()) {
            if (name == ServiceSettingsTagName) {
                ServiceSettingsTag = tag;
            }
            if (name == UserSettingsTagName) {
                SettingsTag = tag;
            }
            if (name == UserFlagsTagName) {
                FlagsTag = tag;
            }
        }
        if (tag.Is<TUserChatShowTag>()) {
            SetChatShowTag(tag);
        }
        if (tag.Is<TUserOriginTag>()) {
            OriginTag = tag;
        }
        if (tag.Is<TOfferHolderTag>()) {
            OfferHolders.push_back(tag);
        }
        if (name == fastRegistrationTagName) {
            FastRegistrationTag = tag;
        }
        if (tag.Is<TScoringUserTag>() && name == aggressiveTagName) {
            DrivingProfile.AggressiveTag = tag;
        }
        if (tag.Is<TScoringUserTag>() && name == aggressiveTrialTagName) {
            DrivingProfile.AggressiveTrialTag = tag;
        }
        if (name == aggressivePriceUpTagName) {
            DrivingProfile.AggressivePriceUpTag = tag;
        }
        if (tag.Is<TAchievementUserTag>()) {
            AchievementTags.push_back(tag);
        }
        if (auto problemTag = tag.GetTagAs<TUserProblemTag>()) {
            if (aggressiveBlockTagNames.contains(name)) {
                DrivingProfile.AggressiveBlockTag = tag;
            }
            auto currentProblemTag = LeadProblemTag.GetTagAs<TUserProblemTag>();
            if (!currentProblemTag || problemTag->GetPoints(tagsManager.GetTagsMeta()) > currentProblemTag->GetPoints(tagsManager.GetTagsMeta())) {
                LeadProblemTag = tag;
            }
        }
        if (tag.Is<NDrive::IUserState>()) {
            States.push_back(tag);
        }
        if (drivingProfileCashbackNotificationTagNames.contains(name)) {
            DrivingProfile.NotificationTags.push_back(tag);
        }
        if (tag.Is<TUserSubscriptionTag>()) {
            SubscriptionTags.push_back(tag);
        }
    }
    TEventsGuard eg2(report, "FetchInfo");
    FetchUserData = Server->GetDriveAPI()->GetUsersData()->FetchInfo(Permissions->GetUserId(), session);
    if (!FetchUserData) {
        return false;
    }
    UserData = FetchUserData.GetResultPtr(Permissions->GetUserId());
    if (UserData) {
        DrivingProfile.UserStatus = UserData->GetStatus();
    }
    if (driveApi->HasBillingManager()) {
        auto eg = report.BuildEventGuard("GetUserPayments");
        auto payments = driveApi->GetBillingManager().GetActiveUserPayments(Permissions->GetUserId(), session);
        if (!payments) {
            return false;
        }
        ActivePayments = std::move(*payments);
    }
    if (driveApi->HasMDSClient()) {
        auto eg = report.BuildEventGuard("GetHostByImageSourceMapping");
        ImageHostBySourceMapping = TCommonImageData::GetHostByImageSourceMapping(driveApi->GetMDSClient());
    }

    if (driveApi->HasBillingManager() && !fallbackUser && (ReportTraits & NDriveSession::ReportPaymentMethods)) {
        const auto& billingManager = driveApi->GetBillingManager();
        {
            auto eg = report.BuildEventGuard("GetUserCards");
            AllPaymentMethods = driveApi->GetUserPaymentMethodsSync(*Permissions, *Server, true);
        }
        {
            auto eg = report.BuildEventGuard("GetUserPaymentCards");
            PaymentMethods = TBillingManager::GetUserPaymentCards(AllPaymentMethods.Get(), false);
        }
        {
            auto eg = report.BuildEventGuard("GetDefaultAccountName");
            SelectedAccount = billingManager.GetDefaultAccountName(Permissions->GetUserId(), session);
            session.Check();
        }
        {
            auto eg = report.BuildEventGuard("GetSortedUserAccounts");
            auto accounts = billingManager.GetAccountsManager().GetSortedUserAccounts(Permissions->GetUserId(), session);
            if (!accounts) {
                return false;
            }
            UserAccounts = *accounts;
        }
        {
            auto eg = report.BuildEventGuard("GetYandexAccount");
            YandexPaymentMethod = billingManager.GetYandexAccount(AllPaymentMethods.Get());
        }
        {
            auto eg = report.BuildEventGuard("GetYandexAccountDescription");
            auto descriptions = billingManager.GetAccountsManager().GetDescriptionDB().GetDescriptionsByType(NDrive::NBilling::EAccount::YAccount, session);
            if (!descriptions) {
                return false;
            }
            if (!descriptions.empty()) {
                YandexAccountDescription = descriptions.begin()->second;
            }
        }
    }
    TSet<TString> highCashbackRoles = StringSplitter(GetSetting<TString>("aggression.notification.high_cashback.roles").GetOrElse("")).Split(',').SkipEmpty();
    for (auto&& role : Permissions->GetRolesByUser()) {
        if (highCashbackRoles.contains(role.GetRoleId())) {
            DrivingProfile.HighCashbackRole = role;
        }
    }

    for (auto&& action : Permissions->GetActionsActual()) {
        if (action.GetAs<NDrive::TSessionPhotoScreenAction>()) {
            SessionPhotoScreenActions.push_back(action);
        }
    }

    return true;
}

TFeedbackButtons TUserCurrentContext::GetAreaFeedbackButtons() const {
    TFeedbackButtons result;
    if (UserLocation) {
        auto actor = [&result, this](const TFeedbackButtonTag* tag) {
            if (tag && ObservableTags.contains(tag->GetName())) {
                result.push_back(tag->GetValue());
            }
            return true;
        };
        Server->GetDriveAPI()->GetAreasDB()->ProcessHardTagsInPoint<TFeedbackButtonTag>(*UserLocation, actor, TInstant::Zero());
    }
    return result;
}
