#pragma once

#include <drive/backend/billing/manager.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/images/image.h>
#include <drive/backend/users/driving_profile.h>

using TFeedbackButton = TFeedbackButtonTag::TValue;
using TFeedbackButtons = TVector<TFeedbackButton>;

struct TUserChatShow;

class TExternalPromoContext {
private:
    TVector<TDBTag> ExternalPromoTags;
    const NDrive::IServer* Server = nullptr;

public:
    TExternalPromoContext(const NDrive::IServer* server)
        : Server(server)
    {
    }

    void Initialize(const TVector<TDBTag>& userTags);

    TFeedbackButtons GetButtons(ELocalization locale) const;
};

class TOrganizationContext {
    struct TWalletContext {
        TString Name;
        ui64 Count;
        NJson::TJsonValue GetReport() const {
            return NJson::TMapBuilder("name", Name)("accounts_count", Count);
        }
    };
private:
    TVector<TDBTag> OrganizationTags;
    TMap<ui32, TDBTag> ExpiredDebtTags;
    TMap<ui32, TMap<TString, TWalletContext>> OrganizationWallets;
    TVector<TDBTag> ApplicationFormTags;
    const NDrive::IServer* Server = nullptr;
    R_READONLY(TSet<TString>, B2BApplicationTagNames);
    TMap<ui32, ui64> AccountsCount;

public:
    TOrganizationContext(const NDrive::IServer* server);
    bool Initialize(const TVector<TDBTag>& userTags, const TMap<TString, TDBTag>& accountTags);

    NJson::TJsonValue GetReport(ELocalization locale) const;
};

class TUserCurrentContext {
private:
    const NDrive::IServer* Server;
    TUserPermissions::TPtr Permissions;
    NDriveSession::TReportTraits ReportTraits;
    TString Origin;
    TString ServiceSettingsTagName;
    TString UserSettingsTagName;
    TString UserFlagsTagName;
    TUsersDB::TFetchResult FetchUserData;
    const TDriveUserData* UserData = nullptr;
    NDrive::TDrivingProfile DrivingProfile;

    R_OPTIONAL(TGeoCoord, UserLocation);
    R_READONLY(TExternalPromoContext, ExternalPromoContext);
    R_READONLY(THostByImageSourceMapping, ImageHostBySourceMapping);
    R_READONLY(TOrganizationContext, OrganizationContext);
    R_OPTIONAL(TDBTag, ChatShowTag);
    R_READONLY(TDBTag, ServiceSettingsTag);
    R_READONLY(TDBTag, SettingsTag);
    R_READONLY(TDBTag, FlagsTag);
    R_READONLY(TDBTag, FastRegistrationTag);
    R_READONLY(TDBTag, OriginTag);
    R_READONLY(TDBTag, LeadProblemTag);
    R_READONLY(TVector<TDBTag>, Futures);
    R_READONLY(TVector<TDBTag>, Delegations);
    R_READONLY(TVector<TDBTag>, IncomingDelegations);
    R_READONLY(TVector<TDBTag>, Radars);
    R_READONLY(TVector<TDBTag>, Fueling);
    R_READONLY(TVector<TDBTag>, OfferHolders);
    R_READONLY(TVector<TDBTag>, States);
    R_READONLY(TSet<TString>, ObservableTags);
    R_READONLY(TVector<TDBAction>, SessionPhotoScreenActions);
    R_READONLY(TVector<TDBTag>, AchievementTags);
    R_READONLY(TSet<TString>, AchievementTagNames);
    R_READONLY(TVector<TDBTag>, SubscriptionTags);

    R_READONLY(TVector<NDrive::NBilling::IBillingAccount::TPtr>, UserAccounts);
    R_READONLY(TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>>, AllPaymentMethods);
    R_READONLY(TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>>, PaymentMethods);
    R_READONLY(TString, SelectedAccount);
    R_READONLY(TMaybe<NDrive::NTrustClient::TPaymentMethod>, YandexPaymentMethod);
    R_READONLY(TVector<TPaymentsData>, ActivePayments);
    R_OPTIONAL(NDrive::NBilling::TAccountDescriptionRecord, YandexAccountDescription);

public:
    TFeedbackButtons GetAreaFeedbackButtons() const;
    TSet<TString> GetDelegationObjectIds() const;
    TMaybe<TUserChatShow> GetDefaultChat(ui64 sessionCount) const;

    NJson::TJsonValue GetAggressionReport(ELocalization locale) const;
    NJson::TJsonValue GetAreaUserReport() const;
    NJson::TJsonValue GetFlagsReport() const;
    NJson::TJsonValue GetPaymentReport(ELocalization locale, TInstant timestamp, const TMaybe<TSet<TString>>& filtredAccounts = {}, const TString& creditCard = {}) const;
    NJson::TJsonValue GetIncomingDelegationsReport() const;
    NJson::TJsonValue GetReport(ELocalization locale, TInstant timestamp, ui64 sessionCount) const;

    const TUserPermissions& GetPermissions() const {
        return *Yensured(Permissions);
    }
    const TDriveUserData* GetUserData() const {
        return UserData;
    }
    NDriveSession::TReportTraits GetReportTraits() const {
        return ReportTraits;
    }
    template <class T>
    TMaybe<T> GetSetting(const TString& key) const {
        return Permissions->GetSetting<T>(Server->GetSettings(), key);
    }
    template <class T>
    T GetSetting(const TString& key, T default_) const {
        return Permissions->GetSetting<T>(Server->GetSettings(), key, std::move(default_));
    }

    TUserCurrentContext(
        const NDrive::IServer* server,
        TUserPermissions::TPtr permissions,
        NDriveSession::TReportTraits reportTraits,
        const TString& origin,
        TMaybe<TGeoCoord> userLocation
    );

    bool Initialize(NDrive::TEntitySession& session, TJsonReport& report);
};
