#include "user_data.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/billing/accounts/trust.h>
#include <drive/backend/billing/trust/charge_logic.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/trust/lpm.h>


void TSetDefaultCardProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TString& payMethod = GetString(requestData, "paymethod_id", false);
    const TString mobilePaymethodId = GetString(requestData, "mobile_paymethod_id", false);

    const auto& billingManager = Server->GetDriveAPI()->GetBillingManager();
    auto session = billingManager.BuildSession(false);
    if (!mobilePaymethodId && !billingManager.SetDefaultCreditCard(payMethod, permissions->GetUserId(), true, session)) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    TBillingTask::TExecContext context;
    if (payMethod || mobilePaymethodId) {
        context.SetPaymethod(payMethod);
        context.SetMobilePaymethod(mobilePaymethodId);
        if (GetValue<bool>(requestData, "boost_same_persons", false).GetOrElse(false)) {
            TSet<TString> userIds;
            auto samePersons = Server->GetDriveAPI()->GetUsersData()->GetSamePersons(permissions->GetUserId(), session);
            if (!samePersons) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            Transform(samePersons->begin(), samePersons->end(), std::inserter(userIds, userIds.end()), [](const auto& user) { return user.GetUserId(); });
            for (auto&& userId : userIds) {
                context.SetUserId(permissions->GetUserId());
                if (!billingManager.GetActiveTasksManager().BoostUserSessions(userId, permissions->GetUserId(), context, session)) {
                    session.DoExceptionOnFail(ConfigHttpStatus);
                }
            }
        } else if (auto samePersonId = GetString(requestData, "same_person", false)) {
            auto samePersons = Server->GetDriveAPI()->GetUsersData()->GetSamePersons(permissions->GetUserId(), session);
            if (!samePersons) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            auto personIt = FindIf(*samePersons, [samePersonId](const auto& person) { return person.GetUserId() == samePersonId; });
            R_ENSURE(personIt != samePersons->end(), ConfigHttpStatus.UserErrorState, "Users are not the same");
            context.SetUserId(permissions->GetUserId());
            if (!billingManager.GetActiveTasksManager().BoostUserSessions(samePersonId, permissions->GetUserId(), context, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        } else {
            if (!billingManager.GetActiveTasksManager().BoostUserSessions(permissions->GetUserId(), permissions->GetUserId(), context, session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    DriveApi->UpdateUserPaymentMethods(*permissions, *Server);
    g.SetCode(HTTP_OK);
}

void TSetPersonalSettingsProcessorConfig::InitFeatures(const TYandexConfig::Section* section) {
    if (!section) {
        return;
    }
    const auto& directives = section->GetDirectives();
    TagNameSetting = directives.Value("TagNameSetting", TagNameSetting);
}

void TSetPersonalSettingsProcessorConfig::ToStringFeatures(IOutputStream& os) const {
    os << "TagNameSetting: " << TagNameSetting << Endl;
}

void TSetPersonalSettingsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TString& tagName = GetString(Context->GetCgiParameters(), "tag_name", false);
    TVector<std::pair<TString, TString>> settings;
    if (requestData.IsArray()) {
        const auto& values = requestData.GetArray();
        R_ENSURE(!values.empty(), ConfigHttpStatus.UserErrorState, "settings array is empty");
        for (auto&& i : values) {
            TString id = GetString(i, "id");
            TString value = GetString(i, "value");
            settings.emplace_back(id, value);
        }
    } else {
        TString id = GetString(requestData, "id");
        TString value = GetString(requestData, "value");
        settings.emplace_back(id, value);
    }

    const TString& userId = GetString(Context->GetCgiParameters(), "user_id", false);
    const auto& tagId = GetUUID(Context->GetCgiParameters(), "tag_id", false);

    auto session = BuildTx<NSQL::Writable>();
    if (tagId) {
        TUserDictionaryTag::SetSettings(tagId, settings, permissions, *Server, session);
    } else {
        TUserDictionaryTag::SetSettings(tagName ? tagName : Config.GetTagNameSetting(), settings, permissions, userId, *Server, session);
    }
    R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "cannot Commit ", session);
    g.SetCode(HTTP_OK);
}

void TGetPersonalSettingsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto locale = GetLocale();
    auto session = BuildTx<NSQL::ReadOnly>();
    auto settingsTag = Server->GetDriveAPI()->GetUserDictionaryTag(permissions->GetUserId(), Config.GetTagNameSetting(), session);
    R_ENSURE(settingsTag.HasData(), ConfigHttpStatus.UnknownErrorStatus, "incorrect settings configuration");
    TSet<TString> fields = MakeSet(GetStrings(Context->GetCgiParameters(), "fields", false));

    TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
    R_ENSURE(settingsData, ConfigHttpStatus.UnknownErrorStatus, "incorrect settings configuration");
    g.MutableReport().AddReportElement("settings", settingsData->GetPublicReport(locale, Server, permissions, fields));
    g.SetCode(HTTP_OK);
}

void TGetDebtInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TInstant requestTime = Now();
    TMaybe<ui32> debt;
    bool inProc = false;
    if (DriveApi->HasBillingManager()) {
        auto session = BuildTx<NSQL::ReadOnly>();
        TSet<TString> userIds = {permissions->GetUserId()};
        if (GetValue<bool>(Context->GetCgiParameters(), "use_same_persons", false).GetOrElse(false)) {
            auto samePersons = Server->GetDriveAPI()->GetUsersData()->GetSamePersons(permissions->GetUserId(), session);
            if (!samePersons) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
            Transform(samePersons->begin(), samePersons->end(), std::inserter(userIds, userIds.end()), [](const auto& user) { return user.GetUserId(); });
        }
        debt = DriveApi->GetBillingManager().GetDebt(userIds, session, nullptr, {}, &inProc);
        if (!debt) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    NJson::TJsonValue debtJson(NJson::JSON_MAP);
    debtJson.InsertValue("amount", debt ? *debt : 0);
    if (debt > 0) {
        if (inProc) {
            debtJson.InsertValue("status", "waiting");
        } else {
            debtJson.InsertValue("status", "no_funds");
        }
    } else {
        debtJson.InsertValue("status", "ok");
    }
    g.MutableReport().AddReportElement("debt", std::move(debtJson));
    g.MutableReport().AddReportElement("server_time", requestTime.Seconds());
    g.SetCode(HTTP_OK);
}
