#include "processor.h"

#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/chat_robots/robot_manager.h>

TMaybe<TUserDocumentsCheck> ExtractCheckFromTag(const TString& userId, const TString& tagField, const TString& checkType, const TUserDictionaryTag& registrationTag) {
    auto assignmentRaw = registrationTag.GetField(tagField);
    if (!assignmentRaw) {
        return {};
    }

    NJson::TJsonValue assignment;
    R_ENSURE(NJson::ReadJsonTree(*assignmentRaw, &assignment) && assignment[0].IsDefined(), HTTP_INTERNAL_SERVER_ERROR, "failed to parse registration tag info");
    assignment = assignment[0];

    TUserDocumentsCheck check;
    check.SetUserId(userId);
    check.SetType(tagField);

    if (!assignment["result"]["verdicts"][checkType][0].IsString()) {
        return {};
    }
    check.SetStatus(assignment["result"]["verdicts"][checkType][0].GetString());

    bool parsed =
        NJson::ParseField(assignment, "assignmentId", check.MutableAssignmentId()) &&
        NJson::ParseField(assignment, "secret", check.MutableSecretId()) &&
        NJson::ParseField(assignment, "worker_id", check.MutableWorkerId()) &&
        NJson::ParseField(assignment, "poolId", check.MutablePoolId()) &&
        NJson::ParseField(assignment, "projectId", check.MutableProjectId()) &&
        NJson::ParseField<unsigned long &>(assignment, "submit_ts", check.MutableTimestamp()) &&
        NJson::ParseField(assignment, "login", check.MutableLogin());
    R_ENSURE(parsed, HTTP_INTERNAL_SERVER_ERROR, "failed to parse registration tag info");

    return check;
}

TMaybe<TRegistrationTagInfo> GetRegistrationTagSettings(const NDrive::IServer& server) {
    auto tagFields = server.GetSettings().GetJsonValue("user_documents_checks.registration_tag_settings");
    if (!tagFields.IsMap()) {
        return Nothing();
    }

    if (!tagFields["mapping"].IsMap()
    || !tagFields["name"].IsString()) {
        return Nothing();
    }

    TRegistrationTagInfo tagInfo;
    const auto& tagMapping = tagFields["mapping"];
    if (!NJson::ParseField(tagMapping, NJson::Dictionary(tagInfo.RegistrationTagMap))
    || !NJson::ParseField(tagFields, "name", tagInfo.RegistrationTagName)) {
        return Nothing();
    }

    return tagInfo;
}

TMaybe<TUserDocumentsChecksHistoryManager::TCheckHistoryEvent> ExtractCheckEventFromTag(const TString& userId, const TString& tagField, const TString& documentType, const TObjectEvent<TConstDBTag>& tagEvent) {
    const auto* registrationTag = tagEvent.GetTagAs<TUserDictionaryTag>();
    R_ENSURE(registrationTag, HTTP_INTERNAL_SERVER_ERROR, "failed to cast registration tag");

    auto check = ExtractCheckFromTag(userId, tagField, documentType, *registrationTag);
    if (!check) {
        return {};
    }

    return TUserDocumentsChecksHistoryManager::TCheckHistoryEvent(
        *check,
        tagEvent.GetHistoryAction(),
        tagEvent.GetHistoryTimestamp(),
        tagEvent.GetHistoryUserId(),
        tagEvent.GetHistoryOriginatorId(),
        tagEvent.GetHistoryComment()
    );
}

void TSetUserDocumentsCheckStatusProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::UserDocumentsCheck);

    TUserDocumentsCheck check;
    R_ENSURE(TryFromJson(requestData, check), HTTP_INTERNAL_SERVER_ERROR, "Failed to parse check from request");
    R_ENSURE(Server->GetUserDocumentsChecksManager(), HTTP_INTERNAL_SERVER_ERROR, "UserDocumentsChecksManager not configured");

    TUserDocumentsChecksManager::TChecksConfiguration checksConfig;
    TMessagesCollector errors;
    R_ENSURE(checksConfig.Init(
        "user_documents_checks.settings",
        "user_documents_checks.generic_checks",
        "user_documents_checks.photo_settings",
        "user_documents_checks.chats_to_move",
        *Server, errors),
        HTTP_INTERNAL_SERVER_ERROR,
        "failed to init checks: " + errors.GetStringReport()
    );

    TVector<TString> photoIds;
    if (check.GetSecretId()) {
        auto session = BuildTx<NSQL::ReadOnly>();

        auto optionalPhotoIds = TUserDocumentsChecksManager::GetPhotoIds(check.GetType(), check.GetSecretId(), checksConfig.GenericMap, *Server, session);
        R_ENSURE(optionalPhotoIds, HTTP_INTERNAL_SERVER_ERROR, "failed to get photo ids", session);

        photoIds = std::move(*optionalPhotoIds);
    }

    TSet<TString> chatIdsSet = MakeSet(SplitString(GetHandlerSettingDef<TString>("move_chat_ids", ""), ","));
    {
        auto eg = g.BuildEventGuard("set_documents_check_status");
        auto chatSession = BuildChatSession(false);
        auto tagSession = BuildTx<NSQL::Writable>();
        R_ENSURE(Server->GetUserDocumentsChecksManager()->SetStatus(check, permissions->GetUserId(), photoIds, checksConfig, *Server, chatSession, tagSession), HTTP_INTERNAL_SERVER_ERROR, "cannot set status", chatSession);
        // TODO: use external chat session for conditions
        R_ENSURE(
            chatSession.Commit() && tagSession.Commit(),
            HTTP_INTERNAL_SERVER_ERROR,
            TStringBuilder() << "cannot commit set status;"
            << " chat session: " << chatSession.GetStringReport()
            << " tag session: " << tagSession.GetStringReport());
        eg.AddEvent(NJson::TMapBuilder
            ("topic", "registration")
            ("user_id", check.GetUserId())
            ("type", check.GetType())
            ("status", check.GetStatus())
        );
    }
    g.SetCode(HTTP_OK);
}

void TGetUserDocumentsCheckProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetUserDocumentsChecksManager(), HTTP_INTERNAL_SERVER_ERROR, "UserDocumentsChecksManager not configured");
    R_ENSURE(Server->GetDriveAPI(), HTTP_INTERNAL_SERVER_ERROR, "DriveAPI not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::UserDocumentsCheck);

    MergeRegistrationTag = GetHandlerSettingDef<bool>("merge_registration_tag", false);
    if (MergeRegistrationTag) {
        RegistrationTagInfo = GetRegistrationTagSettings(*Server);
        R_ENSURE(
            RegistrationTagInfo,
            HTTP_INTERNAL_SERVER_ERROR,
            "failed to parse tag fields map"
        );
    }

    auto userId = GetString(requestData, "user_id");
    TSet<TString> types;
    if (requestData.Has("types")) {
        R_ENSURE(NJson::ParseField(requestData, "types", types, true), HTTP_BAD_REQUEST, "cannot parse 'types' field");
    }

    auto session = BuildChatSession(NSQL::ReadOnly);

    const auto* documentsChecksManager = Server->GetUserDocumentsChecksManager();
    TVector<TUserDocumentsCheck> checks;
    auto optionalResult = documentsChecksManager->GetChecks(userId, types, session);
    R_ENSURE(optionalResult, HTTP_INTERNAL_SERVER_ERROR, "cannot get checks", session);
    checks = std::move(*optionalResult);

    if (MergeRegistrationTag) {
        auto tagsSession = BuildTx<NSQL::ReadOnly>();
        MergeWithRegistrationTag(checks, userId, types, tagsSession);
    }

    NJson::TJsonValue report = NJson::ToJson(checks);
    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

void TGetUserDocumentsCheckProcessor::MergeWithRegistrationTag(TVector<TUserDocumentsCheck>& checks, const TString& userId, const TSet<TString>& types, NDrive::TEntitySession& session) const {
    R_ENSURE(RegistrationTagInfo, HTTP_INTERNAL_SERVER_ERROR, "registration tag info is not configured");
    const auto& userTagsManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();

    TMap<TString, TUserDocumentsCheck> checksFromTag;
    TVector<TDBTag> registrationDBTag;
    if (!userTagsManager.RestoreTags({userId}, {RegistrationTagInfo->RegistrationTagName}, registrationDBTag, session)) {
        session.Check();
        return;
    }

    if (registrationDBTag.empty()) {
        return;
    }

    R_ENSURE(registrationDBTag.size() == 1, HTTP_INTERNAL_SERVER_ERROR, "ambiguous registration info tag");
    auto registrationTag = registrationDBTag.front().GetTagAs<TUserDictionaryTag>();
    R_ENSURE(registrationTag, HTTP_INTERNAL_SERVER_ERROR, "can't cast registration info tag");

    for (const auto& [type, tagType] : RegistrationTagInfo->RegistrationTagMap) {
        if (!types.empty() && !types.contains(type)) {
            continue;
        }

        if (auto checkFromTag = ExtractCheckFromTag(userId, tagType, type, *registrationTag)) {
            checks.emplace_back(std::move(*checkFromTag));
        }
    }
}

void TGetUserDocumentsCheckHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    R_ENSURE(Server->GetUserDocumentsChecksManager(), HTTP_INTERNAL_SERVER_ERROR, "UserDocumentsChecksManager not configured");
    R_ENSURE(Server->GetDriveAPI(), HTTP_INTERNAL_SERVER_ERROR, "DriveAPI not configured");
    EnrichWithPhotoPaths = GetHandlerSettingDef<bool>("enrich_with_photo_path", false);
    R_ENSURE(!EnrichWithPhotoPaths || Server->GetDriveAPI()->HasDocumentPhotosManager(), HTTP_INTERNAL_SERVER_ERROR, "DocumentPhotosManager not configured");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::UserDocumentsCheck);

    if (EnrichWithPhotoPaths) {
        auto optionalMap = TUserDocumentsChecksManager::GetChecksGenericMap("type_names_generic_map", *Server);
        R_ENSURE(optionalMap, HTTP_INTERNAL_SERVER_ERROR, "failed to get type_names_generic_map");
        GenericTypeMap = std::move(*optionalMap);
    }

    MergeRegistrationTag = GetHandlerSettingDef<bool>("merge_registration_tag", false);
    if (MergeRegistrationTag) {
        RegistrationTagInfo = GetRegistrationTagSettings(*Server);
        R_ENSURE(
            RegistrationTagInfo,
            HTTP_INTERNAL_SERVER_ERROR,
            "failed to parse tag fields map"
        );
    }

    TString userId, type;
    R_ENSURE(
        NJson::GetString(requestData, "user_id", &userId) &&
        NJson::GetString(requestData, "type", &type),
        HTTP_BAD_REQUEST,
        "\"user_id\" or \"type\" aren't present"
    );

    TInstant since = TInstant::Zero();
    R_ENSURE(
        NJson::ParseField(requestData, "since", since, false),
        HTTP_BAD_REQUEST,
        "failed to parse \"since\" field"
    );

    const auto* documentsChecksManager = Server->GetUserDocumentsChecksManager();
    auto session = BuildChatSession(NSQL::ReadOnly);
    auto events = documentsChecksManager->GetChecksHistory(userId, type, since, session);

    if (MergeRegistrationTag) {
        auto tagsSession = BuildTx<NSQL::ReadOnly>();
        MergeEventsWithTagHistory(events, userId, type, since, tagsSession);
    }

    NJson::TJsonValue report;
    TTypesByIds typesByIds;
    for (const auto& event: events) {
        auto reportEntry = event.BuildReportItem();

        if (EnrichWithPhotoPaths && GenericTypeMap.contains(event.GetType())) {
            const auto& types = GenericTypeMap.at(event.GetType());
            typesByIds[event.GetSecretId()].insert(types.begin(), types.end());
        }

        report.AppendValue(std::move(reportEntry));
    }

    if (EnrichWithPhotoPaths) {
        auto photoPaths = GetPhotosAccessPaths(
            typesByIds,
            Server->GetDriveAPI()->GetDocumentPhotosManager(),
            session
        );

        for (auto& reportEntry : report.GetArraySafe()) {
            const auto secretId = reportEntry["secret_id"].GetString();
            const auto type = reportEntry["type"].GetString();
            reportEntry["photo_path"] = std::move(photoPaths[secretId][type]);
        }
    }

    g.MutableReport().SetExternalReport(std::move(report));
    g.SetCode(HTTP_OK);
}

TVector<TGetUserDocumentsCheckHistoryProcessor::THistoryEvent> TGetUserDocumentsCheckHistoryProcessor::BuildEventsFromTagHistory(const TString& userId, const TString& type, const TInstant& since, NDrive::TEntitySession& session) const {
    R_ENSURE(RegistrationTagInfo, HTTP_INTERNAL_SERVER_ERROR, "registration tag info is not configured");
    const auto& userTagsManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();

    TTagEventsManager::TQueryOptions options;
    options.SetObjectIds({userId});
    options.SetTags({RegistrationTagInfo->RegistrationTagName});

    auto events = userTagsManager.GetHistoryManager().GetEventsByRanges({}, {since}, session, options);
    if (!events) {
        session.Check();
        return {};
    }

    TVector<THistoryEvent> result;
    auto typeTagField = RegistrationTagInfo->RegistrationTagMap.contains(type) ? RegistrationTagInfo->RegistrationTagMap.at(type) : type;
    for (const auto& event: *events) {
        if (auto checkEvent = ExtractCheckEventFromTag(userId, typeTagField, type, event)) {
            result.push_back(*checkEvent);
        }
    }
    return result;
}

bool TGetUserDocumentsCheckHistoryProcessor::MergeEventsWithTagHistory(TVector<THistoryEvent>& events, const TString& userId, const TString& type, const TInstant& since, NDrive::TEntitySession& session) const {
    auto eventsFromTag = BuildEventsFromTagHistory(userId, type, since, session);
    if (!eventsFromTag) {
        return false;
    }

    TVector<THistoryEvent> mergedEvents(events.size() + eventsFromTag.size());
    auto comp = [](const THistoryEvent& first, const THistoryEvent& second) {
        return first.GetHistoryTimestamp() < second.GetHistoryTimestamp();
    };
    std::merge(events.begin(), events.end(), eventsFromTag.begin(), eventsFromTag.end(), mergedEvents.begin(), comp);
    events = std::move(mergedEvents);
    return true;
}

TGetUserDocumentsCheckHistoryProcessor::TPathsByIds TGetUserDocumentsCheckHistoryProcessor::GetPhotosAccessPaths(const TTypesByIds& typeBySecretId, const TDocumentPhotosManager& photoManager, NDrive::TEntitySession& session) const {
    auto fetchAssignmentResult = photoManager.GetDocumentVerificationAssignments().FetchInfo(NContainer::Keys(typeBySecretId), session);
    R_ENSURE(fetchAssignmentResult, HTTP_INTERNAL_SERVER_ERROR, "failed to fetch assignments");

    TGetUserDocumentsCheckHistoryProcessor::TPathsByIds pathsByIds;
    for (const auto& [secretId, types] : typeBySecretId) {
        auto& paths = pathsByIds[secretId];
        const auto* assignment = fetchAssignmentResult.GetResultPtr(secretId);
        if (!assignment) {
            continue;
        }

        if (types.contains(NUserDocument::EType::PassportBiographical) &&
            assignment->GetPassportBiographicalId()) {
            TString photoId = assignment->GetPassportBiographicalId();
            paths[::ToString(NUserDocument::EType::PassportBiographical)].AppendValue(photoManager.BuildPhotoAccessPath(photoId));
        } else if (types.contains(NUserDocument::EType::PassportRegistration) &&
            assignment->GetPassportRegistrationId()) {
            TString photoId = assignment->GetPassportRegistrationId();
            paths[::ToString(NUserDocument::EType::PassportRegistration)].AppendValue(photoManager.BuildPhotoAccessPath(photoId));
        } else if (types.contains(NUserDocument::EType::LicenseFront) &&
            assignment->GetLicenseFrontId()) {
            TString photoId = assignment->GetLicenseFrontId();
            paths[::ToString(NUserDocument::EType::LicenseFront)].AppendValue(photoManager.BuildPhotoAccessPath(photoId));
        } else if (types.contains(NUserDocument::EType::LicenseBack) &&
            assignment->GetLicenseBackId()) {
            TString photoId = assignment->GetLicenseBackId();
            paths[::ToString(NUserDocument::EType::LicenseBack)].AppendValue(photoManager.BuildPhotoAccessPath(photoId));
        }
    }
    return pathsByIds;
}
