#include "processor.h"

#include <drive/backend/actions/administrative.h>
#include <drive/backend/actions/tag.h>
#include <drive/backend/actions/info_access.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/billing/interfaces/account_description.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/database/drive/url.h>
#include <drive/backend/doc_packages/manager.h>
#include <drive/backend/notifications/manager.h>
#include <drive/backend/offers/actions/correctors.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/promo_codes/common/action.h>
#include <drive/backend/rt_background/manager/manager.h>

#include <drive/telematics/protocol/actions.h>

#include <util/generic/serialized_enum.h>

void TAddRoleActionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    NJson::TJsonValue::TArray arrActionJson;
    NJson::TJsonValue::TArray arrRolesJson;
    if (requestData.IsArray()) {
        requestData.GetArray(&arrActionJson);
    } else {
        R_ENSURE(requestData.IsMap(), ConfigHttpStatus.SyntaxErrorStatus, "Post data have to be an array");
        if (requestData["actions"].IsArray()) {
            requestData["actions"].GetArray(&arrActionJson);
        }
        if (requestData["roles"].IsArray()) {
            requestData["roles"].GetArray(&arrRolesJson);
        }
    }
    TVector<TLinkedRoleActionHeader> roleActions;
    TSet<TString> actionIds;
    for (auto&& i : arrActionJson) {
        TLinkedRoleActionHeader linkedInfo;
        R_ENSURE(linkedInfo.DeserializeFromJson(i), ConfigHttpStatus.SyntaxErrorStatus, "incorrect json data about action for role");
        roleActions.push_back(linkedInfo);
        actionIds.emplace(linkedInfo.GetSlaveObjectId());
    }
    TVector<TLinkedRoleRoleHeader> roleRoles;
    TSet<TString> roleIds;
    for (auto&& i : arrRolesJson) {
        TLinkedRoleRoleHeader linkedInfo;
        R_ENSURE(linkedInfo.DeserializeFromJson(i), ConfigHttpStatus.SyntaxErrorStatus, "incorrect json data about action for role");
        roleRoles.push_back(linkedInfo);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Role, linkedInfo.GetRoleId());
        roleIds.emplace(linkedInfo.GetSlaveObjectId());
    }

    {
        TSet<TString> actionRoles = Server->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetActions(roleIds);
        actionIds.insert(actionRoles.begin(), actionRoles.end());
        const TVector<TDBAction> actionObjects = Server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetCachedObjectsVector(actionIds);
        for (auto&& i : actionObjects) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Action, i->GetName(), i->GetGrouppingTags());
        }
    }

    auto session = BuildTx<NSQL::Writable>();
    if (!DriveApi->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(roleActions, permissions->GetUserId(), session)
        || !DriveApi->GetRolesManager()->GetRolesInfoDB().GetRoleRoles().Link(roleRoles, permissions->GetUserId(), session)) {
        g.MutableReport().AddReportElement("error", "Incorrect link");
        g.MutableReport().AddReportElement("error_details", session.GetTransaction()->GetErrors().GetStringReport());
        g.SetCode(ConfigHttpStatus.UserErrorState);
    } else if (!session.Commit()) {
        g.MutableReport().AddReportElement("error", "Cannot add into database");
        g.MutableReport().AddReportElement("error_details", session.GetTransaction()->GetErrors().GetStringReport());
        g.SetCode(ConfigHttpStatus.UserErrorState);
    } else {
        auto notifier = Config.GetNotifier() ? Server->GetNotifier(Config.GetNotifier()) : nullptr;
        if (notifier) {
            TSet<TString> roleIds;
            for (auto&& roleAction : roleActions) {
                roleIds.insert(roleAction.GetRoleId());
            }
            for (auto&& roleRole : roleRoles) {
                roleIds.insert(roleRole.GetRoleId());
            }
            for (auto&& roleId : roleIds) {
                notifier->Notify(Sprintf(
                    "Role <a href=\"%s\">%s</a> has been expanded by %s",
                    TCarsharingUrl().RolePage(roleId).c_str(),
                    roleId.c_str(),
                    permissions->GetHRReport().c_str()
                ));
            }
        }
        g.SetCode(HTTP_OK);
    }
}

void TRemoveRoleActionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();

    const TVector<TString> actionNames = SplitString(cgi.Get("actions"), ",");
    const TVector<TString> roleNames = SplitString(cgi.Get("roles"), ",");
    const TString roleId = cgi.Get("role");

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Role, roleId);

    auto session = BuildTx<NSQL::Writable>();
    if (!DriveApi->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Unlink(actionNames, roleId, permissions->GetUserId(), session) || !DriveApi->GetRolesManager()->GetRolesInfoDB().GetRoleRoles().Unlink(roleNames, roleId, permissions->GetUserId(), session)) {
        g.MutableReport().AddReportElement("error", "Cannot remove link");
        g.MutableReport().AddReportElement("error_details", session.GetReport());
        g.SetCode(ConfigHttpStatus.UserErrorState);
        return;
    }

    if (!session.Commit()) {
        g.MutableReport().AddReportElement("error", "Cannot remove link from db database");
        g.MutableReport().AddReportElement("error_details", session.GetReport());
        g.SetCode(ConfigHttpStatus.UserErrorState);
    } else {
        auto notifier = Config.GetNotifier() ? Server->GetNotifier(Config.GetNotifier()) : nullptr;
        if (notifier) {
            notifier->Notify(Sprintf(
                "Role <a href=\"%s\">%s</a> has been reduced by %s",
                TCarsharingUrl().RolePage(roleId).c_str(),
                roleId.c_str(),
                permissions->GetHRReport().c_str()
            ));
        }
        g.SetCode(HTTP_OK);
    }
}

void TUpsertRoleSnapshotProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const NJson::TJsonValue::TArray* arrPtr;
    ReqCheckCondition(requestData.GetArrayPointer(&arrPtr), ConfigHttpStatus.SyntaxErrorStatus, "user_error_incorrect_request_data");

    TVector<TRoleSnapshot> snapshots;
    for (auto&& i : *arrPtr) {
        TRoleSnapshot snapshot;
        ReqCheckCondition(snapshot.DeserializeFromJson(i), ConfigHttpStatus.SyntaxErrorStatus, "user_error_incorrect_request_data_elements");
        snapshots.emplace_back(std::move(snapshot));
    }

    auto rolesManager = Yensured(DriveApi->GetRolesManager());
    bool isUpdate = false;
    for (auto&& snapshot : snapshots) {
        if (!snapshot.GetRoleRoles().GetSlaves().empty() || !snapshot.GetRoleActions().GetSlaves().empty()) {
            isUpdate = true;
            break;
        }
        auto optionalRole = rolesManager->GetRolesDB().GetObject(snapshot.GetRole().GetRoleId());
        if (optionalRole) {
            isUpdate = true;
            break;
        }
    }
    if (isUpdate) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Role);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::Role);
    } else {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Role);
    }

    auto tx = BuildTx<NSQL::Writable>();
    for (auto&& s : snapshots) {
        R_ENSURE(DriveApi->GetRolesManager()->ApplySnapshot(s, permissions->GetUserId(), tx), {}, "cannot ApplySnapshot for " << s.GetRole().GetName(), tx);
        tx.Committed().Subscribe([this, permissions, roleId = s.GetRole().GetName()](const NThreading::TFuture<void>& c) {
            if (!c.HasValue()) {
                return;
            }
            auto notifier = Config.GetNotifier() ? Server->GetNotifier(Config.GetNotifier()) : nullptr;
            if (notifier) {
                notifier->Notify(Sprintf(
                    "RoleSnapshot <a href=\"%s\">%s</a> has been upserted by %s",
                    TCarsharingUrl().RolePage(roleId).c_str(),
                    roleId.c_str(),
                    permissions->GetHRReport().c_str()
                ));
            }
        });
    }
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}

void TConfirmRoleSnapshotProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TVector<TString> propositionIds = GetStrings(requestData, "proposition_ids");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Confirm, TAdministrativeAction::EEntity::Role);

    auto tx = BuildTx<NSQL::Writable>();

    auto optionalSessions = DriveApi->GetRolesManager()->GetSnapshotPropositions().Get(propositionIds, tx);
    R_ENSURE(optionalSessions, {}, "cannot get propositions", tx);
    auto sessions = *optionalSessions;

    for (auto&& i : propositionIds) {
        auto it = sessions.find(i);
        if (it == sessions.end()) {
            continue;
        }

        auto confirmResult = DriveApi->GetRolesManager()->GetSnapshotPropositions().Confirm(it->first, permissions->GetUserId(), tx);
        if (confirmResult == EPropositionAcceptance::ConfirmWaiting) {
        } else if (confirmResult == EPropositionAcceptance::ReadyForCommit) {
            R_ENSURE(DriveApi->GetRolesManager()->ApplySnapshot(it->second, permissions->GetUserId(), tx), {}, "cannot ApplySnapshot of proposition " << i, tx);
            tx.Committed().Subscribe([this, permissions, roleId = it->second.GetRole().GetName()](const NThreading::TFuture<void>& c) {
                if (!c.HasValue()) {
                    return;
                }
                auto notifier = Config.GetNotifier() ? Server->GetNotifier(Config.GetNotifier()) : nullptr;
                if (notifier) {
                    notifier->Notify(Sprintf(
                        "RoleSnapshot <a href=\"%s\">%s</a> has been confirmed by %s",
                        TCarsharingUrl().RolePage(roleId).c_str(),
                        roleId.c_str(),
                        permissions->GetHRReport().c_str()
                    ));
                }
            });
        } else {
            tx.Check();
        }
    }
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    g.SetCode(HTTP_OK);
}

void TListUserActionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& externalUserId = GetUUID(cgi, "user_id", false);

    TUserPermissionsFeatures userPermissionsFeatures;
    TUserPermissions::TPtr viewed = externalUserId ? DriveApi->GetUserPermissions(externalUserId, userPermissionsFeatures) : permissions;
    R_ENSURE(viewed, ConfigHttpStatus.UnknownErrorStatus, "cannot acquire user permissions");
    const TString& userId = viewed->GetUserId();
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::User, userId);

    auto userRoles = DriveApi->GetUsersData()->GetRoles().GetCachedUserRoles(userId);
    TSet<TString> userRoleIds;
    {
        for (auto&& i : userRoles->GetRoles()) {
            userRoleIds.insert(i.GetRoleId());
        }
    }

    const auto& activeActions = viewed->GetActionsActual();
    const auto serialize = [](const TUserAction& action, bool enabled) {
        NJson::TJsonValue result;
        result["action_id"] = action.GetName();
        result["enabled"] = enabled;
        const auto& description = action.GetDescription();
        if (description) {
            result["action_description"] = description;
        }
        const auto& icon = action.GetAdminIcon();
        if (icon) {
            result["action_meta"]["admin_icon"] = icon;
        }
        return result;
    };

    NJson::TJsonValue actions;
    {
        TSet<TString> actionIds;
        for (auto&& i : activeActions) {
            const auto& id = i->GetName();
            actions.AppendValue(serialize(*i, true));
            actionIds.insert(id);
        }
        TVector<TDBAction> allActions;
        ReqCheckCondition(DriveApi->GetRolesManager()->GetActions(userRoleIds, TInstant::Zero(), "", TInstant::Max(), allActions, {}, true),
            ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
        for (auto&& i : allActions) {
            const auto& id = i->GetName();
            if (actionIds.contains(id)) {
                continue;
            }

            actions.AppendValue(serialize(*i, false));
            actionIds.insert(id);
        }
    }
    g.MutableReport().AddReportElement("actions", std::move(actions));

    NJson::TJsonValue evolutionsJson = NJson::JSON_MAP;
    for (auto&& from : *permissions->GetEvolutions()) {
        NJson::TJsonValue& toJson = evolutionsJson.InsertValue(from.first, NJson::JSON_ARRAY);
        for (auto&& to : from.second) {
            NJson::TJsonValue& info = toJson.AppendValue(NJson::JSON_MAP);
            info.InsertValue("to", to.first);
        }
    }

    g.MutableReport().AddReportElement("evolutions", std::move(evolutionsJson));
    g.MutableReport().AddReportElement("user_id", userId);
    g.SetCode(HTTP_OK);
}

void TUserRolesHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString userId = GetString(cgi, "user_id");
    const TInstant now = Now();
    const TInstant since = GetTimestamp(cgi, "since", now - TDuration::Days(7));
    const TInstant until = GetTimestamp(cgi, "until", now);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::User, userId);

    const TUsersDB& usersDb = *DriveApi->GetUsersData();
    const TUserRolesHistoryManager& manager = usersDb.GetRoles().GetHistoryManager();
    auto session = usersDb.BuildSession(true);
    auto optionalEvents = manager.GetEventsByUser(userId, session, since);
    R_ENSURE(optionalEvents, {}, "cannot GetEventsByUser " << userId, session);

    TSet<TString> userIds;
    for (auto&& ev : *optionalEvents) {
        if (ev.GetHistoryInstant() > until) {
            break;
        }
        const TString& actor = ev.GetHistoryUserId();
        const TString& originator = ev.GetHistoryOriginatorId();
        if (actor) {
            userIds.insert(actor);
        }
        if (originator) {
            userIds.insert(originator);
        }
    }

    NJson::TJsonValue users = NJson::JSON_MAP;
    auto userData = usersDb.FetchInfo(userIds, session);
    for (auto&&[id, data] : userData.GetResult()) {
        users[id] = data.GetReport(permissions->GetUserReportTraits());
    }
    g.MutableReport().AddReportElement("users", std::move(users));

    NJson::TJsonValue records = NJson::JSON_ARRAY;
    for (auto&& ev : *optionalEvents) {
        if (ev.GetHistoryInstant() > until) {
            break;
        }
        records.AppendValue(ev.BuildReportItem());
    }
    g.MutableReport().AddReportElement("records", std::move(records));
    g.SetCode(HTTP_OK);
}

void TListConstantsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requesData*/) {
    const auto& cgi = Context->GetCgiParameters();
    auto ids = MakeSet(GetStrings(cgi, "ids", false));
    auto traitsSet = GetValues<EConstantTraits>(cgi, "traits", false);

    TConstantTraits traits = Max<TConstantTraits>();
    if (traitsSet.size()) {
        traits = 0;
        for (auto&& i : traitsSet) {
            traits |= (TConstantTraits)i;
        }
    }

    if (traits & (TConstantTraits)EConstantTraits::Actions) {
        TEventsGuard eg(g.MutableReport(), "actions");
        TSet<TString> actionKeys;
        TUserAction::TFactory::GetRegisteredKeys(actionKeys);
        NJson::TJsonValue iFace;
        for (auto&& i : actionKeys) {
            if (ids.size() && !ids.contains(i)) {
                continue;
            }
            TEventsGuard eg(g.MutableReport(), i);
            TUserAction::TPtr action = TUserAction::TFactory::Construct(i);
            CHECK_WITH_LOG(!!action);
            iFace.InsertValue(i, action->GetScheme(Server).SerializeToJson());
        }
        g.MutableReport().AddReportElement("iface", std::move(iFace));
    }

    if (traits & (TConstantTraits)EConstantTraits::RTBackground) {
        {
            TEventsGuard eg(g.MutableReport(), "rt_background");
            TSet<TString> keys;
            IRTBackgroundProcess::TFactory::GetRegisteredKeys(keys);

            NJson::TJsonValue iFace = NJson::JSON_MAP;
            for (auto&& i : keys) {
                if (ids.size() && !ids.contains(i)) {
                    continue;
                }
                IRTBackgroundProcess::TPtr obj = IRTBackgroundProcess::TFactory::Construct(i);
                TRTBackgroundProcessContainer container(obj);
                iFace.InsertValue(i, container.GetScheme(*Server).SerializeToJson());
            }
            g.MutableReport().AddReportElement("iface_rt_background_settings", std::move(iFace));
        }
        {
            TEventsGuard eg(g.MutableReport(), "rt_background_states");
            TSet<TString> keys;
            IRTBackgroundProcessState::TFactory::GetRegisteredKeys(keys);

            NJson::TJsonValue iFace = NJson::JSON_MAP;
            for (auto&& i : keys) {
                if (ids.size() && !ids.contains(i)) {
                    continue;
                }
                IRTBackgroundProcessState::TPtr obj = IRTBackgroundProcessState::TFactory::Construct(i);
                TRTBackgroundProcessStateContainer container(obj);
                iFace.InsertValue(i, container.GetScheme().SerializeToJson());
            }
            g.MutableReport().AddReportElement("iface_rt_background_state", std::move(iFace));
        }
    }

    if (traits & (TConstantTraits)EConstantTraits::Notifiers) {
        TEventsGuard eg(g.MutableReport(), "notifiers");
        TSet<TString> keys;
        NDrive::INotifierConfig::TFactory::GetRegisteredKeys(keys);

        NJson::TJsonValue iFace = NJson::JSON_MAP;
        for (auto&& i : keys) {
            if (ids.size() && !ids.contains(i)) {
                continue;
            }
            NDrive::INotifierConfig::TPtr obj = NDrive::INotifierConfig::TFactory::Construct(i);
            TNotifierContainer container(obj);
            iFace.InsertValue(i, container.GetScheme(*Server).SerializeToJson());
        }
        g.MutableReport().AddReportElement("iface_notifiers", std::move(iFace));
    }

    if (traits & (TConstantTraits)EConstantTraits::TagDescriptions) {
        TEventsGuard eg(g.MutableReport(), "tag_descriptions");
        TSet<TString> keys;
        TTagDescription::TFactory::GetRegisteredKeys(keys);
        NJson::TJsonValue iFace = NJson::JSON_MAP;
        for (auto&& i : keys) {
            if (ids.size() && !ids.contains(i)) {
                continue;
            }
            TTagDescription::TPtr td = TTagDescription::TFactory::Construct(i);
            iFace.InsertValue(i, td->GetScheme(Server).SerializeToJson());
        }
        g.MutableReport().AddReportElement("iface_tag_descriptions", std::move(iFace));
    }

    if (traits & EConstantTraits::Documents) {
        TEventsGuard eg(g.MutableReport(), "document_manager");
        if (Server->GetDocumentsManager()) {
            g.MutableReport().AddReportElement(TDocumentsManager::GetTypeName(), Server->GetDocumentsManager()->GetScheme(*Server));
        }
    }

    if ((traits & EConstantTraits::Accounts) && Server->GetDriveAPI()->HasBillingManager()) {
        TEventsGuard eg(g.MutableReport(), "billing");
        auto accounts = TVector<TString>();
        auto registeredAccounts = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetRegisteredAccounts();
        for (auto&& description : registeredAccounts) {
            accounts.emplace_back(description.GetName());
        }
        g.AddReportElement("accounts", NJson::ToJson(accounts));
        const NDrive::NBilling::TAccountsManager& accountsManager = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager();
        g.MutableReport().AddReportElement("billing", accountsManager.GetScheme(Server));
    }

    if (traits & (TConstantTraits)EConstantTraits::Tags) {
        TEventsGuard eg(g.MutableReport(), "tags");
        TSet<TString> keys;
        ITag::TFactory::GetRegisteredKeys(keys);

        NJson::TJsonValue iFace = NJson::JSON_MAP;
        TMap<TString, NJson::TJsonValue> schemesByName;
        for (auto&& i : keys) {
            if (ids.size() && !ids.contains(i)) {
                continue;
            }
            TEventsGuard eg(g.MutableReport(), i);
            ITag::TPtr t = ITag::TFactory::Construct(i);
            auto tags = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(i);

            TString schemeStr;
            bool needByName = false;
            for (auto&& tag : tags) {
                t->SetName(tag->GetName());
                NDrive::TScheme scheme = t->GetScheme(Server);
                if (scheme.IsEmpty()) {
                    continue;
                }
                NJson::TJsonValue jsonValue = scheme.SerializeToJson();
                if (!schemeStr) {
                    schemeStr = jsonValue.GetStringRobust();
                } else if (!needByName && schemeStr != jsonValue.GetStringRobust()) {
                    needByName = true;
                }
                schemesByName.emplace(tag->GetName(), std::move(jsonValue));
            }
            if (needByName) {
                for (auto&& scheme : schemesByName) {
                    iFace.InsertValue(scheme.first, std::move(scheme.second));
                }
            } else {
                NDrive::TScheme tagScheme = t->GetScheme(Server);
                if (!tagScheme.IsEmpty()) {
                    iFace.InsertValue(i, tagScheme.SerializeToJson());
                }
            }
        }
        for (auto&& id : ids) {
            if (schemesByName.contains(id)) {
                continue;
            }
            auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(id);
            if (!tag) {
                continue;
            }
            NDrive::TScheme tagScheme = tag->GetScheme(Server);
            if (!tagScheme.IsEmpty()) {
                iFace.InsertValue(id, tagScheme.SerializeToJson());
            }
        }

        g.MutableReport().AddReportElement("iface_tag_implementations", std::move(iFace));
    }

    if (traits & EConstantTraits::ActionTypes) {
        TEventsGuard eg(g.MutableReport(), "action_types");
        NJson::TJsonValue userActionsJson;
        TSet<TString> keys;
        TUserAction::TFactory::GetRegisteredKeys(keys);
        for (auto&& tagActions : keys) {
            userActionsJson.AppendValue(tagActions);
        }
        g.MutableReport().AddReportElement("action_types", std::move(userActionsJson));
    }
    if (traits & EConstantTraits::AreaScheme) {
        TEventsGuard eg(g.MutableReport(), "area_scheme");
        g.MutableReport().AddReportElement("area_scheme", TArea::GetScheme(*Server).SerializeToJson());
    }
    if (traits & EConstantTraits::TagActions) {
        TEventsGuard eg(g.MutableReport(), "tag_actions");
        TVector<TString> carActions = {
            "ignore-telematics",
            "order",
            "fuel-card-number",
            "install"
        };
        NJson::TJsonValue tagActionsJson;
        for (ui32 i = 0; i < carActions.size(); ++i) {
            tagActionsJson.AppendValue(carActions[i]);
        }
        for (auto&& action : NDrive::ListActions()) {
            tagActionsJson.AppendValue(action);
        }
        g.MutableReport().AddReportElement("car_actions", std::move(tagActionsJson));
    }
    if (traits & EConstantTraits::TagTypes) {
        TEventsGuard eg(g.MutableReport(), "tag_types");
        TSet<TString> keys;
        ITag::TFactory::GetRegisteredKeys(keys);
        NJson::TJsonValue tagTypesJson;
        for (auto&& i : keys) {
            tagTypesJson.AppendValue(i);
        }
        g.MutableReport().AddReportElement("tag_types", std::move(tagTypesJson));
    }
    if (traits & EConstantTraits::TagFilters) {
        TEventsGuard eg(g.MutableReport(), "tag_filters");
        auto registeredTags = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();
        NJson::TJsonValue billingTagsFilters(NJson::JSON_MAP);
        for (auto&& tag : registeredTags) {
            const TBillingTagDescription* billingTag = dynamic_cast<const TBillingTagDescription*>(tag.second.Get());
            if (!!billingTag) {
                for (auto&& account : billingTag->GetAvailableAccounts()) {
                    billingTagsFilters[::ToString(account)].AppendValue(billingTag->GetName());
                }
            }
        }
        g.MutableReport().AddReportElement("iface_tag_filters", std::move(billingTagsFilters));
    }
    if (traits & EConstantTraits::TelematicsCommands) {
        TEventsGuard eg(g.MutableReport(), "telematics_commands");
        NJson::TJsonValue telematicsCommands;
        for (auto&& command : GetEnumAllValues<NDrive::NVega::ECommandCode>()) {
            telematicsCommands.AppendValue(ToString(command));
        }
        g.MutableReport().AddReportElement("telematics_commands", std::move(telematicsCommands));
    }
    if (traits & EConstantTraits::DeviceTags) {
        TEventsGuard eg(g.MutableReport(), "device_tags");
        auto tagDescriptions = DriveApi->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car);
        auto tagNames = NContainer::Keys(tagDescriptions);
        g.AddReportElement("device_tags", NJson::RangeToJson(tagNames.begin(), tagNames.end()));
    }
    if (traits & EConstantTraits::UserTags) {
        TEventsGuard eg(g.MutableReport(), "user_tags");
        auto userTagDescriptions = DriveApi->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User);
        auto userTagNames = NContainer::Keys(userTagDescriptions);
        g.AddReportElement("user_tags", NJson::RangeToJson(userTagNames.begin(), userTagNames.end()));
    }
    g.SetCode(HTTP_OK);
}

void TAddRoleProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    NStorage::TTableRecord row;
    R_ENSURE(row.DeserializeFromJson(requestData), ConfigHttpStatus.SyntaxErrorStatus, "cannot parse TableRecord from POST data");

    TDriveRoleHeader role;
    R_ENSURE(role.Parse(row), ConfigHttpStatus.UserErrorState, "cannot parse DriveRoleHeader from POST data");

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Role, role.GetName());

    auto session = BuildTx<NSQL::Writable>();
    if (!DriveApi->GetRolesManager()->UpsertRole(role, permissions->GetUserId(), session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    auto notifier = Config.GetNotifier() ? Server->GetNotifier(Config.GetNotifier()) : nullptr;
    if (notifier) {
        notifier->Notify(Sprintf(
            "Role <a href=\"%s\">%s</a> has been added by %s",
            TCarsharingUrl().RolePage(role.GetRoleId()).c_str(),
            role.GetRoleId().c_str(),
            permissions->GetHRReport().c_str()
        ));
    }
    g.SetCode(HTTP_OK);
}

void TRemoveRolesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto roleIds = GetStrings(cgi, "roles");

    for (auto&& roleId : roleIds) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Role, roleId);
    }

    auto tx = BuildTx<NSQL::Writable>();
    R_ENSURE(DriveApi->GetRolesManager()->RemoveRoles(roleIds, permissions->GetUserId(), tx), {}, "cannot RemoveRoles", tx);
    R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    auto notifier = Config.GetNotifier() ? Server->GetNotifier(Config.GetNotifier()) : nullptr;
    if (notifier && !roleIds.empty()) {
        auto message = TStringBuilder() << "Roles ";
        for (auto&& roleId : roleIds) {
            message << "<a href=\"" << TCarsharingUrl().RolePage(roleId) << "\">" << roleId << "</a> ";
        }
        message << "have been removed by " << permissions->GetHRReport();
        notifier->Notify(message);
    }
    g.SetCode(HTTP_OK);
}

void TConfirmActionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TVector<TString> propositionIds = GetStrings(requestData, "proposition_ids");

    auto session = BuildTx<NSQL::Writable>();

    auto optionalSessions = DriveApi->GetRolesManager()->GetActionsDB().GetPropositions().Get(propositionIds, session);
    R_ENSURE(optionalSessions, {}, "cannot get propositions", session);
    auto sessions = *optionalSessions;

    for (auto&& i : propositionIds) {
        auto it = sessions.find(i);
        R_ENSURE(it != sessions.end(), ConfigHttpStatus.UserErrorState, "incorrect propose_id " + i, EDriveSessionResult::IncorrectRequest);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Confirm, TAdministrativeAction::EEntity::Action, it->second->GetType());
    }

    for (auto&& i : propositionIds) {
        auto it = sessions.find(i);
        const EPropositionAcceptance confirmResult = DriveApi->GetRolesManager()->GetActionsDB().GetPropositions().Confirm(it->first, permissions->GetUserId(), session);
        if (confirmResult == EPropositionAcceptance::ConfirmWaiting) {
        } else if (confirmResult == EPropositionAcceptance::ReadyForCommit) {
            if (!DriveApi->GetRolesManager()->GetActionsDB().Upsert(it->second.Impl(), permissions->GetUserId(), session)) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        } else {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TRejectActionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const TVector<TString> propositionIds = GetStrings(requestData, "proposition_ids");

    auto session = BuildTx<NSQL::Writable>();

    auto optionalSessions = DriveApi->GetRolesManager()->GetActionsDB().GetPropositions().Get(propositionIds, session);
    R_ENSURE(optionalSessions, {}, "cannot get propositions", session);
    auto sessions = *optionalSessions;

    for (auto&& i : propositionIds) {
        auto it = sessions.find(i);
        R_ENSURE(it != sessions.end(), ConfigHttpStatus.UserErrorState, "incorrect propose_id " + i, EDriveSessionResult::IncorrectRequest);
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Reject, TAdministrativeAction::EEntity::Action, it->second->GetType());
    }

    for (auto&& i : propositionIds) {
        auto it = sessions.find(i);
        const EPropositionAcceptance confirmResult = DriveApi->GetRolesManager()->GetActionsDB().GetPropositions().Reject(it->first, permissions->GetUserId(), session);
        if (confirmResult != EPropositionAcceptance::Rejected) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TActionsHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TVector<TAtomicSharedPtr<TObjectEvent<TDBAction>>> result;
    R_ENSURE(DriveApi->GetRolesManager()->GetActionsDB().GetHistoryManager().GetEventsAll(TInstant::Zero(), result, TInstant::Zero()), ConfigHttpStatus.UnknownErrorStatus, "cannot_fetch_history");
    NJson::TJsonValue actionsHistory(NJson::JSON_ARRAY);
    TSet<TString> users;
    for (auto it = result.rbegin(); it != result.rend(); ++it) {
        if (CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Action, (**it)->GetName())) {
            actionsHistory.AppendValue((*it)->SerializeToJson());
            users.emplace((*it)->GetHistoryUserId());
        }
    }
    auto userData = DriveApi->GetUsersData()->FetchInfo(users);
    NJson::TJsonValue usersJson = NJson::JSON_MAP;
    for (auto&& [_, i] : userData) {
        usersJson.InsertValue(i.GetUserId(), i.GetReport(permissions->GetUserReportTraits()));
    }
    g.MutableReport().AddReportElement("actions_history", std::move(actionsHistory));
    g.MutableReport().AddReportElement("users", std::move(usersJson));

    g.SetCode(HTTP_OK);
}

void TAddActionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TUserAction::TPtr action = TUserAction::BuildFromJson(requestData);
    R_ENSURE(action, HTTP_BAD_REQUEST, "incorrect input post data for action construction");

    TActionsDB::TAvailableActions abilities = 0;
    if (CheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Action, action->GetType())) {
        abilities |= (ui32)TActionsDB::EAvailableActions::Add;
    }
    if (CheckAdmActions(permissions, TAdministrativeAction::EAction:: Modify, TAdministrativeAction::EEntity::Action, action->GetType())) {
        abilities |= (ui32)TActionsDB::EAvailableActions::Modify;
    }

    auto session = BuildTx<NSQL::Writable>();
    if (!DriveApi->GetRolesManager()->GetActionsDB().Upsert(action, permissions->GetUserId(), session, {}, abilities) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    } else {
        NDrive::INotifier::TPtr notifier = Config.GetNotifier() ? Server->GetNotifier(Config.GetNotifier()) : nullptr;
        if (notifier) {
            notifier->Notify(Sprintf(
                "UserAction <a href=\"%s\">%s</a> has been upserted by %s",
                TCarsharingUrl().ActionPage(action->GetName()).c_str(),
                action->GetName().c_str(),
                permissions->GetHRReport().c_str()
            ));
        }
    }
    g.SetCode(HTTP_OK);
}

void TProposeActionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    auto action = TUserAction::BuildFromJson(requestData);
    R_ENSURE(action, HTTP_BAD_REQUEST, "incorrect input post data for action construction");

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Propose, TAdministrativeAction::EEntity::Action, action->GetType());

    const TString& comment = Context->GetCgiParameters().Get("comment");
    R_ENSURE(comment, ConfigHttpStatus.SyntaxErrorStatus, "comment must being", EDriveSessionResult::IncorrectRequest);

    auto session = BuildTx<NSQL::Writable>();
    TObjectProposition<TDBAction> proposition(TDBAction(std::move(action)), 1);
    proposition.SetDescription(comment);
    if (DriveApi->GetRolesManager()->GetActionsDB().GetPropositions().Propose(proposition, permissions->GetUserId(), session) == EPropositionAcceptance::Problems || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }
    g.SetCode(HTTP_OK);
}

void TListActionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto actuality = GetTimestamp(cgi, "actuality", Context->GetRequestStartTime());
    const auto reportType = GetString(cgi, "report", false);
    const auto selectedActionIds = MakeSet(GetStrings(cgi, "action_id", false));

    bool serializeMeta = true;
    bool serializeEmptyPropositions = true;
    if (reportType == "compact") {
        serializeMeta = false;
        serializeEmptyPropositions = false;
    }

    TVector<TDBAction> actions = DriveApi->GetRolesManager()->GetActions(actuality);

    auto session = BuildTx<NSQL::ReadOnly>();

    auto optionalActionPropositions = DriveApi->GetRolesManager()->GetActionsDB().GetPropositions().Get(session);
    R_ENSURE(optionalActionPropositions, {}, "cannot get propositions", session);
    auto actionPropositions = *optionalActionPropositions;

    NJson::TJsonValue report(NJson::JSON_ARRAY);
    NJson::TJsonValue reportDeprecated(NJson::JSON_ARRAY);
    TSet<TString> actionIds;
    for (auto&& i : actions) {
        const auto& actionId = i->GetName();
        if (!selectedActionIds.empty() && !selectedActionIds.contains(actionId)) {
            continue;
        }
        if (!permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Action, TypeName, i->GetType())) {
            continue;
        }
        actionIds.emplace(actionId);
        NJson::TJsonValue actionJson = i->BuildJsonReport(serializeMeta);
        NJson::TJsonValue propositionsJson = NJson::JSON_ARRAY;
        for (auto&& proposition : actionPropositions) {
            if (proposition.second->GetName() == i->GetName()) {
                propositionsJson.AppendValue(proposition.second.BuildJsonReport());
            }
        }
        if (!propositionsJson.GetArray().empty() || serializeEmptyPropositions) {
            actionJson.InsertValue("propositions", std::move(propositionsJson));
        }
        if (i->GetDeprecated()) {
            reportDeprecated.AppendValue(std::move(actionJson));
        } else {
            report.AppendValue(std::move(actionJson));
        }
    }
    TSet<TString> propositionUserIds;
    for (auto&& i : actionPropositions) {
        i.second.FillUsers(propositionUserIds);
        if (actionIds.contains(i.second->GetName())) {
            continue;
        }
        if (!permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Action, TypeName, i.second->GetType())) {
            continue;
        }
        actionIds.emplace(i.second->GetName());
        NJson::TJsonValue actionJson = NJson::JSON_MAP;
        actionJson.InsertValue("action_id", i.second->GetName());
        actionJson.InsertValue("is_proposition_only", true);
        NJson::TJsonValue& propositionsJson = actionJson.InsertValue("propositions", NJson::JSON_ARRAY);
        for (auto&& proposition : actionPropositions) {
            if (proposition.second->GetName() == i.second->GetName()) {
                propositionsJson.AppendValue(proposition.second.BuildJsonReport());
            }
        }
        if (i.second->GetDeprecated()) {
            reportDeprecated.AppendValue(std::move(actionJson));
        } else {
            report.AppendValue(std::move(actionJson));
        }
    }
    g.MutableReport().AddReportElement("report", std::move(report));
    g.MutableReport().AddReportElement("report_deprecated", std::move(reportDeprecated));

    auto gPropositionUsers = DriveApi->GetUsersData()->FetchInfo(propositionUserIds, session);
    NJson::TJsonValue usersJson = NJson::JSON_ARRAY;
    for (auto&& user : gPropositionUsers) {
        usersJson.AppendValue(user.second.GetReport(permissions->GetUserReportTraits()));
    }
    g.MutableReport().AddReportElement("proposition_users", std::move(usersJson));
    g.SetCode(HTTP_OK);
}

void TRemoveActionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TVector<TString> actions = GetStrings(cgi, "actions");
    const bool erase = GetValue<bool>(cgi, "full_remove", false).GetOrElse(false);

    for (auto&& action : actions) {
        auto optionalAction = DriveApi->GetRolesManager()->GetAction(action);
        if (optionalAction && *optionalAction) {
            const auto& act = *optionalAction;
            const TString& type = act->GetType();
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Action, type);
        }
    }

    auto session = BuildTx<NSQL::Writable>();
    if (erase) {
        if (!DriveApi->GetRolesManager()->RemoveActions(actions, permissions->GetUserId(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    } else {
        if (!DriveApi->GetRolesManager()->DeprecateActions(actions, permissions->GetUserId(), session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }
    if (!session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    } else {
        NDrive::INotifier::TPtr notifier = Config.GetNotifier() ? Server->GetNotifier(Config.GetNotifier()) : nullptr;
        if (notifier) {
            notifier->Notify("UserActions " + JoinStrings(actions, ",") + " have been removed by " + permissions->GetHRReport());
        }
        g.SetCode(HTTP_OK);
    }
}

void TViewPermissionsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString userId = cgi.Has("user_id") ? GetUUID(cgi, "user_id", false) : permissions->GetUserId();
    R_ENSURE(userId, ConfigHttpStatus.UserErrorState, "user_id is empty");

    TUserPermissions::TPtr viewed;
    if (userId == permissions->GetUserId()) {
        viewed = permissions;
    } else {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::User, userId);
        viewed = Server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures(), Context->GetRequestStartTime());
    }
    R_ENSURE(viewed, ConfigHttpStatus.EmptySetStatus, "no permissions for user_id " << userId);

    TJsonReport& report = g.MutableReport();
    report.AddReportElement("user_id", viewed->GetUserId());
    {
        NJson::TJsonValue generatorsJson = NJson::JSON_ARRAY;
        auto generators = permissions->GetAdministrativeInstances(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::PromoCodes);
        if (generators.Defined()) {
            for (auto&& i : *generators) {
                generatorsJson.AppendValue(i);
            }
        } else {
            for (auto&& action : Server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionNamesWithType<IPromoCodeGenerator>(/*reportDeprecated=*/false)) {
                generatorsJson.AppendValue(action);
            }
        }
        g.MutableReport().AddReportElement("promo_generators", std::move(generatorsJson));
    }
    {
        NJson::TJsonValue roles(NJson::JSON_ARRAY);
        for (auto&& i : viewed->GetRolesByUser()) {
            if (i.GetActive()) {
                roles.AppendValue(i.GetRoleId());
            }
        }
        report.AddReportElement("roles", std::move(roles));
    }
    {
        NJson::TJsonValue icons(NJson::JSON_ARRAY);
        for (auto&& i : viewed->GetActionsActual()) {
            const TString& icon = i->GetAdminIcon();
            if (icon) {
                icons.AppendValue(icon);
            }
        }
        report.AddReportElement("admin_icons", std::move(icons));
    }
    {
        NJson::TJsonValue actions(NJson::JSON_ARRAY);
        for (auto&& i : viewed->GetActionsActual()) {
            actions.AppendValue(i->GetName());
        }
        report.AddReportElement("actions", std::move(actions));
    }
    {
        NJson::TJsonValue offerBuilders(NJson::JSON_ARRAY);
        for (auto&& i : viewed->GetOfferBuilders()) {
            if (i) {
                offerBuilders.AppendValue(i->GetName());
            }
        }
        report.AddReportElement("offer_builders", std::move(offerBuilders));
    }
    {
        NJson::TJsonValue offerCorrectors(NJson::JSON_ARRAY);
        for (auto&& i : viewed->GetOfferCorrections()) {
            if (i) {
                offerCorrectors.AppendValue(i->GetName());
            }
        }
        report.AddReportElement("offer_correctors", std::move(offerCorrectors));
    }
    {
        NJson::TJsonValue documents(NJson::JSON_ARRAY);
        for (auto&& i : viewed->GetAvailableDocuments()) {
            documents.AppendValue(i);
        }
        report.AddReportElement("available_documents", std::move(documents));
    }
    for (auto&& action : GetEnumAllValues<TTagAction::ETagAction>()) {
        NJson::TJsonValue tags(NJson::JSON_ARRAY);
        for (auto&& i : viewed->GetTagNamesByAction(action)) {
            tags.AppendValue(i);
        }
        report.AddReportElement(ToString(action) + "_tags", std::move(tags));
    }
    {
        NJson::TJsonValue evolutionsJson = NJson::JSON_MAP;
        for (auto&& from : *viewed->GetEvolutions()) {
            NJson::TJsonValue& toJson = evolutionsJson.InsertValue(from.first, NJson::JSON_ARRAY);
            for (auto&& to : from.second) {
                NJson::TJsonValue& info = toJson.AppendValue(NJson::JSON_MAP);
                info.InsertValue("to", to.first);
                info.InsertValue("details", to.second.GetPublicJsonReport());
            }
        }

        report.AddReportElement("evolutions", std::move(evolutionsJson));
    }
    {
        report.AddReportElement("resources_locked_limit", viewed->LockedResourcesLimit());
        auto carRegistryReportTraits = viewed->GetCarRegistryReportTraits();
        report.AddReportElement("car_registry_traits", NJson::ToJson(NJson::BitMask<NRegistryDocumentReport::EReportTraits>(carRegistryReportTraits)));
        auto carReportTraits = viewed->GetDeviceReportTraits();
        report.AddReportElement("car_report_traits", NJson::ToJson(NJson::BitMask<NDeviceReport::EReportTraits>(carReportTraits)));
        auto carSearchTraits = viewed->GetDeviceSearchTraits();
        report.AddReportElement("car_search_traits", NJson::ToJson(NJson::BitMask<NDeviceReport::EReportTraits>(carSearchTraits)));
        auto userReportTraits = viewed->GetUserReportTraits();
        report.AddReportElement("user_report_traits", NJson::ToJson(NJson::BitMask<NUserReport::EReportTraits>(userReportTraits)));
        auto userSearchTraits = viewed->GetUserSearchTraits();
        report.AddReportElement("user_search_traits", NJson::ToJson(NJson::BitMask<NUserReport::EReportTraits>(userSearchTraits)));
    }

    g.SetCode(HTTP_OK);
}

void TRootifyProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(permissions, ConfigHttpStatus.UnknownErrorStatus, "null permissions");
    const TString& userId = permissions->GetUserId();
    const TInstant timestamp = Context->GetRequestStartTime();

    const ISettings& settings = Server->GetSettings();
    const auto rootAdministrativeActionName = settings.GetValue<TString>("root.administrative_action.name").GetOrElse("__root_administrative");
    const auto rootSettingsActionName = settings.GetValue<TString>("root.settings_action.name").GetOrElse("__root_settings");
    const auto rootTagActionName = settings.GetValue<TString>("root.tag_action.name").GetOrElse("__root_tag_action");
    const auto rootInfoAccessActionName = settings.GetValue<TString>("root.info_access.name").GetOrElse("__root_info_access");
    const auto rootRoleName = settings.GetValue<TString>("root.role.name").GetOrElse("__root");

    auto rolesManager = DriveApi->GetRolesManager();
    auto session = BuildTx<NSQL::Writable>();

    auto userFetchResult = DriveApi->GetUsersData()->FetchInfo(userId, session);
    R_ENSURE(userFetchResult, ConfigHttpStatus.UnknownErrorStatus, "cannot fetch user data", session);
    R_ENSURE(!userFetchResult.empty(), ConfigHttpStatus.UnknownErrorStatus, "no such user", session);

    if (auto userData = std::move(userFetchResult.MutableResult().begin()->second); userData.GetStatus() == NDrive::UserStatusOnboarding) {
        userData.SetStatus(NDrive::UserStatusActive);
        if (!DriveApi->GetUsersData()->UpdateUser(userData, userId, session)) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    auto rootAdministrativeAction = rolesManager->GetAction(rootAdministrativeActionName, timestamp);
    if (!rootAdministrativeAction) {
        auto action = MakeAtomicShared<TAdministrativeAction>();
        action->SetName(rootAdministrativeActionName);
        action->SetEnabled(true);
        auto allActions = GetEnumAllValues<TAdministrativeAction::EAction>();
        auto allEntities = GetEnumAllValues<TAdministrativeAction::EEntity>();
        action->MutableActions() = MakeSet(allActions);
        action->MutableEntities() = MakeSet(allEntities);
        R_ENSURE(
            rolesManager->GetActionsDB().ForceUpsert(action, userId, session),
            ConfigHttpStatus.UnknownErrorStatus,
            TStringBuilder() << "cannot upsert action " << rootAdministrativeActionName, session
        );
    }

    auto rootSettingsAction = rolesManager->GetAction(rootSettingsActionName, timestamp);
    if (!rootSettingsAction) {
        auto action = MakeAtomicShared<TAdministrativeAction>();
        action->SetName(rootSettingsActionName);
        action->SetEnabled(true);
        action->MutableActions() = MakeSet(GetEnumAllValues<TAdministrativeAction::EAction>());
        action->MutableEntities().insert(TAdministrativeAction::EEntity::Settings);
        action->MutableSettingPrefixes().insert("*");
        R_ENSURE(
            rolesManager->GetActionsDB().ForceUpsert(action, userId, session),
            ConfigHttpStatus.UnknownErrorStatus,
            TStringBuilder() << "cannot upsert action " << rootSettingsActionName, session
        );
    }

    auto rootTagAction = rolesManager->GetAction(rootTagActionName, timestamp);
    if (!rootTagAction) {
        auto action = MakeAtomicShared<TTagAction>();
        action->SetName(rootTagActionName);
        action->SetEnabled(true);
        action->SetTagName("$.*");
        for (auto&& i : GetEnumAllValues<TTagAction::ETagAction>()) {
            action->AddTagAction(i);
        }
        R_ENSURE(
            rolesManager->GetActionsDB().ForceUpsert(action, userId, session),
            ConfigHttpStatus.UnknownErrorStatus,
            TStringBuilder() << "cannot upsert action " << rootTagActionName, session
        );
    }

    auto rootInfoAccessAction = rolesManager->GetAction(rootInfoAccessActionName, timestamp);
    if (!rootInfoAccessAction) {
        auto action = MakeAtomicShared<TInfoAccessAction>();
        action->SetName(rootInfoAccessActionName);
        action->SetEnabled(true);
        action->MutableUserTraits() = NUserReport::ReportAll;
        R_ENSURE(
            rolesManager->GetActionsDB().ForceUpsert(action, userId, session),
            ConfigHttpStatus.UnknownErrorStatus,
            TStringBuilder() << "cannot upsert action " << rootInfoAccessActionName, session
        );
    }

    auto rootRole = rolesManager->GetRolesDB().GetObject(rootRoleName, timestamp);
    if (!rootRole) {
        TDriveRoleHeader role;
        role.SetName(rootRoleName);
        R_ENSURE(rolesManager->UpsertRole(role, userId, session), ConfigHttpStatus.UnknownErrorStatus, "cannot UpsertRole", session);

        TLinkedRoleActionHeader roleAction;
        roleAction.SetRoleId(rootRoleName);
        roleAction.SetSlaveObjectId(rootAdministrativeActionName);
        R_ENSURE(
            rolesManager->GetRolesInfoDB().GetRoleActions().Link(roleAction, userId, session),
            ConfigHttpStatus.UnknownErrorStatus,
            TStringBuilder() << "cannot link role action " << rootAdministrativeActionName, session
        );

        roleAction.SetSlaveObjectId(rootSettingsActionName);
        R_ENSURE(
            rolesManager->GetRolesInfoDB().GetRoleActions().Link(roleAction, userId, session),
            ConfigHttpStatus.UnknownErrorStatus,
            TStringBuilder() << "cannot link role action " << rootSettingsActionName, session
        );

        roleAction.SetSlaveObjectId(rootTagActionName);
        R_ENSURE(
            rolesManager->GetRolesInfoDB().GetRoleActions().Link(roleAction, userId, session),
            ConfigHttpStatus.UnknownErrorStatus,
            TStringBuilder() << "cannot link role action " << rootTagActionName, session
        );

        roleAction.SetSlaveObjectId(rootInfoAccessActionName);
        R_ENSURE(
            rolesManager->GetRolesInfoDB().GetRoleActions().Link(roleAction, userId, session),
            ConfigHttpStatus.UnknownErrorStatus,
            TStringBuilder() << "cannot link role action " << rootInfoAccessActionName, session
        );
    }

    auto userRoles = DriveApi->GetUsersData()->GetRoles().RestoreUserRoles(userId, session);
    R_ENSURE(userRoles, ConfigHttpStatus.UnknownErrorStatus, "cannot restore user roles ", session);
    const auto& roles = userRoles->GetRoles();
    const auto hasRootRole = std::find_if(roles.begin(), roles.end(), [&rootRoleName](const TUserRole& role) {
        return role.GetRoleId() == rootRoleName;
    }) != roles.end();

    if (!hasRootRole) {
        TUserRole role;
        role.SetRoleId(rootRoleName);
        role.SetUserId(userId);
        role.SetForced(true);
        R_ENSURE(
            rolesManager->UpsertRoleForUser(role, userId, session),
            ConfigHttpStatus.UnknownErrorStatus,
            "cannot UpsertRoleForUser",
            session
        );
    }

    R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "cannot Commit", session);
    g.SetCode(HTTP_OK);
}
