#pragma once

#include "config.h"

#include <drive/backend/processors/service_app/config.h>
#include <drive/backend/processors/service_app/processor.h>

#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/roles/roles.h>
#include <drive/backend/users/login.h>

#include <util/generic/serialized_enum.h>
#include <util/string/split.h>

class TAddUserRoleProcessor: public TAppCommonProcessor<TAddUserRoleProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TAddUserRoleProcessor, TEmptyConfig>;

public:
    TAddUserRoleProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "user_roles/add";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        NStorage::TTableRecord row;
        Y_ENSURE_EX(row.DeserializeFromJson(requestData), TCodedException(ConfigHttpStatus.UserErrorState) << "incorrect input post data");

        if (!row.Has("user_id")) {
            row.Set("user_id", permissions->GetUserId());
        }

        TUserRole userRole;
        Y_ENSURE_EX(userRole.Parse(row), TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "incorrect input post data for role construction");

        const TSet<TString> actions = Server->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetActions({userRole.GetRoleId()});
        const TVector<TDBAction> actionObjects = Server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetCachedObjectsVector(actions);
        for (auto&& i : actionObjects) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Action, i->GetName(), i->GetGrouppingTags());
        }

        auto enforceIdm = Server->GetSettings().GetValue<bool>("idm.enforce").GetOrElse(true);
        auto session = BuildTx<NSQL::Writable>();

        TMap<TString, TDriveRoleHeader> fetchRoles;
        ReqCheckCondition(DriveApi->GetRolesManager()->GetRolesDB().GetAllObjectsFromCache(fetchRoles, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

        const TDriveRoleHeader* hRole = fetchRoles.FindPtr(userRole.GetRoleId());
        R_ENSURE(hRole, ConfigHttpStatus.EmptySetStatus, "incorrect_user_roles", EDriveSessionResult::IncorrectRequest);
        R_ENSURE(!enforceIdm || !hRole->GetIsIDM(), ConfigHttpStatus.ConflictRequest, "please use IDM to manage this role", NDrive::MakeError("idm_required"), session);

        auto allowStaffAccounts = GetHandlerSettingDef<bool>("allow_staff_accounts", false);
        if (!allowStaffAccounts) {
            auto userData = Server->GetDriveAPI()->GetUserManager().GetCachedObject(userRole.GetUserId());
            R_ENSURE(userData, HTTP_INTERNAL_SERVER_ERROR, "user data not found", session);
            R_ENSURE(!enforceIdm || !userData->IsStaffAccount(), ConfigHttpStatus.ConflictRequest, "please use IDM to to manage staff accounts roles", NDrive::MakeError("idm_required_for_this_account"), session);
        }

        if (userRole.GetActive()) {
            const ui32 group = hRole->GetGroup();
            if (group) {
                auto optionalUserRoles = DriveApi->GetUsersData()->GetRoles().RestoreUserRoles(userRole.GetUserId(), session);
                R_ENSURE(optionalUserRoles, {}, "cannot RestoreUserRoles for " << userRole.GetUserId(), session);
                {
                    for (auto&& uRole : optionalUserRoles->GetRoles()) {
                        const TDriveRoleHeader* h = fetchRoles.FindPtr(uRole.GetRoleId());
                        if (h && h->GetGroup() == group && uRole.GetActive()) {
                            userRole.SetActive(false);
                            break;
                        }
                    }
                }
            }
        }
        if (!DriveApi->GetRolesManager()->UpsertRoleForUser(userRole, permissions->GetUserId(), session)) {
            g.MutableReport().AddReportElement("error", "Incorrect link");
            g.MutableReport().AddReportElement("error_details", session.GetTransaction()->GetErrors().GetStringReport());
            g.SetCode(ConfigHttpStatus.UserErrorState);
        } else if (!session.Commit()) {
            g.MutableReport().AddReportElement("error", "Cannot add into database");
            g.MutableReport().AddReportElement("error_details", session.GetTransaction()->GetErrors().GetStringReport());
            g.SetCode(ConfigHttpStatus.UserErrorState);
        } else {
            g.SetCode(HTTP_OK);
        }
    }
};

class TListUserRolesProcessor: public TAppCommonProcessor<TListUserRolesProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TListUserRolesProcessor, TEmptyConfig>;

public:
    TListUserRolesProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "user_roles/list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) {
        const TCgiParameters& cgi = Context->GetCgiParameters();
        const TString userIdRequest = cgi.Has("user_id") ? GetUUID(cgi, "user_id") : permissions->GetUserId();
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::User, userIdRequest);

        auto session = BuildTx<NSQL::ReadOnly>();
        auto gUserRoles = DriveApi->GetUsersData()->GetRoles().RestoreUserRoles(userIdRequest, session);
        R_ENSURE(gUserRoles, {}, "cannot RestoreUserRoles " << userIdRequest, session);

        TSet<TString> roleIds;
        for (auto&& i : gUserRoles->GetRoles()) {
            roleIds.emplace(i.GetRoleId());
        }
        TMap<TString, TDriveRoleHeader> gUserRolesInfo;
        ReqCheckCondition(DriveApi->GetRolesManager()->GetRolesDB().GetCustomObjectsMap(roleIds, gUserRolesInfo, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

        NJson::TJsonValue rolesJson(NJson::JSON_ARRAY);
        for (auto&& i : gUserRoles->GetRoles()) {
            TUserRoleInfo roleInfo(i, gUserRolesInfo.FindPtr(i.GetRoleId()));
            rolesJson.AppendValue(roleInfo.BuildJsonReport());
        }
        g.MutableReport().AddReportElement("report", std::move(rolesJson));
        g.SetCode(HTTP_OK);
    }
};

class TListUserActionsProcessor: public TAppCommonProcessor<TListUserActionsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TListUserActionsProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_actions/list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUserRolesHistoryProcessor: public TAppCommonProcessor<TUserRolesHistoryProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserRolesHistoryProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_roles/history";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TUserRoleActivationProcessor: public TAppCommonProcessor<TUserRoleActivationProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserRoleActivationProcessor, TEmptyConfig>;

public:
    TUserRoleActivationProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "user_role_activation";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        const TCgiParameters& cgi = Context->GetCgiParameters();
        const TString& roleId = cgi.Get("role_id");
        const bool enabled = IsTrue(cgi.Get("enabled"));

        TString userId = permissions->GetUserId();
        bool validatedPermissions = false;
        if (cgi.Has("user_id")) {
            userId = cgi.Get("user_id");
            validatedPermissions = true;
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::User, userId);
        }

        auto session = BuildTx<NSQL::Writable>();
        auto optionalUserRoles = DriveApi->GetUsersData()->GetRoles().RestoreUserRoles(userId, session);
        R_ENSURE(optionalUserRoles, HTTP_INTERNAL_SERVER_ERROR, "cannot RestoreUserRoles", session);
        R_ENSURE(!optionalUserRoles->GetRoles().empty(), HTTP_NOT_FOUND, "user has no roles", session);

        {
            const TSet<TString> actions = Server->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetActions({roleId});
            const TVector<TDBAction> actionObjects = Server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetCachedObjectsVector(actions);
            for (auto&& i : actionObjects) {
                ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Action, i->GetName(), i->GetGrouppingTags());
            }
        }

        TVector<TDriveRoleHeader> roles = DriveApi->GetRolesManager()->GetRoles(Context->GetRequestStartTime());
        TVector<TUserRole> userRoles = optionalUserRoles->GetRoles();

        ui32 group = 0;
        TMap<TString, TDriveRoleHeader> rolesById;
        for (auto&& role : roles) {
            rolesById.emplace(role.GetName(), role);
        }
        {
            auto it = rolesById.find(roleId);
            R_ENSURE(it != rolesById.end(), ConfigHttpStatus.UserErrorState, "incorrect role_id", EDriveSessionResult::IncorrectRequest);
            group = it->second.GetGroup();
        }

        for (auto&& i : userRoles) {
            auto it = rolesById.find(i.GetRoleId());
            if (it == rolesById.end()) {
                continue;
            }
            auto& roleDescription = it->second;
            if ((i.GetRoleId() == roleId) || (enabled && group && (roleDescription.GetGroup() == group))) {
                const bool newActive = enabled && (i.GetRoleId() == roleId);
                if (i.GetActive() != newActive) {
                    R_ENSURE(validatedPermissions || roleDescription.GetIsPublic(), ConfigHttpStatus.PermissionDeniedStatus, "no_permissions", EDriveSessionResult::NoUserPermissions);
                    R_ENSURE(validatedPermissions || roleDescription.GetOptional(), ConfigHttpStatus.UserErrorState, "no_optional", NDrive::MakeError("role_not_optional"), session);
                    i.SetActive(newActive);
                    if (!DriveApi->GetUsersData()->GetRoles().Link(i, permissions->GetUserId(), session)) {
                        session.DoExceptionOnFail(ConfigHttpStatus);
                    }
                }
            }
        }
        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        g.SetCode(HTTP_OK);
    }
};

class TCurrentUserRolesProcessor: public TAppCommonProcessor<TCurrentUserRolesProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCurrentUserRolesProcessor, TEmptyConfig>;

public:
    TCurrentUserRolesProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "current_user_roles";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        auto session = BuildTx<NSQL::ReadOnly>();
        auto gResult = DriveApi->GetUsersData()->GetRoles().RestoreUserRoles(permissions->GetUserId(), session);
        R_ENSURE(!!gResult, HTTP_INTERNAL_SERVER_ERROR, "cannot RestoreUserRoles", session);
        TVector<TDriveRoleHeader> roles = DriveApi->GetRolesManager()->GetRoles(Context->GetRequestStartTime());
        NJson::TJsonValue rolesJson(NJson::JSON_ARRAY);
        for (auto&& i : gResult->GetRoles()) {
            for (auto&& role : roles) {
                if (role.GetName() == i.GetRoleId()) {
                    if (role.GetIsPublic()) {
                        NJson::TJsonValue roleDescription;
                        roleDescription.InsertValue("role_description", role.BuildJsonReport());
                        roleDescription.InsertValue("user_role", i.BuildJsonReport());
                        rolesJson.AppendValue(roleDescription);
                    }
                    break;
                }
            }
        }
        g.MutableReport().AddReportElement("report", std::move(rolesJson));
        g.SetCode(HTTP_OK);
    }
};

class TCurrentUserActionsProcessor: public TAppCommonProcessor<TCurrentUserActionsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TCurrentUserActionsProcessor, TEmptyConfig>;

public:
    TCurrentUserActionsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "current_user_actions";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        NJson::TJsonValue rolesJson(NJson::JSON_ARRAY);
        for (auto&& action : permissions->GetActionsActual()) {
            rolesJson.AppendValue(action.BuildJsonReport());
        }
        g.MutableReport().AddReportElement("report", std::move(rolesJson));
        g.SetCode(HTTP_OK);
    }
};

class TListUsersByRolesProcessor: public TAppCommonProcessor<TListUsersByRolesProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TListUsersByRolesProcessor, TEmptyConfig>;

public:
    TListUsersByRolesProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "users_by_roles";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::User);
        const TCgiParameters& cgi = Context->GetCgiParameters();
        TVector<TString> rolesRequest = StringSplitter(cgi.Get("roles")).Split(',').SkipEmpty();
        for (auto&& i : rolesRequest) {
            i = StripInPlace(i);
        }

        auto limit = GetValue<ui32>(cgi, "limit", false).GetOrElse(0);
        size_t page = GetValue<ui32>(cgi, "page", false).GetOrElse(0);
        size_t pageSize = GetValue<ui32>(cgi, "page_size", false).GetOrElse(50);

        auto session = BuildTx<NSQL::ReadOnly>();

        TVector<TString> users;
        DriveApi->GetUsersData()->GetRoles().GetUsersWithRoles(MakeSet(rolesRequest), users, false, limit);

        size_t totalResults = users.size();
        size_t start = page * pageSize;
        size_t end = Min((page + 1) * pageSize, users.size());
        TSet<TString> matchedUserIds;
        for (size_t i = start; i < end; ++i) {
            matchedUserIds.emplace(users[i]);
        }
        auto gUsers = DriveApi->GetUsersData()->FetchInfo(matchedUserIds, session);
        NJson::TJsonValue usersJson(NJson::JSON_ARRAY);
        for (auto&& i : gUsers.GetResult()) {
            usersJson.AppendValue(i.second.GetReport());
        }

        NJson::TJsonValue paginationReport(NJson::JSON_MAP);
        paginationReport["total_results"] = totalResults;
        paginationReport["total_pages"] = totalResults / pageSize + (totalResults % pageSize > 0);
        paginationReport["current_page"] = page;

        g.MutableReport().AddReportElement("report", std::move(usersJson));
        g.MutableReport().AddReportElement("pagination", std::move(paginationReport));
        g.SetCode(HTTP_OK);
    }
};

class TRemoveUserRolesProcessor: public TAppCommonProcessor<TRemoveUserRolesProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRemoveUserRolesProcessor, TEmptyConfig>;

public:
    TRemoveUserRolesProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "user_roles/remove";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) override {
        const TCgiParameters& cgi = Context->GetCgiParameters();
        const TString roleNames = cgi.Get("roles");
        TVector<TString> roles = SplitString(roleNames, ",");
        const TString userIdRequest = cgi.Has("user_id") ? cgi.Get("user_id") : permissions->GetUserId();
        Y_ENSURE_EX(!!userIdRequest && !GetUuid(userIdRequest).IsEmpty(),
            TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "incorrect user identification info");

        for (auto&& role : roles) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::User, role);
        }

        auto enforceIdm = Server->GetSettings().GetValue<bool>("idm.enforce").GetOrElse(true);
        if (enforceIdm) {
            TMap<TString, TDriveRoleHeader> fetchRoles;
            ReqCheckCondition(DriveApi->GetRolesManager()->GetRolesDB().GetAllObjectsFromCache(fetchRoles, Context->GetRequestStartTime()), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
            for (auto&& role : roles) {
                const TDriveRoleHeader* roleHeader = fetchRoles.FindPtr(role);
                if (!roleHeader) {
                    continue;
                }
                R_ENSURE(!roleHeader->GetIsIDM(), ConfigHttpStatus.ConflictRequest, "please use IDM to manage this role", NDrive::MakeError("idm_required"));
            }
        }

        auto session = BuildTx<NSQL::Writable>();
        if (!DriveApi->GetRolesManager()->RemoveRolesForUser(userIdRequest, roles, permissions->GetUserId(), session)) {
            g.MutableReport().AddReportElement("error", "Cannot remove link");
            g.MutableReport().AddReportElement("error_details", session.GetTransaction()->GetErrors().GetStringReport());
            g.SetCode(ConfigHttpStatus.UserErrorState);
        } else if (!session.Commit()) {
            g.MutableReport().AddReportElement("error", "Cannot remove link from db database");
            g.MutableReport().AddReportElement("error_details", session.GetTransaction()->GetErrors().GetStringReport());
            g.SetCode(ConfigHttpStatus.UserErrorState);
        } else {
            g.SetCode(HTTP_OK);
        }
    }
};

class TAddRoleActionProcessor: public TAppCommonProcessor<TAddRoleActionProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TAddRoleActionProcessor, TProcessorWithNotifierConfig>;

public:
    TAddRoleActionProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "roles_action/add";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TListRoleActionsProcessor: public TAppCommonProcessor<TListRoleActionsProcessor> {
private:
    using TBase = TAppCommonProcessor<TListRoleActionsProcessor>;

public:
    TListRoleActionsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "roles_action/list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) {
        const TCgiParameters& cgi = Context->GetCgiParameters();

        const TString roleName = cgi.Get("role");

        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::Role);

        TVector<TDriveRoleActions> roleActionsInfo = DriveApi->GetRolesManager()->GetRolesInfoDB().GetRoleActions().GetInfo({roleName}, Context->GetRequestStartTime());
        TVector<TDriveRoleRoles> roleRolesInfo = DriveApi->GetRolesManager()->GetRolesInfoDB().GetRoleRoles().GetInfo({roleName}, Context->GetRequestStartTime());
        {
            NJson::TJsonValue wideReport;
            if (roleActionsInfo.size()) {
                wideReport.InsertValue("actions", roleActionsInfo.front().BuildJsonReport());
            }
            if (roleRolesInfo.size()) {
                wideReport.InsertValue("roles", roleRolesInfo.front().BuildJsonReport());
            }
            g.MutableReport().AddReportElement("wide_report", std::move(wideReport));
        }

        {
            NJson::TJsonValue jsonActions(NJson::JSON_ARRAY);
            for (auto&& i : roleActionsInfo) {
                jsonActions.AppendValue(i.BuildJsonReport());
            }
            g.MutableReport().AddReportElement("report", std::move(jsonActions));
        }
        g.SetCode(HTTP_OK);
    }
};

class TRemoveRoleActionsProcessor: public TAppCommonProcessor<TRemoveRoleActionsProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TRemoveRoleActionsProcessor, TProcessorWithNotifierConfig>;

public:
    TRemoveRoleActionsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "roles_action/remove";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TAddRoleProcessor: public TAppCommonProcessor<TAddRoleProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TAddRoleProcessor, TProcessorWithNotifierConfig>;

public:
    TAddRoleProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "manage_roles/add";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData);
};

class TListRolesProcessor: public TAppCommonProcessor<TListRolesProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TListRolesProcessor, TEmptyConfig>;

public:
    TListRolesProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "manage_roles/list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) {
        const auto& cgi = Context->GetCgiParameters();
        const auto actuality = GetTimestamp(cgi, "actuality", Context->GetRequestStartTime());
        const auto actionIds = MakeSet(GetStrings(cgi, "action_id", false));
        const auto selectedRoleIds = MakeSet(GetStrings(cgi, "role_id", false));
        const auto reportType = GetString(cgi, "report", false);

        bool serializeSlaveObjects = true;
        if (reportType == "compact") {
            serializeSlaveObjects = false;
        }

        auto session = BuildTx<NSQL::ReadOnly>();
        TVector<TDriveRoleHeader> roles;
        if (actionIds.empty()) {
            roles = DriveApi->GetRolesManager()->GetRoles(actuality);
        } else {
            roles = DriveApi->GetRolesManager()->GetRolesWithActions(actionIds, actuality);
        }

        TSet<TString> roleIds;
        for (auto&& roleHeader : roles) {
            const auto& roleId = roleHeader.GetName();
            if (!selectedRoleIds.empty() && !selectedRoleIds.contains(roleId)) {
                continue;
            }
            if (!permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Role, TypeName, roleHeader.GetRoleId())) {
                continue;
            }
            roleIds.emplace(roleId);
        }

        TVector<TRoleSnapshot> snapshots;
        ReqCheckCondition(DriveApi->GetRolesManager()->GetRoleSnapshots(roleIds, actuality, snapshots), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);

        std::multimap<TString, TString> parentRoleIds;
        for (auto&& i : snapshots) {
            for (auto&& slaveRole : i.GetRoleRoles().GetSlaves()) {
                const auto& parentRoleId = slaveRole.GetRoleId();
                const auto& slaveRoleId = slaveRole.GetSlaveObjectId();
                parentRoleIds.emplace(slaveRoleId, parentRoleId);
            }
        }

        NJson::TJsonValue snapshotsJson(NJson::JSON_ARRAY);
        for (auto&& i : snapshots) {
            auto report = i.BuildJsonReport(true, serializeSlaveObjects);
            if (actionIds.empty() && selectedRoleIds.empty()) {
                auto& parentRoleIdReport = report.InsertValue("parent_role_ids", NJson::JSON_ARRAY);
                const auto& roleId = i.GetRole().GetRoleId();
                for (auto&& [_, parentRoleId] : MakeIteratorRange(parentRoleIds.equal_range(roleId))) {
                    parentRoleIdReport.AppendValue(parentRoleId);
                }
            }
            snapshotsJson.AppendValue(std::move(report));
        }
        g.MutableReport().AddReportElement("report", std::move(snapshotsJson));

        NJson::TJsonValue propositionsJson = NJson::JSON_ARRAY;
        auto optionalSessions = DriveApi->GetRolesManager()->GetSnapshotPropositions().Get(session);
        R_ENSURE(optionalSessions, {}, "cannot get propositions", session);
        auto sessions = *optionalSessions;

        TSet<TString> userIds;
        for (auto&& i : sessions) {
            if (!permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Role, TypeName, i.second.GetRole().GetRoleId())) {
                continue;
            }
            i.second.FillUsers(userIds);
            propositionsJson.AppendValue(i.second.BuildJsonReport());
        }
        g.MutableReport().AddReportElement("propositions", std::move(propositionsJson));

        auto gUsers = DriveApi->GetUsersData()->FetchInfo(userIds, session);
        NJson::TJsonValue usersJson = NJson::JSON_MAP;
        for (auto&& i : gUsers.GetResult()) {
            usersJson.InsertValue(i.first, i.second.GetReport(permissions->GetUserReportTraits()));
        }
        g.MutableReport().AddReportElement("users", std::move(usersJson));
        g.SetCode(HTTP_OK);
    }
};

class TUpsertRoleSnapshotProcessor: public TAppCommonProcessor<TUpsertRoleSnapshotProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TUpsertRoleSnapshotProcessor, TProcessorWithNotifierConfig>;

public:
    TUpsertRoleSnapshotProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server)
    {
    }

    static TString GetTypeName() {
        return "manage_roles/snapshot/upsert";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TProposeRoleSnapshotProcessor: public TAppCommonProcessor<TProposeRoleSnapshotProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TProposeRoleSnapshotProcessor, TEmptyConfig>;

public:
    TProposeRoleSnapshotProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "manage_roles/snapshot/propose";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Propose, TAdministrativeAction::EEntity::Role);

        const NJson::TJsonValue::TArray* arrPtr;
        ReqCheckCondition(requestData.GetArrayPointer(&arrPtr), ConfigHttpStatus.SyntaxErrorStatus, "user_error_incorrect_request_data");

        TVector<TRoleSnapshot> snapshots;
        for (auto&& i : *arrPtr) {
            TRoleSnapshot snapshot;
            ReqCheckCondition(snapshot.DeserializeFromJson(i), ConfigHttpStatus.SyntaxErrorStatus, "user_error_incorrect_request_data_elements");
            snapshots.emplace_back(std::move(snapshot));
        }

        const TString& comment = Context->GetCgiParameters().Get("comment");
        R_ENSURE(comment, ConfigHttpStatus.SyntaxErrorStatus, "comment must being", EDriveSessionResult::IncorrectRequest);

        auto session = BuildTx<NSQL::Writable>();
        for (auto&& i : snapshots) {
            TObjectProposition<TRoleSnapshot> proposition(i, 1);
            proposition.SetDescription(comment);
            if (DriveApi->GetRolesManager()->GetSnapshotPropositions().Propose(proposition, permissions->GetUserId(), session) == EPropositionAcceptance::Problems) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus, Server->GetLocalization());
        }
        g.SetCode(HTTP_OK);
    }
};

class TRejectRoleSnapshotProcessor: public TAppCommonProcessor<TRejectRoleSnapshotProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRejectRoleSnapshotProcessor, TEmptyConfig>;

public:
    TRejectRoleSnapshotProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "manage_roles/snapshot/reject";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Reject, TAdministrativeAction::EEntity::Role);
        const TVector<TString> propositionIds = GetStrings(requestData, "proposition_ids");

        auto session = BuildTx<NSQL::Writable>();

        auto optionalSessions = DriveApi->GetRolesManager()->GetSnapshotPropositions().Get(propositionIds, session);
        R_ENSURE(optionalSessions, {}, "cannot get propositions", session);
        auto sessions = *optionalSessions;

        for (auto&& i : propositionIds) {
            auto it = sessions.find(i);
            R_ENSURE(it != sessions.end(), ConfigHttpStatus.UserErrorState, "incorrect propose_id " + i, EDriveSessionResult::IncorrectRequest);
        }

        for (auto&& i : propositionIds) {
            const EPropositionAcceptance confirmResult = DriveApi->GetRolesManager()->GetSnapshotPropositions().Reject(i, permissions->GetUserId(), session);
            if (confirmResult != EPropositionAcceptance::Rejected) {
                session.DoExceptionOnFail(ConfigHttpStatus);
            }
        }

        if (!session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        g.SetCode(HTTP_OK);
    }
};

class TConfirmRoleSnapshotProcessor: public TAppCommonProcessor<TConfirmRoleSnapshotProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TConfirmRoleSnapshotProcessor, TProcessorWithNotifierConfig>;

public:
    TConfirmRoleSnapshotProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "manage_roles/snapshot/confirm";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TRemoveRolesProcessor: public TAppCommonProcessor<TRemoveRolesProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TRemoveRolesProcessor, TProcessorWithNotifierConfig>;

public:
    TRemoveRolesProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "manage_roles/remove";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) override;
};

class TAddActionProcessor: public TAppCommonProcessor<TAddActionProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TAddActionProcessor, TProcessorWithNotifierConfig>;

public:
    TAddActionProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "manage_action/add";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TActionsHistoryProcessor: public TAppCommonProcessor<TActionsHistoryProcessor> {
private:
    using TBase = TAppCommonProcessor<TActionsHistoryProcessor>;

public:
    TActionsHistoryProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "manage_action/history";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TProposeActionProcessor: public TAppCommonProcessor<TProposeActionProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TProposeActionProcessor, TProcessorWithNotifierConfig>;

public:
    TProposeActionProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "manage_action/propose";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TConfirmActionProcessor: public TAppCommonProcessor<TConfirmActionProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TConfirmActionProcessor, TProcessorWithNotifierConfig>;

public:
    TConfirmActionProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "manage_action/confirm";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TRejectActionProcessor: public TAppCommonProcessor<TRejectActionProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TRejectActionProcessor, TProcessorWithNotifierConfig>;

public:
    TRejectActionProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "manage_action/reject";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TListActionsProcessor: public TAppCommonProcessor<TListActionsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TListActionsProcessor, TEmptyConfig>;

public:
    TListActionsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "manage_action/list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) override;
};

class TRemoveActionsProcessor: public TAppCommonProcessor<TRemoveActionsProcessor, TProcessorWithNotifierConfig> {
private:
    using TBase = TAppCommonProcessor<TRemoveActionsProcessor, TProcessorWithNotifierConfig>;

public:
    TRemoveActionsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "manage_action/remove";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) override;
};

class TAddLoginProcessor: public TAppCommonProcessor<TAddLoginProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TAddLoginProcessor, TEmptyConfig>;

public:
    TAddLoginProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "manage_login/add";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Login);

        NStorage::TTableRecord row;
        Y_ENSURE_EX(row.DeserializeFromJson(requestData), TCodedException(ConfigHttpStatus.UserErrorState) << "incorrect input post data");

        TUserLogin login;
        Y_ENSURE_EX(login.DeserializeFromTableRecord(row), TCodedException(ConfigHttpStatus.UserErrorState) << "incorrect input post data for login construction");

        auto session = BuildTx<NSQL::Writable>();
        if (!DriveApi->GetLoginsManager()->AddLogin(login, session) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        g.SetCode(HTTP_OK);
    }
};

class TListLoginsProcessor: public TAppCommonProcessor<TListLoginsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TListLoginsProcessor, TEmptyConfig>;

public:
    TListLoginsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "manage_login/list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requesData*/) {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::Login);
        const TCgiParameters& cgi = Context->GetCgiParameters();

        const TString userIdRequest = cgi.Has("user_id") ? cgi.Get("user_id") : permissions->GetUserId();
        Y_ENSURE_EX(!!userIdRequest && !GetUuid(userIdRequest).IsEmpty(),
            TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "incorrect user identification info");

        auto session = BuildTx<NSQL::ReadOnly>();
        TVector<TUserLogin> logins;
        if (!DriveApi->GetLoginsManager()->GetLogins(userIdRequest, logins, session)) {
            g.MutableReport().AddReportElement("error", session.GetTransaction()->GetErrors().GetStringReport());
            g.SetCode(ConfigHttpStatus.UnknownErrorStatus);
            return;
        }
        {
            NJson::TJsonValue result;
            for (auto&& i : logins) {
                result.AppendValue(i.BuildJsonReport());
            }
            g.MutableReport().AddReportElement("logins", std::move(result));
            g.SetCode(HTTP_OK);
        }
    }
};

class TRemoveLoginsProcessor: public TAppCommonProcessor<TRemoveLoginsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRemoveLoginsProcessor, TEmptyConfig>;

public:
    TRemoveLoginsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "manage_login/remove";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override {
        ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::Login);
        NStorage::TTableRecord row;
        Y_ENSURE_EX(row.DeserializeFromJson(requestData), TCodedException(ConfigHttpStatus.UserErrorState) << "incorrect input post data");

        TUserLogin login;
        Y_ENSURE_EX(login.DeserializeFromTableRecord(row), TCodedException(ConfigHttpStatus.UserErrorState) << "incorrect input post data for login remove");

        const TCgiParameters& cgi = Context->GetCgiParameters();

        const TString userIdRequest = cgi.Has("user_id") ? cgi.Get("user_id") : permissions->GetUserId();
        Y_ENSURE_EX(!!userIdRequest && !GetUuid(userIdRequest).IsEmpty(),
            TCodedException(ConfigHttpStatus.SyntaxErrorStatus) << "incorrect user identification info");

        auto session = BuildTx<NSQL::Writable>();
        if (!DriveApi->GetLoginsManager()->RemoveLogin(login, session)) {
            g.MutableReport().AddReportElement("error", "Cannot remove login");
            g.MutableReport().AddReportElement("error_details", session.GetTransaction()->GetErrors().GetStringReport());
            g.SetCode(ConfigHttpStatus.UserErrorState);
        } else if (!session.Commit()) {
            g.MutableReport().AddReportElement("error", "Cannot remove login from db database");
            g.MutableReport().AddReportElement("error_details", session.GetTransaction()->GetErrors().GetStringReport());
            g.SetCode(ConfigHttpStatus.UserErrorState);
        } else {
            g.SetCode(HTTP_OK);
        }
    }
};

class TListConstantsProcessor: public TAppCommonProcessor<TListConstantsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TListConstantsProcessor, TEmptyConfig>;
    using TConstantTraits = ui32;

public:
    enum EConstantTraits: ui64 {
        Actions = 1/* "actions" */,
        Tags = 1 << 1/* "tags" */,
        TagDescriptions = 1 << 2 /* "tag_descriptions" */,
        RTBackground = 1 << 3 /* "rt_background" */,
        Notifiers = 1 << 4 /* "notifiers" */,
        Documents = 1 << 5 /* "documents" */,
        Accounts = 1 << 6 /* "accounts" */,
        ActionTypes = 1 << 7 /* "action_types" */,
        AreaScheme = 1 << 8 /* "area_scheme" */,
        TagActions = 1 << 9 /* "tag_actions" */,
        TagFilters = 1 << 10 /* "tag_filters" */,
        TagTypes = 1 << 11 /* "tag_types" */,
        TelematicsCommands = 1 << 12 /* "telematics_commands" */,
        UserTags = 1 << 13 /* "user_tags" */,
        DeviceTags = 1 << 14 /* "device_tags" */,
    };

    TListConstantsProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "constants/list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr /*permissions*/, const NJson::TJsonValue& /*requesData*/);
};

class TViewPermissionsProcessor : public TAppCommonProcessor<TViewPermissionsProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TViewPermissionsProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_permissions";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/);
};

class TRootifyProcessor: public TAppCommonProcessor<TRootifyProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TRootifyProcessor, TEmptyConfig>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_rootify";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/);
};
