#include "processor.h"

#include "callback_persdata.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/chat_robots/registration/bot.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/data/event_tag.h>
#include <drive/backend/data/user_tags.h>

#include <drive/backend/registrar/manager.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/roles/roles.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/tags/tags_search.h>

#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/user_events_api/client.h>
#include <drive/library/cpp/image_transformation/transformation.h>

#include <rtline/util/algorithm/type_traits.h>

#include <util/generic/serialized_enum.h>

void IUserBasicDeletionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TString& userId = permissions->GetUserId();
    TMaybe<TUsersDB::TUserDeleteStatus> deleteInfo;
    {
        TVector<TString> deletionNames = SplitString(GetHandlerSettingDef<TString>("deletion.request_to_delete_tag_names", "blocked_to_delete,ban_to_check"), ",");
        auto tx = BuildTx<NSQL::ReadOnly>();
        deleteInfo = Server->GetDriveAPI()->GetUsersData()->CheckStatusBeforeDelete(userId, *Server, tx, deletionNames, {});
        R_ENSURE(deleteInfo, ConfigHttpStatus.UnknownErrorStatus, "fail to check user status", tx);
    }
    if (GetNewUserPolicy() != "fallback" || GetFallbackUserId() != userId) {
        AddDeletionAttempt(userId);
    }
    if (auto tagName = GetTagToAdd(*deleteInfo, g)) {
        auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tagName);
        auto tx = BuildTx<NSQL::Writable>();
        R_ENSURE(Server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userId, userId, Server, tx) && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "fail to add tag", tx);
        g.AddReportElement("status", "ok");
        g.SetCode(HTTP_OK);
    }
}

void IUserBasicDeletionProcessor::AddDeletionAttempt(const TString& userId) {
    TVector<TDBTag> tags;
    TString tagName = GetHandlerSettingDef<TString>("deletion.deletion_watcher_tag_name", "");
    if (!tagName) {
        return;
    }
    const auto& userTagManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    auto tx = BuildTx<NSQL::Writable>();
    R_ENSURE(userTagManager.RestoreTags({ userId }, { tagName }, tags, tx), ConfigHttpStatus.UnknownErrorStatus, "fail to fetch tag");
    if (tags.empty()) {
        auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tagName);
        auto eventTag = std::dynamic_pointer_cast<TUserEventTag>(tag);
        R_ENSURE(eventTag, ConfigHttpStatus.UnknownErrorStatus, "takeout event tag has wrong type", tx);
        eventTag->AddEvent(GetType(), Now());
        R_ENSURE(userTagManager.AddTag(tag, userId, userId, Server, tx) && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "fail to add tag", tx);
    } else {
        auto tag = tags.front();
        auto tagData = tag.GetData();
        auto eventTag = std::dynamic_pointer_cast<TUserEventTag>(tagData);
        R_ENSURE(eventTag, ConfigHttpStatus.UnknownErrorStatus, "takeout event tag has wrong type", tx);
        eventTag->AddEvent(GetType(), Now());
        R_ENSURE(userTagManager.UpdateTagData(tag, tagData, userId, Server, tx) && tx.Commit(), ConfigHttpStatus.UnknownErrorStatus, "fail to update tag", tx);
    }
}

TString TUserDeletionPassportProcessor::GetTagToAdd(TUsersDB::TUserDeleteStatus userStatus, TJsonReport::TGuard& g) {
    switch (userStatus) {
        case TUsersDB::TUserDeleteStatus::HasDebt:
        case TUsersDB::TUserDeleteStatus::HasActiveSession:
        case TUsersDB::TUserDeleteStatus::DeleteInProgress:
        case TUsersDB::TUserDeleteStatus::Unknown:
        {
            NJson::TJsonValue report = NJson::JSON_ARRAY;
            NJson::TJsonValue& response = report.AppendValue(NJson::JSON_MAP);
            response["code"] = ToString(userStatus);
            response["message"] = "deletion is prohibited";
            g.AddReportElement("status", "error");
            g.AddReportElement("errors", std::move(report));
            g.SetCode(HTTP_OK);
            return {};
        }
        case TUsersDB::TUserDeleteStatus::Empty:
            g.AddReportElement("status", "ok");
            g.SetCode(HTTP_OK);
            return {};
        case TUsersDB::TUserDeleteStatus::ReadyToDelete:
            return GetHandlerSettingDef<TString>("deletion.ready_to_delete_tag_name", "blocked_to_delete");
        case TUsersDB::TUserDeleteStatus::NeedsHumanVerification:
            return GetHandlerSettingDef<TString>("deletion.request_to_delete_tag_name", "ban_to_check");
    }
}

TString TUserDeletionClientProcessor::GetTagToAdd(TUsersDB::TUserDeleteStatus userStatus, TJsonReport::TGuard& g) {
    switch (userStatus) {
        case TUsersDB::TUserDeleteStatus::ReadyToDelete:
            return GetHandlerSettingDef<TString>("deletion.ready_to_delete_tag_name", "blocked_to_delete");
        case TUsersDB::TUserDeleteStatus::NeedsHumanVerification:
            return GetHandlerSettingDef<TString>("deletion.request_to_delete_tag_name", "ban_to_check");
        default:
            R_ENSURE(userStatus == TUsersDB::TUserDeleteStatus::Empty, ConfigHttpStatus.UserErrorState, ToString(userStatus));
            g.SetCode(HTTP_OK);
            return {};
    }
}

void TUserDeleteStatusProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    TString userId = permissions->GetUserId();
    auto tx = BuildTx<NSQL::Writable>();

    TVector<TString> deletionNames{"blocked_to_delete"};
    deletionNames.push_back(GetHandlerSetting<TString>("deletion.request_to_delete_tag_name").GetOrElse("ban_to_check"));

    TMaybe<TUsersDB::TUserDeleteStatus> deleteInfo = Server->GetDriveAPI()->GetUsersData()->CheckStatusBeforeDelete(userId, *Server, tx, deletionNames, {});
    R_ENSURE(deleteInfo, ConfigHttpStatus.UnknownErrorStatus, "fail to fetch status", tx);


    TMap<TUsersDB::TUserDeleteStatus, TString> passportNames{
        {TUsersDB::TUserDeleteStatus::Empty, "empty"},
        {TUsersDB::TUserDeleteStatus::DeleteInProgress, "delete_in_progress"},
        {TUsersDB::TUserDeleteStatus::NeedsHumanVerification, "ready_to_delete"},
        {TUsersDB::TUserDeleteStatus::ReadyToDelete, "ready_to_delete"}
    };

    NJson::TJsonValue report = NJson::JSON_ARRAY;
    NJson::TJsonValue& response = report.AppendValue(NJson::JSON_MAP);
    response["id"] = "drive_user_data";
    response["slug"] = "all";

    switch (auto deleteStatus = *deleteInfo) {
        case TUsersDB::TUserDeleteStatus::HasDebt:
        case TUsersDB::TUserDeleteStatus::HasActiveSession:
        case TUsersDB::TUserDeleteStatus::Unknown:
            response["state"] = "ready_to_delete";
            break;
        case TUsersDB::TUserDeleteStatus::NeedsHumanVerification:
        case TUsersDB::TUserDeleteStatus::DeleteInProgress:
        case TUsersDB::TUserDeleteStatus::ReadyToDelete:
        case TUsersDB::TUserDeleteStatus::Empty:
        {
            auto historyQueryOptions = TTagEventsManager::TQueryOptions();
            historyQueryOptions.SetOrderBy({"history_event_id"}).SetDescending(true);
            historyQueryOptions.SetLimit(1);

            TSet<TString> tagNames(deletionNames.begin(), deletionNames.end());
            tagNames.insert("user_deleted_finally");
            historyQueryOptions.SetGenericCondition("tag", tagNames);

            auto tagHistory = Server->GetDriveAPI()->GetTagsManager().GetUserTags().GetHistoryManager().GetEventsByObject(userId, tx, 0, TInstant::Zero(), historyQueryOptions);
            R_ENSURE(tagHistory, ConfigHttpStatus.UnknownErrorStatus, "fail to fetch history", tx);
            if (!tagHistory->empty()) {
                response["update_date"] = static_cast<TInstant>(tagHistory->front().GetHistoryTimestamp()).ToStringUpToSeconds();
            }

            auto statusPtr = passportNames.FindPtr(deleteStatus);
            R_ENSURE(statusPtr, ConfigHttpStatus.UnknownErrorStatus, "unhandled passport status");
            response["state"] = *statusPtr;
        }
    }
    g.AddEvent(NJson::TMapBuilder
        ("user_delete_status", ToString(*deleteInfo))
    );

    g.AddReportElement("status", "ok");
    g.AddReportElement("data", std::move(report));
    g.SetCode(HTTP_OK);
}

void TUserDupsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& userId = GetString(cgi, "user_id", true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);
    CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess, true);

    TVector<TString> reportTagNames = GetValues<TString>(cgi, "report_tags", false);
    TVector<TString> duplicateTagNames = GetValues<TString>(cgi, "duplicate_tags", false);

    const auto& billingManager = DriveApi->GetBillingManager();
    auto session = BuildTx<NSQL::ReadOnly>();
    auto userDb = Yensured(DriveApi->GetUsersData());
    auto user = userDb->RestoreUser(userId, session);
    R_ENSURE(user, ConfigHttpStatus.EmptySetStatus, "cannot restore user " << userId, session);

    auto userRegistrationManager = Server->GetUserRegistrationManager();
    auto emptyHash = userRegistrationManager ? userRegistrationManager->GetDocumentNumberHash(TString{}) : TString{};
    g.AddEvent(NJson::TMapBuilder
        ("event", "CalcEmptyHash")
        ("hash", emptyHash)
    );

    TSet<TString> duplicateUserIds = { userId };
    const auto& passportHash = user->GetPassportNumberHash();
    if (passportHash && passportHash != emptyHash) {
        auto eg = g.BuildEventGuard("GetPassportHashDups");
        auto optionalPassportHashDups = userDb->GetDocumentOwnersByHash(passportHash, session);
        R_ENSURE(optionalPassportHashDups, {}, "cannot GetDocumentOwnersByHash " << passportHash, session);

        const auto& passportHashDups = *optionalPassportHashDups;
        duplicateUserIds.insert(passportHashDups.begin(), passportHashDups.end());
        g.AddEvent(NJson::TMapBuilder
            ("event", "PassportHashDups")
            ("hash", passportHash)
            ("user_ids", NJson::ToJson(passportHashDups))
        );
    } else {
        g.AddEvent(NJson::TMapBuilder
            ("event", "SkipPassportHashDups")
            ("hash", passportHash)
        );
    }

    const auto& driversLicenseHash = user->GetDrivingLicenseNumberHash();
    if (driversLicenseHash && driversLicenseHash != emptyHash) {
        auto eg = g.BuildEventGuard("GetDriversLicenseHashDups");
        auto optionalDriversLicenseHashDups = userDb->GetDocumentOwnersByHash(driversLicenseHash, session);
        R_ENSURE(optionalDriversLicenseHashDups, {}, "cannot GetDocumentOwnersByHash " << driversLicenseHash, session);

        const auto& driversLicenseHashDups = *optionalDriversLicenseHashDups;
        duplicateUserIds.insert(driversLicenseHashDups.begin(), driversLicenseHashDups.end());
        g.AddEvent(NJson::TMapBuilder
            ("event", "DriversLicenseHashDups")
            ("hash", driversLicenseHash)
            ("user_ids", NJson::ToJson(driversLicenseHashDups))
        );
    } else {
        g.AddEvent(NJson::TMapBuilder
            ("event", "SkipDriversLicenseHashDups")
            ("hash", driversLicenseHash)
        );
    }
    const bool verifiedOnly = GetValue<bool>(cgi, "verified_only", false).GetOrElse(false);
    TSet<TString> deviceIdDups;
    if (const auto devicesManager = Server->GetUserDevicesManager()) {
        TSet<TShortUserDevice> devices;
        if (devicesManager->GetUserDevices(duplicateUserIds, devices, session)) {
            TSet<TString> deviceIds;
            for (auto&& device : devices) {
                deviceIds.emplace(device.GetDeviceId());
            }
            if (!deviceIds.empty() && devicesManager->GetUsersByDeviceIds(deviceIds, deviceIdDups, session, verifiedOnly)) {
                duplicateUserIds.insert(deviceIdDups.begin(), deviceIdDups.end());
                g.AddEvent(NJson::TMapBuilder
                    ("event", "DeviceIdDups")
                    ("user_ids", NJson::ToJson(deviceIdDups))
                );
            }
        }
    }

    bool hasNamesHashData = user->HasPassportNamesHashData();
    if (const auto& namesHash = user->GetPassportNamesHash(); namesHash && hasNamesHashData) {
        auto eg = g.BuildEventGuard("GetNamesHashDups");
        auto optionalNamesHashDups = userDb->GetDocumentOwnersByHash(namesHash, session);
        R_ENSURE(optionalNamesHashDups, {}, "cannot GetDocumentOwnersByHash " << namesHash, session);

        const auto& namesHashDups = *optionalNamesHashDups;
        duplicateUserIds.insert(namesHashDups.begin(), namesHashDups.end());
        g.AddEvent(NJson::TMapBuilder
            ("event", "NamesHashDups")
            ("hash", namesHash)
            ("user_ids", NJson::ToJson(namesHashDups))
        );
    }

    TSet<TString> duplicateTagsDups;
    if (!duplicateTagNames.empty()) {
        auto duplicateConnectionTags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(duplicateUserIds, duplicateTagNames, session);
        if (!duplicateConnectionTags) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        for (auto&& tag : *duplicateConnectionTags) {
            auto tagImpl = tag.GetTagAs<TConnectionUserTag>();
            if (tagImpl && tagImpl->GetConnectedUserId()) {
                duplicateUserIds.insert(tagImpl->GetConnectedUserId());
                duplicateTagsDups.insert(tagImpl->GetConnectedUserId());
            }
        }
        if (!duplicateTagsDups.empty()) {
            g.AddEvent(NJson::TMapBuilder
                ("event", "DuplicateTagsDups")
                ("user_ids", NJson::ToJson(duplicateTagsDups))
            );
        }
    }

    TMap<TString, TSet<TString>> tagsOnUsers;
    if (!reportTagNames.empty()) {
        auto tagsToReport = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(duplicateUserIds, reportTagNames, session);
        if (!tagsToReport) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
        for (auto&& tag : *tagsToReport) {
            tagsOnUsers[tag.GetObjectId()].insert(tag->GetName());
        }
    }

    auto dups = userDb->FetchInfo(duplicateUserIds, session);
    NJson::TJsonValue users = NJson::JSON_NULL;
    for (auto&& [dupId, info] : dups) {
        if (dupId == userId) {
            continue;
        }
        bool isNamesDup = info.GetPassportNamesHash() && info.GetPassportNamesHash() == user->GetPassportNamesHash();
        bool isPassportDup = info.GetPassportNumberHash() && info.GetPassportNumberHash() == user->GetPassportNumberHash();
        bool isDrivingLicenseDup = info.GetDrivingLicenseNumberHash() && info.GetDrivingLicenseNumberHash() == user->GetDrivingLicenseNumberHash();
        bool isDeviceIdDup = deviceIdDups.contains(dupId);
        bool isDuplicateTagDup = duplicateTagsDups.contains(dupId);
        if (!isDrivingLicenseDup && !isPassportDup && !isDeviceIdDup && !isDuplicateTagDup && isNamesDup) {
            if (info.GetFirstName() != user->GetFirstName()) {
                continue;
            }
            if (info.GetLastName() != user->GetLastName()) {
                continue;
            }
            if (info.GetPName() != user->GetPName()) {
                continue;
            }
        }
        NJson::TJsonValue userReport = info.GetReport(NUserReport::ReportAll);
        auto debt = billingManager.GetDebt(dupId, session);
        R_ENSURE(debt, {}, "cannot GetDebt for " << dupId, session);
        userReport["debt"] = *debt;
        userReport["is_names_dup"] = isNamesDup;
        userReport["is_passport_dup"] = isPassportDup;
        userReport["is_driving_license_dup"] = isDrivingLicenseDup;
        userReport["is_device_id_dup"] = isDeviceIdDup;
        userReport["is_duplicate_tag_dup"] = isDuplicateTagDup;
        if (!reportTagNames.empty()) {
            userReport["tags"] = NJson::ToJson(tagsOnUsers[dupId]);
        }
        users.AppendValue(std::move(userReport));
    }
    g.AddReportElement("users", std::move(users));
    g.SetCode(HTTP_OK);
}

void TUserInfoProcessorBase::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    const auto& cgi = Context->GetCgiParameters();
    auto uid = GetString(cgi, "uid", false);
    auto userId = GetUUID(requestData, "user_id", false);
    if (!userId) {
        userId = GetUUID(cgi, "user_id", true);
    }
    R_ENSURE(userId || uid, HTTP_BAD_REQUEST, "either user_id or uid should be present");

    if (!userId) {
        userId = DriveApi->GetUsersData()->GetUserIdByUid(uid);
    }
    R_ENSURE(userId, HTTP_NOT_FOUND, "cannot find user_id for uid " << uid);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);
    CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess, true);

    if (!CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess)) {
        TDriveUserData user;
        user.SetStatus(NDrive::UserStatusDeleted);
        g.MutableReport().SetExternalReport(user.GetReport(permissions->GetUserReportTraits()));
        g.SetCode(HTTP_OK);
        return;
    }

    auto traits = permissions->GetUserReportTraits();
    auto antitraits = GetHandlerSetting<NUserReport::TReportTraits>("antitraits").GetOrElse(NUserReport::ReportNone);
    traits &= (~antitraits);

    auto user = Server->GetDriveAPI()->GetUsersData()->GetCachedObject(userId);
    R_ENSURE(user, ConfigHttpStatus.EmptySetStatus, "cannot find user " << userId);

    auto datasyncClient = Server->GetDatasyncClient();
    auto enableFavourites = GetHandlerSetting<bool>("favourites_enabled").GetOrElse(true);
    auto favouritesCollection = GetHandlerSetting<TString>("favourites_datasync_collection").GetOrElse("v2/personality/profile/addresses");
    auto favourites = datasyncClient && enableFavourites && favouritesCollection
        ? datasyncClient->Get(favouritesCollection, {}, user->GetPassportUid())
        : NThreading::Uninitialized;

    auto userReport = user->GetFullReport(traits, *Server);

    NThreading::TFuture<void> waiter;
    if (favourites.Initialized()) {
        waiter = NThreading::WaitAll(userReport.IgnoreResult(), favourites.IgnoreResult());
    } else {
        waiter = userReport.IgnoreResult();
    }

    auto report = g.GetReport();
    bool shouldUpdateLogin = GetHandlerSettingDef<bool>("update_login", false);
    TString loginUpdateUser = GetHandlerSettingDef<TString>("update_login_user", "robot-frontend");
    auto driveApi = Server->GetDriveAPI();
    waiter.Subscribe([report, permissions, favourites, userReport, user, shouldUpdateLogin, loginUpdateUser, driveApi](const NThreading::TFuture<void>& /*waiter*/) mutable {
        NDrive::TEventLog::TUserIdGuard userIdGuard(permissions ? permissions->GetUserId() : Default<TString>());
        TJsonReport::TGuard g(report, HTTP_OK);
        NJson::TJsonValue result = userReport.ExtractValue();
        if (favourites.Initialized()) {
            if (favourites.HasValue()) {
                result["favourites"] = favourites.ExtractValue().ExtractValue();
            } else {
                result["favourites"] = NThreading::GetExceptionInfo(favourites);
            }
        }
        TString blackboxLogin;
        if (shouldUpdateLogin && !user->IsStaffAccount() && result["blackbox"].IsMap() && NJson::ParseField(result["blackbox"], "login", blackboxLogin, false)) {
            if (!blackboxLogin.empty() && blackboxLogin != user->GetLogin()) {
                auto session = driveApi->GetTagsManager().GetUserTags().BuildSession(false);
                session.SetOriginatorId(permissions->GetUserId());
                user->SetLogin(blackboxLogin);
                result["username"] = blackboxLogin;
                if (!driveApi->GetUsersData()->UpdateUser(*user, loginUpdateUser, session) || ! session.Commit()) {
                    g.AddEvent(NJson::TMapBuilder
                        ("event", "UserLoginUpdateError")
                        ("error", session.GetReport())
                        ("user_id", user->GetUserId())
                    );
                }
            }
        }
        g.SetExternalReport(std::move(result));
    });
    g.Release();
}

void TUserInfoProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TBase::ProcessServiceRequest(g, permissions, requestData);
}

void TBaseUserInfoEditingProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    TString userId = GetString(requestData, "id");
    bool validate = GetValue<bool>(Context->GetCgiParameters(), "validate", false).GetOrElse(true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::User, userId);
    CheckEditingRights(userId, permissions);

    TDriveUserData userData;
    bool isNewUser = false;

    NDrive::TEntitySession session = BuildTx<NSQL::ReadOnly>();
    {
        auto fetchResult = Server->GetDriveAPI()->GetUsersData()->FetchInfo(userId, session);
        if (fetchResult.empty()) {
            isNewUser = true;
        } else {
            userData = std::move(fetchResult.MutableResult().begin()->second);
        }
    }
    R_ENSURE(
        userData.Patch(requestData, isNewUser, permissions->GetUserReportTraits()),
        ConfigHttpStatus.SyntaxErrorStatus,
        "malformed basic user info",
        session
    );
    R_ENSURE(userData.GetUserId() || userData.GetUid(), HTTP_BAD_REQUEST, "empty UserIdInfo", session);

    TMaybe<TUserPassportData> passportNew;
    TMaybe<TUserDrivingLicenseData> drivingLicenseNew;
    TString newDocsRevision = "update_" + ToString(TInstant::Now().MicroSeconds());

    if (requestData.Has("passport") && (permissions->GetUserReportTraits() & NUserReport::ReportPassport)) {
        passportNew.ConstructInPlace();
        R_ENSURE(passportNew->Parse(requestData["passport"]), ConfigHttpStatus.SyntaxErrorStatus, "malformed passport data", session);
        if (validate) {
            R_ENSURE(passportNew->IsDatasyncCompatible(), ConfigHttpStatus.UserErrorState, "data is not datasync compatible", session);
        }
        userData.SetPassportDatasyncRevision(newDocsRevision);
        if (Server->GetUserRegistrationManager() && passportNew->GetNumber()) {
            userData.SetPassportNumberHash(Server->GetUserRegistrationManager()->GetDocumentNumberHash(passportNew->GetNumber()));
        }
    }
    if (requestData.Has("driving_license") && (permissions->GetUserReportTraits() & NUserReport::ReportDrivingLicense)) {
        drivingLicenseNew.ConstructInPlace();
        R_ENSURE(drivingLicenseNew->Parse(requestData["driving_license"]), ConfigHttpStatus.SyntaxErrorStatus, "malformed driving license data", session);
        if (validate) {
            R_ENSURE(drivingLicenseNew->IsDatasyncCompatible(), ConfigHttpStatus.UserErrorState, "data is not datasync compatible", session);
        }
        userData.SetDrivingLicenseDatasyncRevision(newDocsRevision);
        if (Server->GetUserRegistrationManager() && drivingLicenseNew->GetNumber()) {
            userData.SetDrivingLicenseNumberHash(Server->GetUserRegistrationManager()->GetDocumentNumberHash(drivingLicenseNew->GetNumber()));
        }
    }

    auto processorReport = g.GetReport();
    auto updateUser = [processorReport, userData, isNewUser, operatorUserId = permissions->GetUserId(), server = Server] () {
        TJsonReport::TGuard g(processorReport, HTTP_INTERNAL_SERVER_ERROR);
        auto tx = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        auto updatedUser = server->GetDriveAPI()->GetUsersData()->UpdateUser(userData, operatorUserId, tx, isNewUser);
        R_ENSURE(updatedUser, {}, "cannot UpdateUser", tx);
        R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
        g.MutableReport().AddReportElement("user", userData.SerializeToTableRecord().SerializeToJson());
        g.SetCode(HTTP_OK);
    };

    TVector<NThreading::TFuture<void>> updateFutures;
    if (passportNew) {
        updateFutures.emplace_back(DriveApi->GetPrivateDataClient().UpdatePassport(userData, newDocsRevision, *passportNew));
    }
    if (drivingLicenseNew) {
        updateFutures.emplace_back(DriveApi->GetPrivateDataClient().UpdateDrivingLicense(userData, newDocsRevision, *drivingLicenseNew));
    }
    g.Release();
    NThreading::WaitExceptionOrAll(updateFutures).Subscribe([processorReport, updateUser, eventLogState = NDrive::TEventLog::CaptureState()](const NThreading::TFuture<void>& waiter) {
        NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
        if (!waiter.HasValue()) {
            TJsonReport::TGuard g(processorReport, HTTP_INTERNAL_SERVER_ERROR);
            g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                .SetErrorCode("cannot_update_datasync")
                << NThreading::GetExceptionMessage(waiter)
            );
            return;
        }
        updateUser();
    });
}

void TUserInfoEditingProcessor::DoCheckEditPermissions(const TString& userId, TUserPermissions::TPtr permissions) const {
    auto forceShowObjectsWithoutTags = GetHandlerSetting<bool>("force_show_objects_without_tags").GetOrElse(false);
    CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess | NAccessVerification::EAccessVerificationTraits::ModifyAccess, true, forceShowObjectsWithoutTags);
}

void TUserModificationsHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TString userId = GetString(Context->GetCgiParameters(), "user_id");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);

    auto session = BuildTx<NSQL::ReadOnly>();
    auto fetchResult = Server->GetDriveAPI()->GetUsersData()->FetchInfo(userId, session).GetResult();
    ReqCheckCondition(fetchResult.find(userId) != fetchResult.end(), ConfigHttpStatus.EmptySetStatus, EDriveLocalizationCodes::UserNotFound);
    auto user = fetchResult.find(userId)->second;

    TDriveUserDataHistoryManager::TEvents history;
    if (CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess)) {
        auto optionalEvents = DriveApi->GetUsersData()->GetHistoryManager().GetEvents({}, session, NSQL::TQueryOptions()
            .AddGenericCondition("id", userId)
        );
        R_ENSURE(optionalEvents, {}, "cannot GetEvents", session);
        history = std::move(*optionalEvents);
    }

    TSet<TString> passportRevisions;
    TSet<TString> drivingLicenseRevisions;

    NJson::TJsonValue changesHistoryReport = NJson::JSON_ARRAY;
    for (auto&& historyItem : history) {
        auto historyReport = historyItem.BuildReportItem();
        historyReport["object"] = historyItem.GetReport(permissions->GetUserReportTraits());
        changesHistoryReport.AppendValue(std::move(historyReport));

        if (!!historyItem.GetPassportDatasyncRevision() && (permissions->GetUserReportTraits() & NUserReport::ReportPassport)) {
            passportRevisions.insert(historyItem.GetPassportDatasyncRevision());
        }
        if (!!historyItem.GetDrivingLicenseDatasyncRevision() && (permissions->GetUserReportTraits() & NUserReport::ReportDrivingLicense)) {
            drivingLicenseRevisions.insert(historyItem.GetDrivingLicenseDatasyncRevision());
        }
    }

    NJson::TJsonValue report = NJson::JSON_MAP;
    report["history"] = std::move(changesHistoryReport);

    if (passportRevisions.size() + drivingLicenseRevisions.size() > 0) {
        auto callbackTask = MakeAtomicShared<TPrivateDataAcquisionCallback>(g.GetReport(), permissions->GetUserReportTraits());
        callbackTask->SetExpectedResponses((ui16)(passportRevisions.size() + drivingLicenseRevisions.size()));
        callbackTask->SetBaseReport(std::move(report));

        for (auto&& revision : passportRevisions) {
            DriveApi->GetPrivateDataClient().GetPassport(user, revision, callbackTask);
        }
        for (auto&& revision : drivingLicenseRevisions) {
            DriveApi->GetPrivateDataClient().GetDrivingLicense(user, revision, callbackTask);
        }
        g.Release();
    } else {
        g.MutableReport().SetExternalReport(std::move(report));
    }
    g.SetCode(HTTP_OK);
}

void TUserListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto type = GetString(cgi, "type");

    auto session = BuildTx<NSQL::ReadOnly>();

    NJson::TJsonValue users = NJson::JSON_ARRAY;
    if (type == NDrive::UserStatusStaff) {
        const auto& usersTable = *Yensured(DriveApi->GetUsersData());
        auto result = usersTable.FetchWithCustomQuery(usersTable.MakeQuery("uid >= 1120000000000000 AND uid < 1130000000000000"), session);
        for (auto [id, data] : result.GetResult()) {
            users.AppendValue(data.SerializeToTableRecord().SerializeToJson());
        }
    } else {
        R_ENSURE(false, ConfigHttpStatus.UserErrorState, "unknown type: " << type);
    }
    g.MutableReport().AddReportElement("users", std::move(users));
    g.SetCode(HTTP_OK);
}

void TTagsSearchProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto userId = permissions->GetUserId();

    const auto cgi = Context->GetCgiParameters();
    auto hasAllOf = GetStrings(cgi, "has_all_of", false);
    auto hasNoneOf = GetStrings(cgi, "has_none_of", false);
    auto hasOneOf = GetStrings(cgi, "has_one_of", false);
    auto limit = GetValue<ui64>(cgi, "limit", false).GetOrElse(100);
    auto locale = GetLocale();

    R_ENSURE(hasAllOf.size() || hasOneOf.size(), ConfigHttpStatus.SyntaxErrorStatus, "Incorrect search params: either 'has_all_of' or 'has_one_of' should be non empty", EDriveSessionResult::IncorrectRequest);

    auto tx = BuildTx<NSQL::ReadOnly | NSQL::RepeatableRead>();

    auto search = DriveApi->GetTagsSearch(NEntityTagsManager::EEntityType::User);
    R_ENSURE(search, ConfigHttpStatus.ServiceUnavailable, "User search not configured", NDrive::MakeError("component_unavailable"), tx);
    TTagsSearchRequest req(hasAllOf, hasNoneOf, hasOneOf, limit);
    if (cgi.Has("performer_id")) {
        auto performerIdRaw = GetString(cgi, "performer_id");
        if (performerIdRaw == "0") {
            req.SetPerformer("");
        } else {
            req.SetPerformer(performerIdRaw);
        }
    }

    auto result = search->Search(req, tx);

    NJson::TJsonValue usersReport;
    usersReport.InsertValue("entity_type", "user");
    const auto predRemove = [this, &permissions](const TString& userId) {
        return !CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);
    };
    result.MutableMatchedIds().erase(std::remove_if(result.MutableMatchedIds().begin(), result.MutableMatchedIds().end(), predRemove), result.MutableMatchedIds().end());
    usersReport.InsertValue("report", result.BuildJsonReport(DriveApi->GetUsersData()));
    if (result.HasTotalMatched()) {
        usersReport.InsertValue("total", result.GetTotalMatchedRef());
    } else {
        usersReport.InsertValue("total", NJson::JSON_NULL);
    }

    NJson::TJsonValue allResults = NJson::JSON_ARRAY;
    allResults.AppendValue(std::move(usersReport));

    g.MutableReport().AddReportElement("results", std::move(allResults));
    {
        NJson::TJsonValue descriptions = NJson::JSON_ARRAY;
        auto namesForFetch = MakeSet(req.GetObservedTagNames());
        for (auto&& tagName : namesForFetch) {
            auto tagPtr = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName);
            if (!tagPtr) {
                continue;
            }
            descriptions.AppendValue(tagPtr->BuildJsonReport(locale));
        }
        g.MutableReport().AddReportElement("tag_descriptions", std::move(descriptions));
    }

    g.SetCode(HTTP_OK);
}

void TUserDocumentPhotoStatusProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::User);

    auto userId = permissions->GetUserId();
    auto photoId = GetString(Context->GetCgiParameters(), "photo_id");

    R_ENSURE(DriveApi->HasDocumentPhotosManager(), ConfigHttpStatus.UnknownErrorStatus, "user photo manager undefined");

    auto session = BuildTx<NSQL::Writable>();
    auto photoFetchResult = DriveApi->GetDocumentPhotosManager().GetUserPhotosDB().FetchInfo(photoId, session);
    auto p = photoFetchResult.MutableResult().find(photoId);
    R_ENSURE(p != photoFetchResult.MutableResult().end(), ConfigHttpStatus.UnknownErrorStatus, "no such photo");

    auto photo = std::move(p->second);
    photo.SetVerificationStatus(NUserDocument::EVerificationStatus::Ok);

    if (!DriveApi->GetDocumentPhotosManager().GetUserPhotosDB().Upsert(photo, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TUserForceRegistrationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    R_ENSURE(Server->GetUserRegistrationManager(), ConfigHttpStatus.UnknownErrorStatus, "registration manager absent");
    auto userId = GetString(Context->GetCgiParameters(), "user_id");
    CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess, true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Confirm, TAdministrativeAction::EEntity::User, userId);
    auto operatorUserId = permissions->GetUserId();

    TDriveUserData userData;
    {
        const auto& tagsManager = DriveApi->GetTagsManager().GetUserTags();
        auto session = BuildTx<NSQL::Writable>();
        auto userFetchResult = DriveApi->GetUsersData()->FetchInfo(userId, session);
        auto userPtr = userFetchResult.GetResultPtr(userId);
        R_ENSURE(userPtr, ConfigHttpStatus.EmptySetStatus, "no such user");
        userData = *userPtr;

        auto comment = "Ручной впуск";
        auto tag = IJsonSerializableTag::BuildWithComment<TRegistrationUserTag>("user_registered_manually", comment);
        if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session) || !session.Commit()) {
            session.DoExceptionOnFail(ConfigHttpStatus);
        }
    }

    TString chatId;
    TString topic;
    if (!Server->GetUserRegistrationManager()->DeduceChatRobot(userId, chatId, topic)) {
        chatId = Server->GetUserRegistrationManager()->GetConfig().GetChatId();
    }

    auto chatRobotPtr = Server->GetChatRobot(chatId);
    auto chatRobot = dynamic_cast<const TSimpleChatBot*>(chatRobotPtr.Get());
    R_ENSURE(chatRobot, ConfigHttpStatus.UnknownErrorStatus, "chat robot for approval messages not found");
    R_ENSURE(Server->GetUserRegistrationManager()->Approve(userData, operatorUserId, chatRobot, topic), ConfigHttpStatus.UnknownErrorStatus, "can't register user");

    g.SetCode(HTTP_OK);
}

void TBaseUserForceDeletionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto userId = GetString(Context->GetCgiParameters(), "user_id");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Remove, TAdministrativeAction::EEntity::User, userId);
    CheckDeletePermissions(userId, permissions);

    auto operatorUserId = permissions->GetUserId();

    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetDriveAPI()->GetUsersData()->DeleteUser(userId, operatorUserId, Server, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TUserExternalBlacklistAdditionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) {
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::User);

    EBlacklistUserField fieldType;
    EBanPolicy policy;

    auto fieldTypeStr = GetString(requestData, "field_type");
    auto fieldValue = GetString(requestData, "field_value");
    auto policyStr = GetString(requestData, "policy");
    auto comment = GetString(requestData, "comment", false);

    R_ENSURE(TryFromString(fieldTypeStr, fieldType), ConfigHttpStatus.SyntaxErrorStatus, "Can't parse field type", EDriveSessionResult::IncorrectRequest);
    R_ENSURE(TryFromString(policyStr, policy), ConfigHttpStatus.SyntaxErrorStatus, "Can't parse policy", EDriveSessionResult::IncorrectRequest);

    if (fieldType == EBlacklistUserField::DrivingLicenseNumber || fieldType == EBlacklistUserField::PassportNumber) {
        fieldValue = Server->GetUserRegistrationManager()->GetDocumentNumberHash(fieldValue);
    }

    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetUserRegistrationManager()->GetBlacklistExternal()->AddRecord(fieldType, fieldValue, permissions->GetUserId(), policy, comment, Server, session) || !session.Commit()) {
        session.DoExceptionOnFail(ConfigHttpStatus);
    }

    g.SetCode(HTTP_OK);
}

void TUserRequestsHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    const TString& userId = GetString(cgi, "user_id");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);
    NUserReport::TReportTraits traits = permissions->GetUserReportTraits();
    R_ENSURE(traits & NUserReport::ReportRequestsHistory, ConfigHttpStatus.PermissionDeniedStatus, "Missing ReportRequestsHistory access rights", EDriveSessionResult::NoUserPermissions);
    R_ENSURE(Server->GetUserEventsApi(), ConfigHttpStatus.UnknownErrorStatus, "Missing UserEventsClient");

    const TInstant since = GetTimestamp(cgi, "since", TInstant::Zero());
    const TInstant until = GetTimestamp(cgi, "until", TInstant::Max());
    const auto requestsCount = GetValue<ui32>(cgi, "count", false);

    const NDrive::TUserEventsApi* userEventsApi = Server->GetUserEventsApi();
    auto future = userEventsApi->GetUserRequests(userId, since, until, requestsCount.GetOrElse(0));
    auto report = g.GetReport();
    future.Subscribe([report, traits, since, until](const NThreading::TFuture<NDrive::TRequestHistoryReply>& result) {
        TJsonReport::TGuard g(report, HTTP_OK);
        TJsonReport& r = g.MutableReport();
        r.SetExternalReport(result.GetValue().SerializeToJson(
            traits & NUserReport::ReportRealtimeLocation,
            traits & NUserReport::ReportRealtimeClientAppInfo,
            since,
            until
        ));
    });
    g.Release();
}

void TTagOperatorsProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    auto tagName = GetString(Context->GetCgiParameters(), "tag_name");
    auto tagActionStr = GetString(Context->GetCgiParameters(), "action");

    size_t limit = 100;
    auto limitStr = GetString(Context->GetCgiParameters(), "limit", false);
    if (limitStr) {
        R_ENSURE(TryFromString(limitStr, limit), ConfigHttpStatus.SyntaxErrorStatus, "can't parse limit: should be positive integer", EDriveSessionResult::IncorrectRequest);
    }
    TTagAction::ETagAction tagAction;
    R_ENSURE(TryFromString(tagActionStr, tagAction), ConfigHttpStatus.SyntaxErrorStatus, "can't parse action", EDriveSessionResult::IncorrectRequest);

    auto userIds = MakeVector(Server->GetDriveAPI()->GetRolesManager()->GetUsersWithTagAction(tagName, tagAction, Context->GetRequestStartTime()));
    g.MutableReport().AddReportElement("count", userIds.size());

    NJson::TJsonValue reportUserIds = NJson::JSON_ARRAY;
    for (size_t i = 0; i < Min(userIds.size(), limit); ++i) {
        if (CheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userIds[i])) {
            reportUserIds.AppendValue(userIds[i]);
        }
    }
    g.MutableReport().AddReportElement("user_ids", std::move(reportUserIds));

    g.SetCode(HTTP_OK);
}

void TPhoneBindingHistoryProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    TString userId = GetString(cgi, "user_id");
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);

    auto session = BuildTx<NSQL::ReadOnly>();
    TVector<TObjectEvent<TUserDevice>> events;
    R_ENSURE(Server->GetUserDevicesManager(), ConfigHttpStatus.ServiceUnavailable, "devices manager not configured");
    R_ENSURE(Server->GetUserDevicesManager()->GetFullHistoryByUser(userId, events, session), ConfigHttpStatus.UnknownErrorStatus, "could not get history");
    NJson::TJsonValue history = NJson::JSON_ARRAY;
    for (auto&& entry : events) {
        if (!entry.GetPhone()) {
            continue;
        }
        NJson::TJsonValue data;
        data["phone"] = entry.GetPhone();
        data["event"] = ToString(entry.GetHistoryAction());
        data["timestamp"] = entry.GetHistoryInstant().Seconds();
        history.AppendValue(std::move(data));
    }
    g.MutableReport().AddReportElement("history", std::move(history));
    g.SetCode(HTTP_OK);
}

void TPersonalDataRevisionProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    TString uid = GetString(cgi, "uid", false);
    TString userId = GetString(cgi, "user_id", false);
    TString secretId = GetString(cgi, "secret_id", false);
    if (!secretId) {
        secretId = GetString(cgi, "secretId", false);
    }
    R_ENSURE(!userId.empty() ^ !secretId.empty(), ConfigHttpStatus.SyntaxErrorStatus, "exactly ony of {'user_id', 'secret_id'} should be specified");
    TString assignmentId = GetString(cgi, "assignment_id", false);
    if (!assignmentId) {
        assignmentId = GetString(cgi, "assignmentId", false);
    }
    R_ENSURE(assignmentId, ConfigHttpStatus.SyntaxErrorStatus, "assignment id should be specified");

    if (!userId) {
        auto fetchResult = Server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentVerificationAssignments().FetchInfo(secretId);
        R_ENSURE(fetchResult.size() == 1, ConfigHttpStatus.SyntaxErrorStatus, "no assignment with such secret_id");
        userId = fetchResult.begin()->second.GetUserId();
    }

    if (!uid) {
        auto maybeUser = DriveApi->GetUsersData()->GetCachedObject(userId);
        R_ENSURE(maybeUser.Defined(), ConfigHttpStatus.SyntaxErrorStatus, "no such user");
        uid = maybeUser->GetUid();
    }
    R_ENSURE(uid, ConfigHttpStatus.UnknownErrorStatus, "uid is missing");

    TUserIdInfo info;
    info.SetUid(uid);
    info.SetUserId(userId);

    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);
    auto traits = permissions->GetUserReportTraits();
    auto callbackTask = MakeAtomicShared<TPrivateDataAcquisionCallback>(g.GetReport(), traits);
    callbackTask->SetExpectedResponses(!!(traits & NUserReport::ReportPassport) + !!(traits & NUserReport::ReportDrivingLicense));
    if (traits & NUserReport::ReportPassport) {
        DriveApi->GetPrivateDataClient().GetPassport(info, assignmentId, callbackTask);
    }
    if (traits & NUserReport::ReportDrivingLicense) {
        DriveApi->GetPrivateDataClient().GetDrivingLicense(info, assignmentId, callbackTask);
    }
    g.Release();
}

void TUserDocumentDataProcessorBase::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue&  requestData) {
    const auto& cgi = Context->GetCgiParameters();
    auto userId = GetUUID(requestData, "user_id", false);
    if (!userId) {
        userId = GetUUID(cgi, "user_id", true);
    }
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);
    R_ENSURE(CheckUserGeneralAccessRights(userId, permissions, NAccessVerification::EAccessVerificationTraits::TagsAccess), ConfigHttpStatus.PermissionDeniedStatus, "no permissions to observe requested user");

    bool reportPhotos = GetValue<bool>(cgi, "report_photos", false).GetOrElse(false);
    bool backgroundVideo = GetValue<bool>(cgi, "background_video", false).GetOrElse(false);
    if (reportPhotos) {
        auto userPhotos = Server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetAllForUser(userId);
        NJson::TJsonValue photosReport = NJson::JSON_ARRAY;
        for (auto&& photo : userPhotos) {
            photosReport.AppendValue(photo.GetReport());
        }
        g.MutableReport().AddReportElement("photos", std::move(photosReport));
        g.SetCode(HTTP_OK);
        return;
    } else if (backgroundVideo) {
        auto videoId = GetString(requestData, "photo_id", false);
        if (!videoId) {
            videoId = GetString(cgi, "photo_id", true);
        }
        R_ENSURE(
            !videoId.empty(),
            ConfigHttpStatus.SyntaxErrorStatus,
            "photo_id of the background video is not provided"
        );

        auto future = Server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentBackgroundVideo(videoId, *Server);
        future.Subscribe([
            report = g.GetReport(),
            eventLogState = NDrive::TEventLog::CaptureState(),
            errorStatus = ConfigHttpStatus.SyntaxErrorStatus
        ] (const auto& future) {
            auto elsg = NDrive::TEventLog::Guard(eventLogState);
            try {
                if (!report) {
                    throw yexception() << "incorrect report value";
                }
                NS3::TFile video = std::move(future).GetValue();

                TBuffer data;
                data.Assign(video.GetContent().data(), video.GetContent().size());
                report->Finish(HTTP_OK, video.GetContentType(), data);
            } catch (const yexception& e) {
                TJsonReport::TGuard g(report, errorStatus);
                g.MutableReport().AddReportElement("status", "error");
                g.MutableReport().AddReportElement("error", e.what());
            }
        });
        g.Release();
        return;
    } else {
        auto workerId = permissions->GetUserFeatures().GetYangWorkerId()
            ? permissions->GetUserFeatures().GetYangWorkerId()
            : permissions->GetUserId();
        auto uid = GetString(cgi, "uid", false);
        auto photoId = GetString(requestData, "photo_id", false);
        if (!photoId) {
            photoId = GetString(cgi, "photo_id", true);
        }

        bool transform = GetHandlerSettingDef<bool>("transform", false);
        TString transformationConfigRaw = GetHandlerSettingDef<TString>("transformation_config", "");

        R_ENSURE(!photoId.empty(), ConfigHttpStatus.SyntaxErrorStatus, "photo_id is not provided");
        TString photo;
        auto future = Server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentPhoto(photoId, *Server, uid);
        future.Subscribe([
            userId,
            workerId,
            transform,
            transformationConfigRaw,
            report = g.GetReport(),
            eventLogState = NDrive::TEventLog::CaptureState(),
            errorStatus = ConfigHttpStatus.SyntaxErrorStatus
        ] (const auto& future) {
            auto elsg = NDrive::TEventLog::Guard(eventLogState);
            try {
                if (!report) {
                    throw yexception() << "incorrect report value";
                }
                NS3::TFile photo = future.GetValue();
                TString photoBinary = photo.GetContent();

                if (transform) {
                    NJson::TJsonValue transformationConfigJson;
                    NImageTransformation::TTransformationConfig transformationConfig;

                    if (!transformationConfigRaw
                    || !NJson::ReadJsonTree(transformationConfigRaw, &transformationConfigJson)
                    || !transformationConfig.DeserializeFromJson(transformationConfigJson)) {
                        throw yexception() <<  "failed to parse transformation config";
                    }

                    NImageTransformation::TTransformation tr(transformationConfig, workerId);
                    TMessagesCollector errors;
                    if (!tr.Transform(photoBinary, photoBinary, errors)) {
                        throw yexception() << "failed to transform image: " << errors.GetStringReport();
                    }
                }

                TBuffer data;
                data.Assign(photoBinary.data(), photoBinary.size());
                report->Finish(HTTP_OK, photo.GetContentType(), data);
            } catch (const yexception& e) {
                TJsonReport::TGuard g(report, errorStatus);
                g.MutableReport().AddReportElement("status", "error");
                g.MutableReport().AddReportElement("error", e.what());
            }
        });
        g.Release();
        return;
    }
}

void TUserDocumentDataProcessor::ProcessServiceRequest(TJsonReport::TGuard &g, TUserPermissions::TPtr permissions, const NJson::TJsonValue &requestData) {
    TBase::ProcessServiceRequest(g, permissions, requestData);
}

void TUserFeaturesProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    auto userId = GetString(cgi, "user_id", true);
    ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::User, userId);
    const NDrive::TUserEventsApi* client = nullptr;
    if (!Server->GetSettings().GetValue<bool>("use_features_client").GetOrElse(false)) {
        client = Server->GetUserEventsApi();
    } else {
        client = Server->GetFeaturesClient();
    }
    R_ENSURE(client, ConfigHttpStatus.UnknownErrorStatus, "Missing features client");
    auto featuresFuture = client->GetUserFeatures(userId);
    R_ENSURE(featuresFuture.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.UnknownErrorStatus, "Unable to fetch features");
    R_ENSURE(featuresFuture.HasValue(), ConfigHttpStatus.UnknownErrorStatus, "Unable to fetch features");
    auto features = featuresFuture.GetValue();
    g.MutableReport().AddReportElement("features", features.GetReport());
    g.SetCode(HTTP_OK);
}
