#include "processor.h"

#include <drive/backend/data/common/serializable.h>

void TStartDeviceVerificationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_ENSURE_EX(Server->GetUserDevicesManager(), TCodedException(ConfigHttpStatus.ServiceUnavailable) << "incorrect server configuration");
    Y_ENSURE_EX(GetNewDeviceStatus() != ENewDeviceStatus::Verified, TCodedException(ConfigHttpStatus.UserErrorState) << "incorrect device verification state");
    TDeviceIdVerificationContext divc;
    divc.ApplicationId = GetApplicationId();
    divc.ClientIp = GetClientIp();
    divc.EnableGpsPackageName = IRequestProcessor::IsAndroid(GetUserApp());
    TSet<TString> errorCodes;
    auto verificationMethod = GetValue<IUserDevicesManager::EVerificationMethod>(Context->GetCgiParameters(), "verification_method", false).GetOrElse(GetHandlerSettingDef<IUserDevicesManager::EVerificationMethod>("verification_method", IUserDevicesManager::EVerificationMethod::Sms));
    auto session = BuildTx<NSQL::Writable>();
    auto verificationResult = Server->GetUserDevicesManager()->StartDeviceIdVerification(permissions->GetUserId(), GetDeviceId(), permissions->GetPhone(), divc, Context, session, verificationMethod, errorCodes);
    if (verificationResult) {
        TMaybe<TObjectEvent<TUserDevice>> lastDeviceEvent;
        if (!Server->GetUserDevicesManager()->GetLastDeviceEvent({ permissions->GetUserId() }, {}, { EObjectHistoryAction::Proposition }, session, lastDeviceEvent) ||
            !AddTagOnDeviceChange(permissions->GetUserId(), *verificationResult, lastDeviceEvent, session)) {
            errorCodes.emplace("antifraud_failed");
            verificationResult = Nothing();
        }
    }
    if (!verificationResult || !session.Commit()) {
        if (errorCodes.size()) {
            session.SetError(NDrive::MakeError(*errorCodes.begin()));
            session.SetLocalizedMessageKey("verification.device.messages." + *errorCodes.begin());
        } else {
            session.SetLocalizedMessageKey("verification.device.messages.unknown_code");
        }
        R_ENSURE(false, ConfigHttpStatus.UserErrorState, JoinSeq(",", errorCodes), session);
    }
    g.SetCode(HTTP_OK);
}

void TFinishDeviceVerificationProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_ENSURE_EX(Server->GetUserDevicesManager(), TCodedException(ConfigHttpStatus.ServiceUnavailable) << "incorrect server configuration");
    Y_ENSURE_EX(GetNewDeviceStatus() == ENewDeviceStatus::Verification, TCodedException(ConfigHttpStatus.UserErrorState) << "incorrect device verification state");
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const TString& code = GetString(cgi, "code");
    const TInstant deadline = Context->GetRequestDeadline();

    TSet<TString> errorCodes;
    auto session = BuildTx<NSQL::Writable>();
    if (!Server->GetUserDevicesManager()->FinishDeviceIdVerification(permissions->GetUserId(), GetDeviceId(), code, GetClientIp(), deadline, errorCodes, session)) {
        NDrive::TInfoEntitySession session;
        if (errorCodes.size()) {
            session.SetError(NDrive::MakeError(*errorCodes.begin()));
            session.SetLocalizedMessageKey("verification.device.messages." + *errorCodes.begin());
        } else {
            session.SetLocalizedMessageKey("verification.device.messages.unknown_code");
        }
        R_ENSURE(false, ConfigHttpStatus.UserErrorState, JoinSeq(",", errorCodes), session);
    }
    R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "cannot commit", session);
    g.SetCode(HTTP_OK);
}
