#pragma once

#include "config.h"

#include <drive/backend/processors/common_app/data_check.h>
#include <drive/backend/processors/service_app/config.h>
#include <drive/backend/processors/service_app/processor.h>

#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/roles.h>

#include <util/generic/serialized_enum.h>
#include <util/string/split.h>

class TStartDeviceVerificationProcessor: public TDeviceChangeCheckProcessor<TStartDeviceVerificationProcessor, TEmptyConfig> {
private:
    using TBase = TDeviceChangeCheckProcessor<TStartDeviceVerificationProcessor, TEmptyConfig>;

public:
    TStartDeviceVerificationProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {}

    static TString GetTypeName() {
        return "verification/new_device/submit";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/);
};

class TFinishDeviceVerificationProcessor: public TAppCommonProcessor<TFinishDeviceVerificationProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TFinishDeviceVerificationProcessor, TEmptyConfig>;

public:
    TFinishDeviceVerificationProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "verification/new_device/commit";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/);
};

class TUserDevicesListProcessor: public TAppCommonProcessor<TUserDevicesListProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserDevicesListProcessor, TEmptyConfig>;

public:
    TUserDevicesListProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "user_devices_list";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        Y_ENSURE_EX(Server->GetUserDevicesManager(), TCodedException(ConfigHttpStatus.ServiceUnavailable) << "incorrect server configuration");
        TString userId = Context->GetCgiParameters().Get("user_id");
        if (userId) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::User);
        } else {
            userId = permissions->GetUserId();
        }
        auto session = BuildTx<NSQL::ReadOnly>();

        TVector<TUserDevice> userDevices;
        R_ENSURE(Server->GetUserDevicesManager()->GetFullUserDevices(userId, userDevices, session), ConfigHttpStatus.UserErrorState, "cannot GetFullUserDevices", session);
        NJson::TJsonValue userDevicesReport = NJson::JSON_ARRAY;
        for (auto&& i : userDevices) {
            userDevicesReport.AppendValue(i.GetJsonReport());
        }
        g.MutableReport().AddReportElement("devices", std::move(userDevicesReport));

        TVector<TObjectEvent<TUserDevice>> historyEvents;
        ReqCheckCondition(Server->GetUserDevicesManager()->GetFullHistoryByUser(userId, historyEvents, session), ConfigHttpStatus.UnknownErrorStatus, EDriveLocalizationCodes::InternalServerError);
        NJson::TJsonValue userDevicesHistoryReport = NJson::JSON_ARRAY;
        for (auto&& i : historyEvents) {
            userDevicesHistoryReport.AppendValue(i.BuildReportItem());
        }
        g.MutableReport().AddReportElement("history", std::move(userDevicesHistoryReport));

        g.SetCode(HTTP_OK);
    }
};

class TUserDevicesControlProcessor: public TAppCommonProcessor<TUserDevicesControlProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserDevicesControlProcessor, TEmptyConfig>;

public:
    TUserDevicesControlProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "user_devices_control";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        Y_ENSURE_EX(Server->GetUserDevicesManager(), TCodedException(ConfigHttpStatus.ServiceUnavailable) << "incorrect server configuration");
        TString userId = Context->GetCgiParameters().Get("user_id");
        if (userId) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::User);
        } else {
            userId = permissions->GetUserId();
        }
        TMaybe<bool> enabled;
        TMaybe<bool> verified;
        if (Context->GetCgiParameters().Has("enabled")) {
            enabled = IsTrue(Context->GetCgiParameters().Get("enabled"));
        }
        if (Context->GetCgiParameters().Has("verified")) {
            verified = IsTrue(Context->GetCgiParameters().Get("verified"));
        }
        TSet<TString> devices;
        StringSplitter(Context->GetCgiParameters().Get("devices")).SplitBySet(", ").SkipEmpty().Collect(&devices);
        if (!enabled) {
            Y_ENSURE_EX(!devices.contains(GetDeviceId()), TCodedException(ConfigHttpStatus.UserErrorState) << "Cannot block current device");
        }

        Y_ENSURE_EX(Server->GetUserDevicesManager()->SetDevicesFeatures(userId, devices, enabled, verified, permissions->GetUserId()), TCodedException(ConfigHttpStatus.UnknownErrorStatus) << "Cannot block devices (server error)");
        g.SetCode(HTTP_OK);
    }
};

class TUserDeviceRemoveProcessor: public TAppCommonProcessor<TUserDeviceRemoveProcessor, TEmptyConfig> {
private:
    using TBase = TAppCommonProcessor<TUserDeviceRemoveProcessor, TEmptyConfig>;

public:
    TUserDeviceRemoveProcessor(const THandlerConfig& config, IReplyContext::TPtr context, IAuthModule::TPtr authModule, const NDrive::IServer* server)
        : TBase(config, context, authModule, server) {
    }

    static TString GetTypeName() {
        return "user_device_remove";
    }

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
        Y_ENSURE_EX(Server->GetUserDevicesManager(), TCodedException(ConfigHttpStatus.ServiceUnavailable) << "incorrect server configuration");
        TString userId = Context->GetCgiParameters().Get("user_id");
        if (userId) {
            ReqCheckAdmActions(permissions, TAdministrativeAction::EAction::ModifyStructure, TAdministrativeAction::EEntity::User);
        } else {
            userId = permissions->GetUserId();
        }
        TSet<TString> devices;
        StringSplitter(Context->GetCgiParameters().Get("devices")).SplitBySet(", ").SkipEmpty().Collect(&devices);

        Y_ENSURE_EX(Server->GetUserDevicesManager()->RemoveDevices(userId, devices, permissions->GetUserId()), TCodedException(ConfigHttpStatus.UnknownErrorStatus) << "Cannot remove devices (server error)");
        g.SetCode(HTTP_OK);
    }
};

