#include "booking.h"

#include "helpers.h"

#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>

NJson::TJsonValue TBookingScreenCheck::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    auto api = Yensured(server.GetDriveAPI());
    auto& cgi = context->GetCgiParameters();
    auto offerName = cgi.Get("offer_name");
    auto offerBuilderTypes = MakeSet(StringSplitter(cgi.Get("offer_builder_type")).SplitBySet(", ").SkipEmpty().ToList<TString>());
    auto tx = api->template BuildTx<NSQL::Writable>();
    R_ENSURE(TUserProblemTag::EnsureNotBlocked(permissions->GetUserId(), server, tx), HTTP_INTERNAL_SERVER_ERROR, "blocked user", tx);
    R_ENSURE(server.GetOffersStorage(), HTTP_INTERNAL_SERVER_ERROR, "no offer storage");

    TVector<TString> cars;
    R_ENSURE(api->GetCurrentSessionsObjects(permissions->GetUserId(), cars, Now()), HTTP_INTERNAL_SERVER_ERROR, "cannot fetch current session");
    R_ENSURE(cars.empty(), HTTP_FORBIDDEN, "already book");

    TOffersBuildingContext offersBuildingContext(&server);
    R_ENSURE(offersBuildingContext.Parse(context, *permissions, {}, tx), HTTP_BAD_REQUEST, "cannot parse context", tx);

    {
        TUserOfferContext uoc(&server, permissions, context);
        uoc.SetAccountName("card");
        auto origin = cgi.Get("origin");
        if (origin) {
            uoc.SetOrigin(origin);
        }
        R_ENSURE(uoc.Prefetch(), HTTP_INTERNAL_SERVER_ERROR, "user_offer_context_prefetch_error");
        offersBuildingContext.SetUserHistoryContext(std::move(uoc));
    }

    TVector<IOfferReport::TPtr> offersAll;

    auto correctors = permissions->GetOfferCorrections();

    for (auto&& i : permissions->GetOfferBuilders()) {
        auto action = dynamic_cast<const IOfferBuilderAction*>(i.Get());
        if (!action) {
            continue;
        }

        if (offerName && offerName != action->GetName()) {
            continue;
        }

        if (!offerBuilderTypes.empty() && !offerBuilderTypes.contains(action->GetType())) {
            continue;
        }

        TVector<IOfferReport::TPtr> offersCurrent;
        R_ENSURE(action->BuildOffers(*permissions, correctors, offersCurrent, offersBuildingContext, &server, tx) != EOfferCorrectorResult::Problems, HTTP_INTERNAL_SERVER_ERROR, "cannot build offers", tx);

        const auto pred = [&server](IOfferReport::TPtr offer) {
            return !Yensured(offer->GetOffer())->NeedStore(server);
        };
        offersCurrent.erase(std::remove_if(offersCurrent.begin(), offersCurrent.end(), pred), offersCurrent.end());

        offersAll.insert(offersAll.end(), offersCurrent.begin(), offersCurrent.end());
    }
    R_ENSURE(offersAll.size(), HTTP_FORBIDDEN, "no offers");

    auto compareOffers = [](IOfferReport::TPtr left, IOfferReport::TPtr right) {
        if (!left) {
            return true;
        }
        if (!right) {
            return false;
        }
        return
            std::tuple(left->GetListPriorityDef(0), left->GetInternalPriorityDef(0), left->GetOffer()->GetGroupName(), left->GetOffer()->GetName()) <
            std::tuple(right->GetListPriorityDef(0), right->GetInternalPriorityDef(0), right->GetOffer()->GetGroupName(), right->GetOffer()->GetName());
    };
    std::sort(offersAll.begin(), offersAll.end(), compareOffers);
    const auto& offer = offersAll.front();

    auto asyncOffers = server.GetOffersStorage()->StoreOffers({ offer }, tx);
    Y_ENSURE(asyncOffers.Initialized());
    R_ENSURE(tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "cannot commit", tx);
    auto tmpOffer = offer->GetOffer();
    Y_ENSURE(tmpOffer);

    auto landingString = GetLanding("main", permissions, server.GetSettings());
    auto locale = GetLocale();
    landingString = tmpOffer->FormDescriptionElement(landingString, locale, server.GetLocalization());
    SubstGlobal(landingString, "_OfferId_", tmpOffer->GetOfferId());

    if (offersBuildingContext.HasCarId()) {
        auto tx = api->template BuildTx<NSQL::ReadOnly>();
        auto deviceFetchInfo = server.GetDriveDatabase().GetCarManager().FetchInfo(offersBuildingContext.GetCarIdUnsafe(), tx);
        auto device = deviceFetchInfo.GetResultPtr(offersBuildingContext.GetCarIdUnsafe());
        if (device) {
            SubstGlobal(landingString, "_CarId_", device->GetId());
            SubstGlobal(landingString, "_CarNumber_", device->GetNumber());
            auto model = server.GetDriveDatabase().GetModelsDB().Fetch(device->GetModel(), tx);
            if (model) {
                landingString = model->FormDescriptionElement(landingString, locale, server.GetLocalization());
            }
        }
    }

    return GetLocalizedLanding(landingString, server);
}

const TString TBookingScreenCheck::Name = "booking";
IWarningScreenChecker::TFactory::TRegistrator<TBookingScreenCheck> TBookingScreenCheck::Registrator(TBookingScreenCheck::Name);
