#include "car_check.h"

#include <drive/backend/base/server.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/tags/tags_manager.h>
#include <memory>

const TString TCarCheck::Name = "car_check";
IWarningScreenChecker::TFactory::TRegistrator<TCarCheck>
        TCarCheck::Registrator(TCarCheck::Name);

NJson::TJsonValue TCarCheck::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    const auto& settings = server.GetSettings();
    auto event = context->GetCgiParameters().Get("event");
    TString carId = context->GetCgiParameters().Get("car_id");
    TMaybe<TSet<TString>> offerTags;
    if (!carId) {
        const TString offerId = context->GetCgiParameters().Get("offer_id");
        Y_ENSURE(offerId, "Missing offer parameter");

        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        IOffer::TPtr offer;

        auto baseOffer = server.GetOffersStorage()->RestoreOffer(offerId, permissions->GetUserId(), session).GetValueSync();
        Y_ENSURE(baseOffer, "Missing offer " << offerId);
        offer = std::dynamic_pointer_cast<IOffer>(baseOffer);
        Y_ENSURE(offer, "Not a vehicle offer: " << offerId);
        auto action = server.GetDriveDatabase().GetUserPermissionManager().GetAction(offer->GetBehaviourConstructorId());
        auto builder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
        if (builder) {
            offerTags = builder->GetGrouppingTags();
        }
        carId = offer->GetObjectId();
    }

    auto tx = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().BuildTx<NSQL::ReadOnly>();
    const auto tags = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreEntityTags(carId, {}, tx);
    Y_ENSURE(tags, "cannot fetch tags");

    auto tagsFilterSettingKey = "warning_screens.checkers." + event + "." + GetName() + ".tags_filters";
    auto tagsFilterString = permissions->GetSetting<TString>(settings, tagsFilterSettingKey, "[]");
    NJson::TJsonValue filters;
    Y_ENSURE(NJson::ReadJsonFastTree(tagsFilterString, &filters) && filters.IsArray(), "incorrect filter's json");
    for (const auto& filter : filters.GetArray()) {
        auto carFilter = TTagsFilter::BuildFromString(filter["car_tags"].GetString());
        auto offerFilter = TTagsFilter::BuildFromString(filter["offer_tags"].GetString());
        if (carFilter.IsMatching(*tags) && (offerFilter.IsEmpty() || offerTags && offerFilter.IsMatching(*offerTags))) {
            return GetLocalizedLanding(filter["landing"].GetString(), permissions, server);
        }
    }
    return NJson::JSON_NULL;
}
