#include "checker.h"

#include <drive/backend/abstract/settings.h>
#include <drive/backend/roles/permissions.h>

#include <util/string/builder.h>

TString IWarningScreenChecker::GetLandingSettingId(const TString& landingId) const {
    return TStringBuilder() << "warning_screens.checkers." << GetName() << ".landings." << landingId;
}

IWarningScreenChecker::TResponse IWarningScreenChecker::Call(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    return WrapUnexpected<yexception>(std::bind(&IWarningScreenChecker::CheckImpl, this,
                                                std::ref(server), std::ref(context),
                                                std::ref(permissions)));
}

NJson::TJsonValue IWarningScreenChecker::GetLocalizedLanding(const TString& landingId, TIntrusiveConstPtr<TUserPermissions> permissions, const NDrive::IServer& server) const {
    auto settingId = GetLandingSettingId(landingId);
    auto landing = permissions->GetSettings().GetJsonValue(settingId, /*throwOnParseError=*/true);
    Y_ENSURE(landing.IsDefined(), "missing landing " << settingId);
    return GetLocalizedLanding(std::move(landing), server);
}

NJson::TJsonValue IWarningScreenChecker::GetLocalizedLanding(const TString& landingString, const NDrive::IServer& server) const {
    auto landing = NJson::ReadJsonFastTree(landingString);
    return GetLocalizedLanding(std::move(landing), server);
}

NJson::TJsonValue IWarningScreenChecker::GetLocalizedLanding(NJson::TJsonValue&& landing, const NDrive::IServer& server) const {
    auto localization = server.GetLocalization();
    if (localization) {
        localization->ApplyResourcesForJson(landing, Locale);
    }
    return landing;
}

TString IWarningScreenChecker::GetLanding(const TString& landingId, TUserPermissions::TConstPtr permissions, const NDrive::ISettings& settings) const {
    const TString settingId = GetLandingSettingId(landingId);
    TString landing = permissions->GetSetting<TString>(settings, settingId, "");
    Y_ENSURE(landing, "Missing landing: " << settingId);
    return landing;
}

IWarningScreenChecker::TEventArgs IWarningScreenChecker::GetEventParameters(const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions, const NDrive::ISettings& settings) const {
    const TString event = context->GetCgiParameters().Get("event");
    Y_ASSERT(event);
    const TString eventArgs = permissions->GetSetting<TString>(
            settings,
            TStringBuilder() << "warning_screens.events." << event << "." << GetName() << ".args",
            TString());
    return TCgiParameters(eventArgs);
}
