#include "corp_prepay_check.h"

#include "helpers.h"

#include <drive/backend/base/server.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/billing/manager.h>

const TString TCorpPrepayCheck::Name = "corp_prepay_check";
IWarningScreenChecker::TFactory::TRegistrator<TCorpPrepayCheck> TCorpPrepayCheck::Registrator(TCorpPrepayCheck::Name);

NJson::TJsonValue TCorpPrepayCheck::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    const TString offerId = context->GetCgiParameters().Get("offer_id");
    Y_ENSURE(offerId, "Missing offer parameter");

    ICommonOffer::TPtr offer;
    {
        NDrive::TInfoEntitySession session;
        offer = server.GetOffersStorage()->RestoreOffer(offerId, permissions->GetUserId(), session).GetValueSync();
        Y_ENSURE(!!offer, "Missing offer " << offerId);
    }

    if (!Yensured(server.GetDriveAPI())->HasBillingManager()) {
        return NJson::JSON_NULL;
    }

    const ISettings& settings = server.GetSettings();

    TSet<TString> unavailableMethods = StringSplitter(permissions->GetSetting<TString>(settings, "warning_screens.checkers." + TCorpPrepayCheck::Name + ".unavailable_payment_methods").GetOrElse("card,yandex_account")).Split(',').SkipEmpty();

    if (unavailableMethods.contains(offer->GetSelectedCharge())) {
        return NJson::JSON_NULL;
    }

    NDrive::NBilling::IBillingAccount::TPtr billingAccount;
    auto userAccounts = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetUserAccounts(permissions->GetUserId(), Now());
    for (auto&& account : userAccounts) {
        if (account->GetUniqueName() == offer->GetSelectedCharge()) {
            billingAccount = account;
            break;
        }
    }
    if (!billingAccount) {
        return NJson::JSON_NULL;
    }

    auto getLanding = [&](const TString& key) {
        if (const TString specialLanding = GetSpecialLanding(settings, permissions, *offer, *server.GetDriveAPI(), key)) {
            return specialLanding;
        } else {
            return GetLanding(key + "standart", permissions, settings);
        }
    };

    TMaybe<i64> balance;
    TMaybe<i64> limit;
    TString balanceLanding;
    TString limitLanding;
    if (billingAccount->GetType() == NDrive::NBilling::EAccount::Trust) {
        bool checkCards = permissions->GetSetting<bool>("billing.book.check_cards", true);
        if (!checkCards) {
            return NJson::JSON_NULL;
        }
        auto cards = server.GetDriveAPI()->GetUserPaymentMethodsSync(*permissions, server, false);
        auto cardsList = TBillingManager::GetUserPaymentCards(cards.Get(), false);
        if (!cardsList.Defined()) {
            return NJson::JSON_NULL;
        }
        if (cardsList->empty()) {
            limitLanding = GetLanding("no_cards", permissions, settings);
            limit = -1;
        } else if (offer->GetSelectedCreditCard()) {
            for (const auto& card : *cardsList) {
                if (card.Check(offer->GetSelectedCreditCard())) {
                    if (card.OptionalPayerInfo() && card.OptionalPayerInfo()->OptionalFamilyInfo()) {
                        auto& familyInfo = card.OptionalPayerInfo()->GetFamilyInfoRef();
                        balance = Max<i64>(0, familyInfo.GetLimit() - familyInfo.GetExpenses());
                        limit = familyInfo.GetLimit();
                        balanceLanding = getLanding("family_balance_");
                        limitLanding = getLanding("family_limit_");
                    }
                }
            }
        }
    } else {
        balance = billingAccount->GetBalance();
        limit = billingAccount->GetSoftLimit();
        if (billingAccount->GetParent()) {
            balance = Min<i64>(*balance, billingAccount->GetParent()->GetBalance());
            limit = Min<i64>(*limit, billingAccount->GetParent()->GetSoftLimit());
        }
        balanceLanding = getLanding("");
        limitLanding = getLanding("limit_");
    }

    auto locale = GetLocale();
    auto& localization = *Yensured(server.GetLocalization());

    ui32 price = offer->GetDeposit();
    auto substrInfo = [&] (TString& landing, i64 balance) {
        if (auto packOffer = dynamic_cast<const TPackOffer*>(offer.Get())) {
            SubstGlobal(landing, "_PackPrice_", localization.FormatPrice(locale, packOffer->GetPublicDiscountedPackPrice()));
        } else {
            SubstGlobal(landing, "_PackPrice_", localization.FormatPrice(locale, price));
        }

        SubstGlobal(landing, "_Deposit_", localization.FormatPrice(locale, price));
        SubstGlobal(landing, "_WalletBalance_", (balance < 0 ? "-" : "") + localization.FormatPrice(locale, std::abs(balance)));
        SubstGlobal(landing, "_Difference_", localization.FormatPrice(locale, price - Max<i64>(balance, 0)));
    };

    if (limit && *limit < price) {
        substrInfo(limitLanding, *limit);
        return GetLocalizedLanding(limitLanding, server);
    }

    if (balance && *balance < price) {
        substrInfo(balanceLanding, *balance);
        return GetLocalizedLanding(balanceLanding, server);
    }
    return NJson::JSON_NULL;
}

TString TCorpPrepayCheck::GetSpecialLanding(const ISettings& settings, TUserPermissions::TConstPtr permissions, const ICommonOffer& offer, const TDriveAPI& driveApi, const TString& prefix) const {
    const auto dbConstructor = driveApi.GetRolesManager()->GetActionsDB().GetObject(offer.GetBehaviourConstructorId());
    const IOfferBuilderAction* constructor = dbConstructor ? dbConstructor->GetAs<IOfferBuilderAction>() : nullptr;
    if (constructor) {
        for (auto&& groupingTag : constructor->GetGrouppingTags()) {
            const TString settingId = GetLandingSettingId(prefix + "grouping_tag_" + groupingTag);
            if (const TString landing = permissions->GetSetting<TString>(settings, settingId, "")) {
                return landing;
            }
        }
    }
    return TString();
}
