#include "debt_check.h"

#include <drive/backend/base/server.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/roles/permissions.h>

#include <util/string/cast.h>
#include <util/string/subst.h>

const TString TUserDebtCheck::Name = "user_debt_check";
IWarningScreenChecker::TFactory::TRegistrator<TUserDebtCheck>
        TUserDebtCheck::Registrator(TUserDebtCheck::Name);

NJson::TJsonValue TUserDebtCheck::CheckImpl(const NDrive::IServer& server,
                                            const IReplyContext::TPtr context,
                                            TUserPermissions::TConstPtr permissions) const {
    if (!server.GetDriveAPI()->HasBillingManager()) {
        return NJson::JSON_NULL;
    }

    auto evlog = NDrive::GetThreadEventLogger();
    TMaybe<ui32> debt;
    {
        const auto& billingManager = server.GetDriveAPI()->GetBillingManager();
        auto session = billingManager.BuildSession(true);
        debt = billingManager.GetDebt(permissions->GetUserId(), session);
        if (!debt) {
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("event", "GetDebtError")
                    ("errors", session.GetReport())
                );
            }
            session.ClearErrors();
        }
    }

    const auto eventArgs = GetEventParameters(context, permissions, server.GetSettings());
    const TString event = eventArgs.Get("event");
    Y_ENSURE(event, "Missing event");

    TString settingsPrefix = TStringBuilder() << "warning_screens.checkers." << GetName() << "." << event << ".";

    auto debtLimit = server.GetSettings().GetValueDef<ui64>(settingsPrefix + "debt_limit", 99);
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "UserDebtCheck")
            ("debt", NJson::ToJson(debt))
            ("debt_limit", NJson::ToJson(debtLimit))
        );
    }

    const bool userHasDebt = debt && *debt > debtLimit;
    if (!userHasDebt) {
        return NJson::JSON_NULL;
    }

    TString landing = GetLanding(server.GetSettings().GetValueDef<TString>(settingsPrefix + "landing_id", "main"), permissions, server.GetSettings());
    const TString debtStr = server.GetLocalization()->FormatPrice(GetLocale(), *debt);
    SubstGlobal(landing, "_Debt_", debtStr);
    return GetLocalizedLanding(landing, server);
}
