#include "helpers.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>

TAtomicSharedPtr<const ISession> GetSession(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) {
    const auto& cgi = Yensured(context)->GetCgiParameters();
    const auto& sessionId = cgi.Get("session_id");
    TAtomicSharedPtr<const ISession> userSession;
    R_ENSURE(
        server.GetDriveAPI()->GetUserSession(permissions->GetUserId(), userSession, sessionId, context->GetRequestStartTime()),
        HTTP_INTERNAL_SERVER_ERROR,
        "cannot GetUserSession"
    );
    return userSession;
}

TAtomicSharedPtr<const TBillingSession> GetBillingSession(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) {
    auto session = GetSession(server, context, permissions);
    return std::dynamic_pointer_cast<const TBillingSession>(session);
}

TBillingSession::TBillingCompilation GetBillingSessionCompilationSafe(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) {
    auto userSession = GetSession(server, context, permissions);
    auto eventsSession = dynamic_cast<const IEventsSession<TCarTagHistoryEvent>*>(userSession.Get());
    Y_ENSURE(eventsSession && !eventsSession->GetClosed(), "No active session");

    TMaybe<TBillingSession::TBillingCompilation> compilation =
            eventsSession->GetCompilationAs<TBillingSession::TBillingCompilation>();
    Y_ENSURE(compilation, "No default compilation in session " << userSession->GetSessionId());

    return *compilation;
}

TMaybe<TBillingSession::TBillingCompilation> GetBillingSessionCompilation(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) {
    auto userSession = GetSession(server, context, permissions);
    auto eventsSession = dynamic_cast<const IEventsSession<TCarTagHistoryEvent>*>(userSession.Get());
    if (!eventsSession || eventsSession->GetClosed()) {
        return {};
    }
    return eventsSession->GetCompilationAs<TBillingSession::TBillingCompilation>();
}
