#include "long_term.h"

#include "helpers.h"

#include <drive/backend/data/long_term.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/actions/long_term.h>

NJson::TJsonValue TLongTermCancellationCheck::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    auto locale = GetLocale();
    auto localization = server.GetLocalization();
    auto api = Yensured(server.GetDriveAPI());
    const auto& userTagManager = api->GetTagsManager().GetUserTags();

    const auto& sessionId = context->GetCgiParameters().Get("session_id");

    auto session = userTagManager.BuildSession(true);
    auto optionalUserTags = userTagManager.RestoreObject(permissions->GetUserId(), session);
    if (!optionalUserTags) {
        return {};
    }
    for (auto&& tag : optionalUserTags->GetTags()) {
        auto offerHolder = tag.GetTagAs<TOfferHolderTag>();
        if (!offerHolder) {
            continue;
        }
        auto offer = offerHolder->GetOffer();
        auto longTermOffer = std::dynamic_pointer_cast<TLongTermOffer>(offer);
        if (!longTermOffer) {
            continue;
        }
        if (sessionId && sessionId != offer->GetOfferId()) {
            continue;
        }
        auto stage = offerHolder->GetStage(nullptr);
        auto landingString = GetLanding(stage, permissions, server.GetSettings());
        landingString = offer->FormDescriptionElement(landingString, locale, server.GetLocalization());
        return GetLocalizedLanding(landingString, server);
    }

    auto compilation = GetBillingSessionCompilation(server, context, permissions);
    if (compilation) {
        auto offer = compilation->GetCurrentOffer();
        auto state = compilation->GetCurrentOfferState();
        auto longTermOffer = std::dynamic_pointer_cast<TLongTermOffer>(offer);
        auto longTermOfferState = std::dynamic_pointer_cast<TLongTermOfferState>(state);
        if (longTermOffer && longTermOfferState && longTermOfferState->GetPackRemainingTime() > longTermOffer->GetEarlyReturnRemainder()) {
            bool materialized = longTermOfferState->GetMaterialized();
            auto name = materialized ? "main" : TLongTermOfferHolderTag::Preparation;
            auto landingString = GetLanding(name, permissions, server.GetSettings());
            landingString = offer->FormDescriptionElement(landingString, locale, localization);
            if (longTermOfferState) {
                SubstGlobal(landingString, "_RemainingDuration_", localization->MonthsFormat(longTermOfferState->GetPackRemainingTime(), locale));
                SubstGlobal(landingString, "_RemainingTime_", localization->MonthsFormat(longTermOfferState->GetPackRemainingTime(), locale));
                SubstGlobal(landingString, "_ReturnCost_", localization->FormatPrice(locale, longTermOfferState->GetReturnCost()));
            }

            return GetLocalizedLanding(landingString, server);
        }
    }

    return {};
}

const TString TLongTermCancellationCheck::Name = "long_term_cancellation";
IWarningScreenChecker::TFactory::TRegistrator<TLongTermCancellationCheck> TLongTermCancellationCheck::Registrator(TLongTermCancellationCheck::Name);
