#include "pack_offer_check.h"

#include "helpers.h"

#include <drive/backend/base/server.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/actions/flexipack.h>
#include <drive/backend/offers/actions/pack.h>

const TString TPackOfferUsageCheck::Name = "pack_offer_usage_check";
IWarningScreenChecker::TFactory::TRegistrator<TPackOfferUsageCheck> TPackOfferUsageCheck::Registrator(TPackOfferUsageCheck::Name);

NJson::TJsonValue TPackOfferUsageCheck::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    auto compilation = GetBillingSessionCompilationSafe(server, context, permissions);
    auto locale = GetLocale();
    auto localization = server.GetLocalization();

    auto state = compilation.GetCurrentOfferState();
    Y_ENSURE(state, "No offer state in default compilation in session " << compilation.GetSessionId());

    if (!compilation.IsAccepted() && compilation.GetReportSumPrice() == 0.) {
        return NJson::JSON_NULL;
    }

    auto offer = compilation.GetCurrentOffer();
    if (offer->GetTypeName() != TPackOffer::GetTypeNameStatic() && offer->GetTypeName() != TFlexiblePackOffer::GetTypeNameStatic()) {
        return NJson::JSON_NULL;
    }

    const TPackOfferState* packOfferState = dynamic_cast<const TPackOfferState*>(state.Get());
    Y_ENSURE(packOfferState, "No pack offer state in session " << compilation.GetSessionId());

    auto packOffer = dynamic_cast<const TPackOffer*>(offer.Get());
    Y_ENSURE(packOffer, "No pack offer in session " << compilation.GetSessionId());

    const TDuration minRemainingTime = server.GetSettings().GetValueDef<TDuration>(
        "warning_screens.checkers." + Name + ".min_remaining_time", TDuration::Seconds(1));
    if (packOfferState->GetPackRemainingTime() < minRemainingTime) {
        return NJson::JSON_NULL;
    }

    TString landing;
    if (packOfferState->GetUsedTime() > packOffer->GetReturningDuration()) {
        landing = GetLanding("high", permissions, server.GetSettings());
        if (localization) {
            const TString remainingTime = localization->FormatDuration(locale, packOfferState->GetPackRemainingTime());
            const TString remainingDistance = localization->DistanceFormatKm(locale, packOfferState->GetRemainingDistance(), true);
            SubstGlobal(landing, "_RemainingDuration_", remainingTime);
            SubstGlobal(landing, "_RemainingDistance_", remainingDistance);
        }
    } else {
        landing = GetLanding("low", permissions, server.GetSettings());
    }
    return GetLocalizedLanding(landing, server);
}
