#include "parking_check.h"

#include "helpers.h"

#include <drive/backend/base/server.h>
#include <drive/backend/parking_zones/manager.h>
#include <drive/backend/roles/permissions.h>

namespace {
    TString GetParkingEventId(const ISettings& settings) {
        return settings.GetValueDef<TString>(
                "warning_screens.checkers." + TParkingCheck::Name + ".parking_event", "parking");
    }
}  // namespace

const TString TParkingCheck::Name = "parking_check";
IWarningScreenChecker::TFactory::TRegistrator<TParkingCheck>
        TParkingCheck::Registrator(TParkingCheck::Name);

NJson::TJsonValue TParkingCheck::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context,
                                           TUserPermissions::TConstPtr permissions) const {
    const TString event = context->GetCgiParameters().Get("event");
    const TString precisionStr = context->GetCgiParameters().Get("precision");
    TMaybe<double> precision = precisionStr ? ::FromString<double>(precisionStr) : TMaybe<double>();

    auto compilation = GetBillingSessionCompilationSafe(server, context, permissions);
    if (!compilation.IsAccepted()) {
        return NJson::JSON_NULL;
    }

    auto res = server.GetParkingZonesManager().IsParkingAllowed(permissions->GetUserId(), *context, precision);
    Y_ENSURE(res, res.GetError());

    TString settingId;
    const bool isParkingEvent = GetParkingEventId(server.GetSettings()) == event;
    if (isParkingEvent && res->SignalIssues) {
        settingId = GetLandingSettingId("signal");
    } else if (res->Ability == EParkingAbility::Deny) {
        if (res->DenyDetails == EParkingDenyDetails::ClosedZone) {
            if (!isParkingEvent) {
                settingId = GetLandingSettingId("closed_zone");
            }
        } else {
            settingId = GetLandingSettingId("parking_denied");
        }
    }

    if (!settingId) {
        return NJson::JSON_NULL;
    }

    auto landing = permissions->GetSetting<TString>(server.GetSettings(), settingId, "");
    Y_ENSURE(landing, "Missing landing:" << settingId);
    return GetLocalizedLanding(landing, server);
}
