#include "prepay_check.h"

#include "helpers.h"

#include <drive/backend/base/server.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/flexipack.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/roles/manager.h>

namespace {
    void FillLandingTemplate(TString& landing, ELocalization locale, const ILocalization& localization, const ICommonOffer& offer) {
        const TString localizedDeposit = localization.FormatPrice(locale, offer.GetDeposit());
        SubstGlobal(landing, "_Deposit_", localizedDeposit);
        SubstGlobal(landing, "_OfferPrice_", localizedDeposit);
        SubstGlobal(landing, "_OfferName_", offer.GetName());
        if (auto packOffer = dynamic_cast<const TPackOffer*>(&offer)) {
            SubstGlobal(landing, "_RerunPriceKm_",
                        localization.FormatPrice(
                            locale, packOffer->GetPublicDiscountedPrice(packOffer->GetOverrunKm(), ""),
                            {"units.short." + packOffer->GetCurrency(), "/", "units.short.km"}));
            SubstGlobal(landing, "_RerunPriceMin_",
                        localization.FormatPrice(
                            locale, packOffer->GetPublicDiscountedRiding(),
                            {"units.short." + packOffer->GetCurrency(), "/", "units.short.minutes"}));

            const TDuration duration = packOffer->GetDuration();
            const ui32 milageLimit = packOffer->GetMileageLimit();
            SubstGlobal(landing, "_PackDuration_", localization.FormatDuration(locale, duration));
            SubstGlobal(landing, "_PackDurationHours_", localization.HoursFormat(duration, locale));
            SubstGlobal(landing, "_PackDurationMinutes_", localization.MinutesFormat(duration, locale));
            SubstGlobal(landing, "_PackDurationDays_", localization.DaysFormat(duration, locale));
            SubstGlobal(landing, "_PackDurationHoursOrDays_", duration.Days() >= 1 ? localization.DaysFormat(duration, locale) : localization.HoursFormat(duration, locale));
            SubstGlobal(landing, "_OfferDistance_", localization.DistanceFormatKm(locale, milageLimit, true));

            const TDuration totalDuration = duration + packOffer->GetExtraDuration();
            const ui32 totalMilage = milageLimit + packOffer->GetExtraMileageLimit();
            SubstGlobal(landing, "_PackDurationTotal_", localization.FormatDuration(locale, totalDuration));
            SubstGlobal(landing, "_PackDurationHoursTotal_", localization.HoursFormat(totalDuration, locale));
            SubstGlobal(landing, "_PackDurationMinutesTotal_", localization.MinutesFormat(totalDuration, locale));
            SubstGlobal(landing, "_PackDurationDaysTotal_", localization.DaysFormat(totalDuration, locale));
            SubstGlobal(landing, "_PackDurationHoursOrDaysTotal_", totalDuration.Days() >= 1 ? localization.DaysFormat(totalDuration, locale) : localization.HoursFormat(totalDuration, locale));
            SubstGlobal(landing, "_OfferDistanceTotal_", localization.DistanceFormatKm(locale, totalMilage, true));
        }
    }

    bool GetAlwaysShowStandartLanding(const ISettings& settings) {
        return settings.GetValueDef<bool>(
            "warning_screens.checkers." + TPrepayCheck::Name + ".always_show_standart_landing",
            false
        );
    }

    bool HasCurrentPackOffer(const NDrive::IServer& server, const IReplyContext::TPtr context,
                             TUserPermissions::TConstPtr permissions) {
        const auto compilation = GetBillingSessionCompilation(server, context, permissions);
        if (!compilation) {
            return false;
        }

        if (!compilation->IsAccepted() || compilation->GetReportSumPrice() == 0.) {
            return false;
        }

        auto offer = compilation->GetCurrentOffer();
        return offer->GetTypeName() == TPackOffer::GetTypeNameStatic() || offer->GetTypeName() == TFlexiblePackOffer::GetTypeNameStatic();
    }
}

const TString TPrepayCheck::Name = "prepay_check";
IWarningScreenChecker::TFactory::TRegistrator<TPrepayCheck> TPrepayCheck::Registrator(TPrepayCheck::Name);

NJson::TJsonValue TPrepayCheck::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    const TString offerId = context->GetCgiParameters().Get("offer_id");
    Y_ENSURE(offerId, "Missing offer parameter");

    ICommonOffer::TPtr offer;
    {
        NDrive::TInfoEntitySession session;
        offer = server.GetOffersStorage()->RestoreOffer(offerId, permissions->GetUserId(), session).GetValueSync();
        Y_ENSURE(!!offer, "Missing offer " << offerId);
    }

    if (!offer->GetDeposit()) {
        return NJson::JSON_NULL;
    }

    auto standartOffer = dynamic_cast<const TStandartOffer*>(offer.Get());
    if (standartOffer && !standartOffer->GetUseDeposit()) {
        return NJson::JSON_NULL;
    }

    const ISettings& settings = server.GetSettings();
    TString landing;
    if (const TString specialLanding = GetSpecialLanding(settings, permissions, *offer, *server.GetDriveAPI())) {
        landing = specialLanding;
    } else if (offer->GetTypeName() == TPackOffer::GetTypeNameStatic()) {
        auto packOffer = dynamic_cast<const TPackOffer*>(offer.Get());
        Y_ENSURE(packOffer, "Wrong pack offer cast for offer:" << offer->GetOfferId());

        TString landingId;
        if (HasCurrentPackOffer(server, context, permissions)) {
            landingId = "pack_to_pack_switching";
        } else if (packOffer->GetMileageLimit() > 0.f) {
            landingId = "pack";
        } else {
            if (packOffer->GetOverrunKm()) {
                landingId = "pack_no_mileage";
            } else {
                landingId = "pack_free_mileage";
            }
        }
        landing = GetLanding(landingId, permissions, settings);
    } else if (
        offer->GetTypeName() == TStandartOffer::GetTypeNameStatic() &&
        (permissions->IsFirstRiding() || GetAlwaysShowStandartLanding(settings))
    ) {
        landing = GetLanding("standart", permissions, settings);
    } else {
        return NJson::JSON_NULL;
    }

    FillLandingTemplate(landing, GetLocale(), *server.GetLocalization(), *offer);
    return GetLocalizedLanding(landing, server);
}

TString TPrepayCheck::GetSpecialLanding(const ISettings& settings, TUserPermissions::TConstPtr permissions, const ICommonOffer& offer, const TDriveAPI& driveApi) const {
    const auto dbConstructor = driveApi.GetRolesManager()->GetActionsDB().GetObject(offer.GetBehaviourConstructorId());
    const IOfferBuilderAction* constructor = dbConstructor ? dbConstructor->GetAs<IOfferBuilderAction>() : nullptr;
    if (constructor) {
        for (auto&& groupingTag : constructor->GetGrouppingTags()) {
            const TString settingId = GetLandingSettingId("grouping_tag_" + groupingTag);
            if (const TString landing = permissions->GetSetting<TString>(settings, settingId, "")) {
                return landing;
            }
        }
    }
    return TString();
}
