#include "processor.h"

void TWarningScreenProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const auto& cgi = Context->GetCgiParameters();
    const auto& event = GetString(cgi, "event");
    const auto locale = GetLocale();
    auto checkers = NDrive::GetWarningScreenCheckers(event, locale, permissions);
    NJson::TJsonValue landings = NJson::JSON_ARRAY;
    NJson::TJsonValue errors = NJson::JSON_MAP;
    for (auto&& c : checkers) {
        Y_ENSURE(c, "Expected non-empty value");
        auto landing = c->Call(*Server, Context, permissions);
        if (landing && landing->IsDefined()) {
            landings.AppendValue(std::move(*landing));
        } else if (!landing) {
            errors[c->GetName()] = landing.GetError().what();
        }
    }
    g.MutableReport().AddReportElement("items", std::move(landings));
    g.MutableReport().AddReportElement("errors", std::move(errors));
    g.SetCode(HTTP_OK);
}

namespace NDrive {

    TVector<THolder<IWarningScreenChecker>> GetWarningScreenCheckers(const TString& event, ELocalization locale, TUserPermissions::TPtr permissions) {
        TVector<THolder<IWarningScreenChecker>> checkers;
        {
            TSet<TString> allCheckersIds;
            IWarningScreenChecker::TFactory::GetRegisteredKeys(allCheckersIds);
            TVector<std::pair<TString, ui32>> eventCheckers;
            for (const auto& key : allCheckersIds) {
                const TString settingId = TStringBuilder() << "warning_screens.events." << event << "." << key << ".enabled";
                bool enabled = permissions->GetSetting<bool>(settingId, false);
                if (enabled) {
                    const auto priority = permissions->GetSetting<ui32>(TStringBuilder() << "warning_screens.events." << event << "." << key << ".priority", 0);
                    eventCheckers.emplace_back(key, priority);
                }
            }
            Sort(eventCheckers, [](auto&& left, auto&& right) { return left.second > right.second; });
            for (const auto& c : eventCheckers) {
                checkers.emplace_back(IWarningScreenChecker::TFactory::MakeHolder(c.first));
                checkers.back()->SetLocale(locale);
            }
        }
        return checkers;
    }

}
