#include "rental_finish_fuel_checker.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/roles/permissions.h>

#include <drive/library/cpp/searchserver/context/replier.h>

namespace {
constexpr auto tolerancePetrolPercentage = 5;
}

NJson::TJsonValue TRentalFinishFuelChecker::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    Y_UNUSED(server, context, permissions);
    auto locale = GetLocale();
    auto localization = server.GetLocalization();
    auto driveApi = Yensured(server.GetDriveAPI());
    const auto& cgi = context->GetCgiParameters();
    const auto& offerId = cgi.Get("session_id");
    Y_ENSURE(offerId, "Missing offer parameter");

    TAtomicSharedPtr<const ISession> session;
    R_ENSURE(driveApi->GetUserSession(permissions->GetUserId(), session, offerId, Now()), HTTP_INTERNAL_SERVER_ERROR, "cannot GetRequestUserSession");
    R_ENSURE(session, HTTP_NOT_FOUND, "no session found");
    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(session);
    R_ENSURE(billingSession, HTTP_INTERNAL_SERVER_ERROR, "cannot cast session " << session->GetSessionId() << " to BillingSession");
    auto offer = billingSession->GetCurrentOffer();
    R_ENSURE(offer, HTTP_INTERNAL_SERVER_ERROR, "cannot find Offer in session " << session->GetSessionId());
    auto rentalOffer = std::dynamic_pointer_cast<TRentalOffer>(offer);
    R_ENSURE(rentalOffer, HTTP_INTERNAL_SERVER_ERROR, "cannot cast RentalOffer in session " << session->GetSessionId());

    auto snapshotComplilation = billingSession->MakeCompilation<TSnapshotsDiffCompilation>();
    R_ENSURE(snapshotComplilation, HTTP_INTERNAL_SERVER_ERROR, "cannot make SnapshotsDiffCompilation");
    auto snapshotDiff = snapshotComplilation->GetSnapshotsDiff(&server);
    auto startFuelLevel = snapshotDiff ? snapshotDiff->OptionalStartFuelLevel() : Nothing();
    auto finishFuelLevel = snapshotDiff ? snapshotDiff->OptionalLastFuelLevel() : Nothing();

    if (startFuelLevel && finishFuelLevel && *startFuelLevel - *finishFuelLevel > tolerancePetrolPercentage) {
        auto tx = driveApi->template BuildTx<NSQL::ReadOnly>();
        const auto carsFetchResult = driveApi->GetCarsData()->FetchInfo(offer->GetObjectId(), tx);
        R_ENSURE(carsFetchResult, HTTP_INTERNAL_SERVER_ERROR, "RentalFinishChecker::CheckImpl cannot fetch car");
        const auto carsInfo = carsFetchResult.GetResult();
        R_ENSURE(carsInfo.size() == 1, HTTP_INTERNAL_SERVER_ERROR, "RentalFinishChecker::CheckImpl wrong carsInfo.size()");
        const auto& car = carsInfo.begin()->second;
        auto modelFetchResult = driveApi->GetModelsData()->GetCachedOrFetch(car.GetModel());
        R_ENSURE(modelFetchResult, HTTP_INTERNAL_SERVER_ERROR, "RentalFinishChecker::CheckImpl cannot fetch model");
        auto modelPtr = modelFetchResult.GetResultPtr(car.GetModel());
        R_ENSURE(modelPtr, HTTP_INTERNAL_SERVER_ERROR, "RentalFinishChecker::CheckImpl cannot find modelPtr");

        const auto fuelTankVolume = NDrive::GetFuelTankVolume(&car, modelPtr);
        const auto needFuelingStart = NDrive::GetNeedFuelingLiters(*startFuelLevel, fuelTankVolume);
        const auto needFuelingFinish = NDrive::GetNeedFuelingLiters(*finishFuelLevel, fuelTankVolume);
        const auto roundedLitersStart = std::round(fuelTankVolume - needFuelingStart);
        const auto roundedLitersFinish = std::round(fuelTankVolume - needFuelingFinish);

        auto landingTemplate = GetLanding("fuel_limit_exceed", permissions, server.GetSettings());
        landingTemplate = offer->FormDescriptionElement(landingTemplate, locale, localization);
        SubstGlobal(landingTemplate, "_FuelLevelLitersStart_", ToString(roundedLitersStart));
        SubstGlobal(landingTemplate, "_FuelLevelLitersFinish_", ToString(roundedLitersFinish));
        SubstGlobal(landingTemplate, "_FuelVolumeStart_", ToString(roundedLitersStart));
        SubstGlobal(landingTemplate, "_FuelVolumeFinish_", ToString(roundedLitersFinish));
        return GetLocalizedLanding(landingTemplate, server);
    }

    return NJson::JSON_NULL;
}

const TString TRentalFinishFuelChecker::Name = "rental_finish_fuel_checker";
IWarningScreenChecker::TFactory::TRegistrator<TRentalFinishFuelChecker> TRentalFinishFuelChecker::Registrator(TRentalFinishFuelChecker::Name);
