#include "rental_finish_location_checker.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/drivematics/zone/zone.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/device_snapshot/snapshots/snapshot.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/roles/permissions.h>

#include <drive/library/cpp/searchserver/context/replier.h>

namespace {
    const auto allowDropTagNameDefault{"allow_drop_car"};

    const auto maxDistanceToPoint = 1000.;

    using TAreaIdToArea = TMap<TString, TArea>;
    using TLengthToLocationName = TMap<double, TString>;
}

NJson::TJsonValue TRentalFinishLocationChecker::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    auto locale = GetLocale();
    auto localization = server.GetLocalization();
    auto driveApi = Yensured(server.GetDriveAPI());
    const auto& areaManager = driveApi->GetAreaManager();

    const auto& cgi = context->GetCgiParameters();
    const auto& offerId = cgi.Get("session_id");
    Y_ENSURE(offerId, "Missing offer parameter");

    auto requiredTagsInArea = MakeSet(
        StringSplitter(server.GetSettings().GetValueDef<TString>("rental.drop_off_tags", allowDropTagNameDefault)).SplitBySet(", ").SkipEmpty().ToList<TString>()
    );

    auto tx = driveApi->template BuildTx<NSQL::ReadOnly>();

    TAtomicSharedPtr<const ISession> session;
    R_ENSURE(driveApi->GetUserSession(permissions->GetUserId(), session, offerId, Now()), HTTP_INTERNAL_SERVER_ERROR, "cannot GetRequestUserSession", tx);
    R_ENSURE(session && session->GetSessionId() == offerId, HTTP_NOT_FOUND, "no session found", tx);
    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(session);
    R_ENSURE(billingSession, HTTP_INTERNAL_SERVER_ERROR, "cannot cast session " << session->GetSessionId() << " to BillingSession", tx);
    auto offer = billingSession->GetCurrentOffer();
    R_ENSURE(offer, HTTP_INTERNAL_SERVER_ERROR, "cannot find Offer in session " << session->GetSessionId(), tx);
    auto rentalOffer = std::dynamic_pointer_cast<TRentalOffer>(offer);
    R_ENSURE(rentalOffer, HTTP_INTERNAL_SERVER_ERROR, "cannot cast RentalOffer in session " << session->GetSessionId(), tx);

    if (!rentalOffer->HasReturnLocations()) {
        auto landingTemplateNotFoundLocations = GetLanding("return_locations_not_found", permissions, server.GetSettings());
        return GetLocalizedLanding(landingTemplateNotFoundLocations, server);
    }

    const auto carsFetchResult = driveApi->GetCarsData()->FetchInfo(offer->GetObjectId(), tx);
    R_ENSURE(carsFetchResult, HTTP_INTERNAL_SERVER_ERROR, "RentalFinishChecker::CheckImpl cannot fetch car", tx);
    const auto carsInfo = carsFetchResult.GetResult();
    R_ENSURE(carsInfo.size() == 1, HTTP_INTERNAL_SERVER_ERROR, "RentalFinishChecker::CheckImpl wrong carsInfo.size()", tx);
    const auto& car = carsInfo.begin()->second;

    TRTDeviceSnapshot snapshot = server.GetSnapshotsManager().GetSnapshot(car.GetId());
    auto carLocation = snapshot.GetLocation();
    R_ENSURE(carLocation, HTTP_INTERNAL_SERVER_ERROR, "RentalFinishChecker::CheckImp car location not found", tx);

    auto affiliatedCompanyTagDescription = NDrivematics::TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions->GetUserId(), server, tx);

    TZoneStorage::TOptions options(affiliatedCompanyTagDescription);
    options.SetForMobile(true);
    TMaybe<TVector<NDrivematics::TZone>> allCompanyZone = driveApi->GetZoneDB()->GetObjects(options, Now());

    if (!allCompanyZone || allCompanyZone->empty()) {
        auto landingTemplateNotFoundZones = GetLanding("return_location_zones_not_found", permissions, server.GetSettings());
        return GetLocalizedLanding(landingTemplateNotFoundZones, server);
    }
    NDrive::TZoneIds carInZoneIds;
    {
        NDrive::TZoneIds zoneIds;
        ForEach(allCompanyZone->begin(), allCompanyZone->end(), [&zoneIds](const NDrivematics::TZone& zone) {
            zoneIds.insert(zone.GetInternalId());
        });

        R_ENSURE(
            driveApi->GetZoneDB()->GetZoneIdsInPoint(zoneIds, carLocation->GetCoord(), std::ref(carInZoneIds), areaManager)
            , HTTP_INTERNAL_SERVER_ERROR
            , "RentalFinishChecker::CheckImp zones with car location not found"
            , tx
        );
    }

    TVector<TRentalOfferLocation> allowReturnLocationsWHithCarIn;
    TVector<TRentalOfferLocation> allowReturnLocationsWHithCarOut;
    {
        TAreaIdToArea areaMapInCarPoint;
        TAreaIdToArea areaMapOutCarPoint;
        auto addCarIn = [&areaMapInCarPoint, &requiredTagsInArea](const TString& key, const TArea& entity) -> void {
            if (requiredTagsInArea.empty() || MakeIntersection(requiredTagsInArea, entity.GetTags()).size() == requiredTagsInArea.size()) {
                areaMapInCarPoint[key] = entity;
            }
        };
        auto addCarOut = [&areaMapOutCarPoint, &requiredTagsInArea](const TString& key, const TArea& entity) -> void {
            if (requiredTagsInArea.empty() || MakeIntersection(requiredTagsInArea, entity.GetTags()).size() == requiredTagsInArea.size()) {
                areaMapOutCarPoint[key] = entity;
            }
        };
        for (const auto& zone : *allCompanyZone) {
            if (carInZoneIds.contains(zone.GetInternalId())) {
                R_ENSURE(server.GetDriveAPI()->GetAreasDB()->ForObjectsMap(addCarIn, {}, zone.GetAreaIdsPtr())
                    , {}
                    , "cannot get objects from cache"
                    , NDrive::MakeError("area.object_not_found")
                    , tx
                );
            } else {
                R_ENSURE(server.GetDriveAPI()->GetAreasDB()->ForObjectsMap(addCarOut, {}, zone.GetAreaIdsPtr())
                    , {}
                    , "cannot get objects from cache"
                    , NDrive::MakeError("area.object_not_found")
                    , tx
                );
            }
        }
        const TInternalPointContext internalContext = Default<TInternalPointContext>();
        if (rentalOffer->HasReturnLocations()) {
            auto returnLocationInArea = [&internalContext, offerReturnLocations = &rentalOffer->GetReturnLocationsRef()](const auto& areaMap, auto& returnLocation) -> void {
                for (const auto& [areaId, area] : areaMap) {
                    for (const auto& offerReturnLocation : *offerReturnLocations) {
                        if (offerReturnLocation.GetType() == ELocationType::Coord &&
                            internalContext.IsPointInternal(area.GetPolyline(), offerReturnLocation.GetCoords().front())
                            ) {
                            returnLocation.push_back(offerReturnLocation);
                        }
                    }
                }
            };
            returnLocationInArea(areaMapInCarPoint, allowReturnLocationsWHithCarIn);
            returnLocationInArea(areaMapOutCarPoint, allowReturnLocationsWHithCarOut);
        }

    }
    TMap<double, TString> lengthToLocationNameWithCarIn;
    TMap<double, TString> lengthToLocationNameWithCarOut;
    {
        auto calcsDistance = [&carLocation](const TVector<TRentalOfferLocation>& returnLocations, TMap<double, TString>& result) -> void {
            for (const auto& allowReturnLocation : returnLocations) {
                if (allowReturnLocation.GetType() == ELocationType::Coord) {
                    auto returnPoint = allowReturnLocation.GetCoords().front();
                    auto distance = carLocation->GetCoord().GetLengthTo(returnPoint);
                    result[distance] = allowReturnLocation.GetLocationName();
                }
            }
        };
        calcsDistance(allowReturnLocationsWHithCarIn, lengthToLocationNameWithCarIn);

        for (const auto& [length, locationName] : lengthToLocationNameWithCarIn) {
            if (length <= server.GetSettings().GetValueDef<double>("rental.warning.max_point_distance", maxDistanceToPoint)) {
                return NJson::JSON_NULL;
            }
        }

        if (lengthToLocationNameWithCarIn.empty() && allowReturnLocationsWHithCarOut.empty()) {
            auto landingTemplateNotFoundZones = GetLanding("return_location_zones_not_found", permissions, server.GetSettings());
            return GetLocalizedLanding(landingTemplateNotFoundZones, server);
        }

        calcsDistance(allowReturnLocationsWHithCarOut, lengthToLocationNameWithCarOut);
    }

    {
        TString NearestAdress;
        double NearestDistance = 0.;
        if (!lengthToLocationNameWithCarIn.empty()) {
            NearestAdress = lengthToLocationNameWithCarIn.begin()->second;
            NearestDistance = lengthToLocationNameWithCarIn.begin()->first;
        } else if (!allowReturnLocationsWHithCarOut.empty()) {
            NearestAdress = lengthToLocationNameWithCarOut.begin()->second;
            NearestDistance = lengthToLocationNameWithCarOut.begin()->first;
        }

        auto landingTemplate = GetLanding("return_location_not_allowed", permissions, server.GetSettings());
        landingTemplate = offer->FormDescriptionElement(landingTemplate, locale, localization);
        SubstGlobal(landingTemplate, "_NearestAdress_", NearestAdress);
        SubstGlobal(landingTemplate, "_NearestDistance_", ToString(NearestDistance));
        return GetLocalizedLanding(landingTemplate, server);
    }
}

const TString TRentalFinishLocationChecker::Name = "rental_finish_location_checker";
IWarningScreenChecker::TFactory::TRegistrator<TRentalFinishLocationChecker> TRentalFinishLocationChecker::Registrator(TRentalFinishLocationChecker::Name);
