#include "rental_finish_mileage_checker.h"
#include "rental_finish_fuel_checker.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/device_snapshot/snapshots/snapshot.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/roles/permissions.h>

#include <drive/library/cpp/searchserver/context/replier.h>

NJson::TJsonValue TRentalFinishMileageChecker::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    Y_UNUSED(server, context, permissions);
    auto locale = GetLocale();
    auto localization = server.GetLocalization();
    auto driveApi = Yensured(server.GetDriveAPI());
    const auto& cgi = context->GetCgiParameters();
    const auto& offerId = cgi.Get("session_id");
    Y_ENSURE(offerId, "Missing offer parameter");

    TAtomicSharedPtr<const ISession> session;
    R_ENSURE(driveApi->GetUserSession(permissions->GetUserId(), session, offerId, Now()), HTTP_INTERNAL_SERVER_ERROR, "cannot GetRequestUserSession");
    R_ENSURE(session, HTTP_NOT_FOUND, "no session found");
    auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(session);
    R_ENSURE(billingSession, HTTP_INTERNAL_SERVER_ERROR, "cannot cast session " << session->GetSessionId() << " to BillingSession");
    auto offer = billingSession->GetCurrentOffer();
    R_ENSURE(offer, HTTP_INTERNAL_SERVER_ERROR, "cannot find Offer in session " << session->GetSessionId());
    auto rentalOffer = std::dynamic_pointer_cast<TRentalOffer>(offer);
    R_ENSURE(rentalOffer, HTTP_INTERNAL_SERVER_ERROR, "cannot cast RentalOffer in session " << session->GetSessionId());

    auto snapshotComplilation = billingSession->MakeCompilation<TSnapshotsDiffCompilation>();
    R_ENSURE(snapshotComplilation, HTTP_INTERNAL_SERVER_ERROR, "cannot make SnapshotsDiffCompilation");
    auto snapshotDiff = snapshotComplilation->GetSnapshotsDiff(&server);
    auto startMileage = snapshotDiff ? snapshotDiff->OptionalStartMileage() : Nothing();
    auto finishMileage = snapshotDiff ? snapshotDiff->OptionalLastMileage() : Nothing();

    if (rentalOffer->HasLimitKmPerDay() && startMileage && finishMileage) {
        const auto roundedMileageStart = std::round(*startMileage);
        const auto roundedMileageFinish = std::round(*finishMileage);
        const auto rentalMileage = roundedMileageFinish - roundedMileageStart;

        if (rentalMileage > rentalOffer->GetMileageLimit()) {
            auto landingTemplate = GetLanding("mileage_limit_exceed", permissions, server.GetSettings());
            landingTemplate = offer->FormDescriptionElement(landingTemplate, locale, localization);
            SubstGlobal(landingTemplate, "_RentalMileage_", ToString(rentalMileage));
            SubstGlobal(landingTemplate, "_RentalOverrun_", ToString(rentalMileage - rentalOffer->GetMileageLimit()));
            return GetLocalizedLanding(landingTemplate, server);
        }
    }

    return NJson::JSON_NULL;
}

const TString TRentalFinishMileageChecker::Name = "rental_finish_mileage_checker";
IWarningScreenChecker::TFactory::TRegistrator<TRentalFinishMileageChecker> TRentalFinishMileageChecker::Registrator(TRentalFinishMileageChecker::Name);
