#include "session_check.h"

#include "helpers.h"

#include <drive/backend/base/server.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/sessions/manager/billing.h>

#include <util/string/cast.h>
#include <util/string/subst.h>

namespace {
    void FillSharingDebtTemplate(TString& landing, const IOffer& offer, const NDrive::IServer& server) {
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        auto optionalSharedSession = server.GetDriveDatabase().GetSessionManager().GetSession(offer.GetSharedSessionId(), session);
        R_ENSURE(optionalSharedSession, {}, "cannot GetSession", session);
        auto sharedSession = Yensured(*optionalSharedSession);
        auto sharingUserId = sharedSession->GetUserId();
        auto fetchResult = server.GetDriveAPI()->GetUsersData()->FetchInfo(sharingUserId, session);
        auto sharingUserData = fetchResult.GetResultPtr(sharingUserId);
        auto sharingUserName = sharingUserData ? sharingUserData->GetShortName() : "";
        SubstGlobal(landing, "_SharingUserName_", sharingUserName);
    }

    void FillDebtTemplate(TString& landing, ELocalization locale, TUserPermissions::TConstPtr permissions, const NDrive::IServer& server) {
        TMaybe<ui32> debt;
        if (server.GetDriveAPI()->HasBillingManager()) {
            const auto& billingManager = server.GetDriveAPI()->GetBillingManager();
            auto session = billingManager.BuildSession(true);
            debt = billingManager.GetDebt(permissions->GetUserId(), session);
            if (!debt) {
                session.ClearErrors();
            }
        }
        const TString debtStr = server.GetLocalization()->FormatPrice(locale, debt ? *debt : 0);
        SubstGlobal(landing, "_Debt_", debtStr);
    }

    void FillDepositTemplate(TString& landing, ELocalization locale, const IOffer& offer, const ILocalization& localization) {
        SubstGlobal(landing, "_Deposit_",
                    localization.FormatPrice(locale, offer.GetDeposit()));
    }
}  // namespace

const TString TSessionDebtCheck::Name = "session_debt_check";
IWarningScreenChecker::TFactory::TRegistrator<TSessionDebtCheck> TSessionDebtCheck::Registrator(TSessionDebtCheck::Name);

NJson::TJsonValue TSessionDebtCheck::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    auto compilation = GetBillingSessionCompilationSafe(server, context, permissions);
    auto offer = compilation.GetCurrentOffer();
    Y_ENSURE(offer, "No offer in session " << compilation.GetSessionId());

    if (!compilation.GetLocalEvents().empty() && compilation.GetLocalEvents().back().GetTagName() == TChargableTag::Sharing) {
        // Ignored for DRIVEBACK-3776
        return NJson::JSON_NULL;
    }

    auto res = EDriveSessionResult::Success;
    {
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        res = offer->CheckSession(*permissions, session, &server, false);
        session.ClearErrors();
    }
    TString landing;
    if (offer->GetSharedSessionId() && (res == EDriveSessionResult::PaymentRequired || res == EDriveSessionResult::DepositFails)) {
        landing = GetLanding("sharing_debt", permissions, server.GetSettings());
        FillSharingDebtTemplate(landing, *offer, server);
    } else if (res == EDriveSessionResult::PaymentRequired) {
        landing = GetLanding("debt", permissions, server.GetSettings());
        FillDebtTemplate(landing, GetLocale(), permissions, server);
    } else if (res == EDriveSessionResult::DepositFails) {
        landing = GetLanding("deposit_fails", permissions, server.GetSettings());
        FillDepositTemplate(landing, GetLocale(), *offer, *server.GetLocalization());
    } else {
        return NJson::JSON_NULL;
    }
    return GetLocalizedLanding(landing, server);
}


const TString TActiveSessionCheck::Name = "active_session_check";
IWarningScreenChecker::TFactory::TRegistrator<TActiveSessionCheck> TActiveSessionCheck::Registrator(TActiveSessionCheck::Name);

NJson::TJsonValue TActiveSessionCheck::CheckImpl(const NDrive::IServer& server, const TAtomicSharedPtr<IReplyContext> /* context */, TIntrusiveConstPtr<TUserPermissions> permissions) const {
    TVector<TAtomicSharedPtr<const ISession>> userSessions;
    Y_ENSURE(server.GetDriveAPI()->GetCurrentUserSessions(permissions->GetUserId(), userSessions, TInstant::Zero()), "fail to fetch sessions");
    if (userSessions) {
        return GetLocalizedLanding("main", permissions, server);
    }
    return NJson::JSON_NULL;
}
