#include "validated_images.h"

#include "helpers.h"

#include <drive/backend/database/drive/landing.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/images/database.h>
#include <drive/backend/logging/evlog.h>

namespace {
    const TString SettingName = "vics_count";
}

NJson::TJsonValue TValidatedImagesCheck::CheckImpl(const NDrive::IServer& server, const IReplyContext::TPtr context, TUserPermissions::TConstPtr permissions) const {
    auto session = GetSession(server, context, permissions);
    if (!session) {
        return {};
    }

    auto userSettingsTagName = server.GetSettings().GetValue<TString>(NDrive::UserSettingsTagNameSetting).GetOrElse({});
    ui32 shows = 0;
    {
        NDrive::TInfoEntitySession tx;
        auto expectedShows = permissions->GetUserSetting().GetValueFromCache(userSettingsTagName, SettingName, tx);
        R_ENSURE(expectedShows || expectedShows.GetError() == NotFound, {}, "cannot GetValueFromCache", tx);
        if (expectedShows && !TryFromString(*expectedShows, shows)) {
            auto evlog = NDrive::GetThreadEventLogger();
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder
                    ("event", "BadUserSettingValue")
                    ("name", SettingName)
                    ("value", *expectedShows)
                );
            }
        }
    }

    ui32 showsLimit = 1;
    if (shows >= showsLimit) {
        return {};
    }

    auto validatedImages = server.GetDriveDatabase().GetImageStorage().GetCachedValidatedImages(session->GetObjectId());
    if (!validatedImages->Count()) {
        return {};
    }

    auto landingId = GetLanding("main.landing_id", permissions, server.GetSettings());
    auto landing = server.GetDriveDatabase().GetLandingManager().GetObject(landingId);
    R_ENSURE(landing, HTTP_INTERNAL_SERVER_ERROR, "cannot find landing " << landingId);

    shows += 1;
    {
        auto tx = server.GetDriveDatabase().GetTagsManager().GetUserTags().BuildTx<NSQL::Writable>();
        bool valueSet = permissions->GetUserSetting().SetValue(userSettingsTagName, SettingName, ToString(shows), permissions->GetUserId(), tx);
        R_ENSURE(valueSet, {}, "cannot SetValue", tx);
        R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    }
    auto locale = GetLocale();
    return landing->GetPublicReport(locale);
}

const TString TValidatedImagesCheck::Name = "validated_images_check";
IWarningScreenChecker::TFactory::TRegistrator<TValidatedImagesCheck> TValidatedImagesCheck::Registrator(TValidatedImagesCheck::Name);
