#include "callback_persdata.h"

#include <drive/backend/data/user_tags.h>
#include <drive/backend/registrar/ifaces.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/user_document_photos/manager.h>

namespace NYangAssignment {
    void FillFraudReasons(TYangDocumentVerificationAssignment& assignment, const NJson::TJsonValue& verificationStatuses) {
        if (verificationStatuses.Has("fraud_reasons") && verificationStatuses["fraud_reasons"].IsArray()) {
            TVector<TString> fraudReasons;
            for (auto&& fr : verificationStatuses["fraud_reasons"].GetArray()) {
                if (!fr.IsString()) {
                    continue;
                }
                fraudReasons.push_back(fr.GetString());
            }
            assignment.SetFraudReasons(std::move(fraudReasons));
        }
    }
};

TPrefilledDataAcquisionCallback::TPrefilledDataAcquisionCallback(const NDrive::IServer* server, IServerReportBuilder::TPtr report, const bool passportBioStatus, const bool passportRegStatus, const bool passportSelfieStatus, const bool licenseBackStatus, const bool licenseFrontStatus, const TDriveUserData& userData)
    : IPrivateDataAcquisitionCallback(2)
    , Server(server)
    , Report(report)
    , PassportBioStatus(passportBioStatus)
    , PassportRegStatus(passportRegStatus)
    , LicenseBackStatus(licenseBackStatus)
    , LicenseFrontStatus(licenseFrontStatus)
    , UserData(userData)
{
    TUserPassportData defaultPassportData;
    TUserDrivingLicenseData defaultDrivingLicenseData;
    ResultReport["license_back"] = defaultDrivingLicenseData.SerializeBackToYang(LicenseBackStatus);
    ResultReport["license_front"] = defaultDrivingLicenseData.SerializeFrontToYang(LicenseFrontStatus);
    ResultReport["passport_biographical"] = defaultPassportData.SerializeBioToYang(PassportBioStatus);
    ResultReport["passport_registration"] = defaultPassportData.SerializeRegToYang(PassportRegStatus);
    ResultReport["passport_selfie"] = NJson::JSON_MAP;
    ResultReport["passport_selfie"]["is_verified"] = passportSelfieStatus;
}

void TPrefilledDataAcquisionCallback::DoOnPassportReceipt(const TString& /*revision*/, const TUserPassportData& passport) {
    TGuard<TMutex> g(Mutex);
    ResultReport["passport_biographical"] = passport.SerializeBioToYang(PassportBioStatus);
    ResultReport["passport_registration"] = passport.SerializeRegToYang(PassportRegStatus);

    if (UserData.GetPassportDatasyncRevision().StartsWith("recognized-")) {
        auto photosData = Server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(
            UserData.GetUserId(), {
                NUserDocument::EType::PassportBiographical
            }
        );

        if (photosData.contains(NUserDocument::EType::PassportBiographical) && photosData[NUserDocument::EType::PassportBiographical].GetVerificationStatus() == NUserDocument::EVerificationStatus::NotYetProcessed && photosData[NUserDocument::EType::PassportBiographical].HasRecognizerMeta()) {
            ResultReport["passport_biographical"]["confidence"] = photosData[NUserDocument::EType::PassportBiographical].GetRecognizerMetaUnsafe().SerializePassportBiographical();
        }
    }
}

void TPrefilledDataAcquisionCallback::DoOnDrivingLicenseReceipt(const TString& /*revision*/, const TUserDrivingLicenseData& drivingLicense) {
    TGuard<TMutex> g(Mutex);
    ResultReport["license_back"] = drivingLicense.SerializeBackToYang(LicenseBackStatus);
    ResultReport["license_back"]["data"]["has_at_mark"] = UserData.GetHasATMark();
    ResultReport["license_front"] = drivingLicense.SerializeFrontToYang(LicenseFrontStatus);

    if (UserData.GetDrivingLicenseDatasyncRevision().StartsWith("recognized-")) {
        auto photosData = Server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(
            UserData.GetUserId(), {
                NUserDocument::EType::LicenseBack,
                NUserDocument::EType::LicenseFront
            }
        );

        if (photosData.contains(NUserDocument::EType::LicenseBack) && photosData[NUserDocument::EType::LicenseBack].GetVerificationStatus() == NUserDocument::EVerificationStatus::NotYetProcessed && photosData[NUserDocument::EType::LicenseBack].HasRecognizerMeta()) {
            ResultReport["license_back"]["confidence"] = photosData[NUserDocument::EType::LicenseBack].GetRecognizerMetaUnsafe().SerializeDrivingLicenseBack();
        }

        if (photosData.contains(NUserDocument::EType::LicenseFront) && photosData[NUserDocument::EType::LicenseFront].GetVerificationStatus() == NUserDocument::EVerificationStatus::NotYetProcessed && photosData[NUserDocument::EType::LicenseFront].HasRecognizerMeta()) {
            ResultReport["license_front"]["confidence"] = photosData[NUserDocument::EType::LicenseFront].GetRecognizerMetaUnsafe().SerializeDrivingLicenseFront();
        }
    }
}

void TPrefilledDataAcquisionCallback::ProcessAllResponses() {
    TJsonReport::TGuard g(Report, HTTP_OK);
    TJsonReport& r = g.MutableReport();

    r.SetExternalReport(std::move(ResultReport));
    g.SetCode(HTTP_OK);
}
