#pragma once

#include <drive/backend/processors/service_app/config.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/users/user_documents.h>
#include <drive/backend/user_document_photos/manager.h>

#include <drive/backend/processor/params_processor.h>
#include <drive/backend/processors/users/processor.h>

template <class THandler, class TBaseHandler>
class IYangProcessorWrapper : public TAppCommonProcessor<THandler, TEmptyConfig, TBaseHandler> {
private:
    using TBase = TAppCommonProcessor<THandler, TEmptyConfig, TBaseHandler>;

public:
    using TBase::TBase;

    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override {
        const TCgiParameters& cgi = TBase::Context->GetCgiParameters();
        auto secretId = TBase::GetString(requestData, "secretId", false);
        if (!secretId) {
            secretId = TBase::GetString(cgi, "secretId", true);
        }
        R_ENSURE(
            secretId,
            TBase::ConfigHttpStatus.EmptyRequestStatus,
            "missing secretId"
        );

        TYangDocumentVerificationAssignment assignment;
        {
            auto session = TBase::template BuildTx<NSQL::ReadOnly>();
            R_ENSURE(
                TBase::DriveApi->HasDocumentPhotosManager(),
                TBase::ConfigHttpStatus.UnknownErrorStatus,
                "user photo manager not defined"
            );
            const auto& photoManager = TBase::DriveApi->GetDocumentPhotosManager();
            auto fetchAssignmentResult = photoManager.GetDocumentVerificationAssignments().FetchInfo(secretId, session);
            R_ENSURE(
                fetchAssignmentResult.size() == 1,
                TBase::ConfigHttpStatus.UnknownErrorStatus,
                "can't restore assignment by secretId"
            );
            assignment = fetchAssignmentResult.begin()->second;
        }
        NJson::TJsonValue enrichedRequestData = EnrichRequestData(requestData, assignment);

        TBaseHandler::ProcessServiceRequest(g, permissions, enrichedRequestData);
    }

protected:
    virtual NJson::TJsonValue EnrichRequestData(const NJson::TJsonValue& requestData, const TYangDocumentVerificationAssignment& assignment) const {
        NJson::TJsonValue enrichedRequestData = requestData;
        enrichedRequestData["user_id"] = assignment.GetUserId();
        return enrichedRequestData;
    }
};

class TYangUserInfoProcessor : public IYangProcessorWrapper<TYangUserInfoProcessor, TUserInfoProcessorBase> {
private:
    using TBase = IYangProcessorWrapper<TYangUserInfoProcessor, TUserInfoProcessorBase>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "yang_user_info";
    }
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;
};

class TYangUserDocumentDataProcessor : public IYangProcessorWrapper<TYangUserDocumentDataProcessor, TUserDocumentDataProcessorBase> {
private:
    using TBase = IYangProcessorWrapper<TYangUserDocumentDataProcessor, TUserDocumentDataProcessorBase>;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "yang_user_document_data";
    }
    virtual void ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& requestData) override;

protected:
    NJson::TJsonValue EnrichRequestData(const NJson::TJsonValue& requestData, const TYangDocumentVerificationAssignment& assignment) const override;
};

