#include "dispatcher.h"

void TYangDispatcher::ProcessRequest(TJsonReport::TGuard& g, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString secretId;
    TString assignmentId;
    if (requestData.Has("secretId") && requestData["secretId"].IsString() &&
        requestData.Has("assignmentId") && requestData["assignmentId"].IsString())
    {
        secretId = requestData["secretId"].GetString();
        assignmentId = requestData["assignmentId"].GetString();
    } else {
        secretId = GetString(cgi, "secretId");
        assignmentId = GetString(cgi, "assignmentId");
    }
    R_ENSURE(
        secretId && assignmentId,
        ConfigHttpStatus.EmptyRequestStatus,
        "missing secretId or assignmentId"
    );


    auto session = Server->GetDriveAPI()->template BuildTx<NSQL::Writable>();

    R_ENSURE(
        DriveApi->HasDocumentPhotosManager(),
        ConfigHttpStatus.UnknownErrorStatus,
        "user photo manager not defined"
    );
    const auto& photoManager = DriveApi->GetDocumentPhotosManager();
    auto fetchAssignmentResult = photoManager.GetDocumentVerificationAssignments().FetchInfo(secretId, session);
    R_ENSURE(
        fetchAssignmentResult.size() == 1,
        ConfigHttpStatus.UnknownErrorStatus,
        "can't restore assignment by secretId"
    );

    auto assignment = fetchAssignmentResult.begin()->second;
    auto userId = assignment.GetUserId();

    auto eventLogState = NDrive::TEventLog::CaptureState();
    auto operatorUserId = eventLogState.UserId ? eventLogState.UserId : "yang";
    NDrive::TEventLog::TUserIdGuard userIdGuard(userId);

    auto fetchUserResult = DriveApi->GetUsersData()->FetchInfo(userId, session);
    R_ENSURE(fetchUserResult, {}, "cannot FetchInfo " << userId, session);

    auto queueClient = DriveApi->GetDatasyncQueueClient();
    if (!queueClient) {
        g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
            .AddInfo("session_info", session.GetReport())
            .SetErrorCode("datasync_queue_client_unavailable")
        );
        return;
    }

    if (!queueClient->Push(assignmentId, session) || !session.Commit()) {
        g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
            .AddInfo("session_info", session.GetReport())
            .SetErrorCode("cannot_push_assignment_into_the_queue")
        );
        return;
    }

    g.SetCode(HTTP_OK);
}
