#include "processor.h"

#include "callback_persdata.h"

#include <drive/backend/registrar/ifaces.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/user_document_photos/manager.h>
#include <drive/backend/auth/blackbox2/auth.h>

#include <util/generic/adaptor.h>

void TYangProxyCommonProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& jsonValue) {
    Y_UNUSED(permissions);
    try {
        ProcessRequest(g, jsonValue);
    } catch (const std::exception& e) {
        g.AddReportElement("fatal", FormatExc(e));
    }
}

void TYangUserTextdataProcessorTraits::ProcessGetNewReport(TJsonReport::TGuard& g, const TDriveUserData& user, NDrive::TEntitySession& session) {
    const auto& photoManager = DriveApi->GetDocumentPhotosManager();
    auto userId = user.GetUserId();
    const auto& uid = user.GetUid();
    R_ENSURE(uid, ConfigHttpStatus.UnknownErrorStatus, "malformed uid for user " << userId << ": " << user.GetUid());

    const auto& drivingLicenseRevision = user.GetDrivingLicenseDatasyncRevision();
    const auto& passportRevision = user.GetPassportDatasyncRevision();

    auto allAssignments = photoManager.GetDocumentVerificationAssignments().GetAllAssignmentsForUser(userId);
    std::reverse(allAssignments.begin(), allAssignments.end());
    std::sort(allAssignments.begin(), allAssignments.end(), [](const TYangDocumentVerificationAssignment& lhs, const TYangDocumentVerificationAssignment& rhs) {
        return lhs.GetCreatedAt() < rhs.GetCreatedAt();
    });

    TVector<TString> lbIds;
    TVector<TString> lfIds;
    TVector<TString> lsIds;
    TVector<TString> pbIds;
    TVector<TString> prIds;
    TVector<TString> psIds;
    auto appendId = [](const TString& id, TVector<TString>& ids) {
        if (id && ids.size() < 2) {
            auto p = std::find(ids.begin(), ids.end(), id);
            if (p == ids.end()) {
                ids.push_back(id);
            }
        }
    };
    for (auto&& assignment : Reversed(allAssignments)) {
        appendId(assignment.GetLicenseBackId(), lbIds);
        appendId(assignment.GetLicenseFrontId(), lfIds);
        appendId(assignment.GetLicenseSelfieId(), lsIds);
        appendId(assignment.GetPassportBiographicalId(), pbIds);
        appendId(assignment.GetPassportRegistrationId(), prIds);
        appendId(assignment.GetPassportSelfieId(), psIds);
    }
    g.AddEvent(NJson::TMapBuilder
        ("event", "GetPhotoIds")
        ("lb", NJson::ToJson(lbIds))
        ("lf", NJson::ToJson(lfIds))
        ("ls", NJson::ToJson(lsIds))
        ("pb", NJson::ToJson(pbIds))
        ("pr", NJson::ToJson(prIds))
        ("ps", NJson::ToJson(psIds))
    );

    auto photoIds = pbIds;
    photoIds.insert(photoIds.end(), lbIds.begin(), lbIds.end());
    photoIds.insert(photoIds.end(), lfIds.begin(), lfIds.end());
    photoIds.insert(photoIds.end(), lsIds.begin(), lsIds.end());
    photoIds.insert(photoIds.end(), prIds.begin(), prIds.end());
    photoIds.insert(photoIds.end(), psIds.begin(), psIds.end());
    auto photoFetchResult = photoManager.GetUserPhotosDB().FetchInfo(photoIds, session);
    R_ENSURE(photoFetchResult, {}, "cannot FetchInfo", session);

    auto recentPhotos = TMap<NUserDocument::EType, TUserDocumentPhoto>();
    if (drivingLicenseRevision.StartsWith("recognized-")) {
        auto recentDrivingLicensePhotos = photoManager.GetUserPhotosDB().GetTypeToRecentPhotoMapping(user.GetUserId(), {
            NUserDocument::EType::LicenseBack,
            NUserDocument::EType::LicenseFront,
        });
        g.AddEvent(NJson::TMapBuilder
            ("event", "GetRecentDrivingLicensePhotos")
            ("photos", NJson::ToJson(NJson::Dictionary(recentDrivingLicensePhotos)))
        );
        recentPhotos = std::move(recentDrivingLicensePhotos);
    }
    if (passportRevision.StartsWith("recognized-")) {
        auto recentPassportPhotos = photoManager.GetUserPhotosDB().GetTypeToRecentPhotoMapping(user.GetUserId(), {
            NUserDocument::EType::PassportBiographical,
        });
        g.AddEvent(NJson::TMapBuilder
            ("event", "GetRecentPassportPhotos")
            ("photos", NJson::ToJson(NJson::Dictionary(recentPassportPhotos)))
        );
        recentPhotos.insert(recentPassportPhotos.begin(), recentPassportPhotos.end());
    }

    auto getStatus = [&](const TVector<TString>& ids) -> TMaybe<bool> {
        for (auto&& id : ids) {
            auto photo = photoFetchResult.GetResultPtr(id);
            if (!photo) {
                continue;
            }
            auto status = photo->GetVerificationStatus();
            if (status == NUserDocument::EVerificationStatus::NotYetProcessed) {
                continue;
            }
            return status == NUserDocument::EVerificationStatus::Ok;
        }
        return {};
    };

    auto lbStatus = getStatus(lbIds);
    auto lfStatus = getStatus(lfIds);
    auto lsStatus = getStatus(lbIds);
    auto pbStatus = getStatus(pbIds);
    auto prStatus = getStatus(prIds);
    auto psStatus = getStatus(psIds);

    const auto& privateDataClient = DriveApi->GetPrivateDataClient();
    auto drivingLicense = drivingLicenseRevision ? privateDataClient.GetDrivingLicense(user, drivingLicenseRevision) : NThreading::Uninitialized;
    auto passport = passportRevision ? privateDataClient.GetPassport(user, passportRevision) : NThreading::Uninitialized;
    auto initializedDrivingLicense = NThreading::Initialize(drivingLicense).IgnoreResult();
    auto initializedPassport = NThreading::Initialize(passport).IgnoreResult();
    auto waiter = NThreading::WaitAll(initializedDrivingLicense, initializedPassport);
    auto eventLogState = NDrive::TEventLog::CaptureState();
    auto report = g.GetReport();
    waiter.Subscribe([
        eventLogState = std::move(eventLogState),
        drivingLicense = std::move(drivingLicense),
        passport = std::move(passport),
        recentPhotos = std::move(recentPhotos),
        report = std::move(report),
        user = std::move(user),
        lbStatus,
        lfStatus,
        lsStatus,
        pbStatus,
        prStatus,
        psStatus
    ](const NThreading::TFuture<void>& /*w*/) {
        NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
        TJsonReport::TGuard g(report, HTTP_OK);
        if (drivingLicense.Initialized()) {
            if (drivingLicense.HasValue()) {
                auto licenseBack = drivingLicense.GetValue().SerializeBackToYang(lbStatus.GetOrElse(false));
                auto licenseFront = drivingLicense.GetValue().SerializeFrontToYang(lfStatus.GetOrElse(false));
                licenseBack["data"]["has_at_mark"] = user.GetHasATMark();
                auto licenseBackPhoto = recentPhotos.FindPtr(NUserDocument::EType::LicenseBack);
                if (licenseBackPhoto && licenseBackPhoto->HasRecognizerMeta() && licenseBackPhoto->GetVerificationStatus() == NUserDocument::EVerificationStatus::NotYetProcessed) {
                    licenseBack["confidence"] = licenseBackPhoto->GetRecognizerMetaRef().SerializeDrivingLicenseBack();
                }
                auto licenseFrontPhoto = recentPhotos.FindPtr(NUserDocument::EType::LicenseFront);
                if (licenseFrontPhoto && licenseFrontPhoto->HasRecognizerMeta() && licenseFrontPhoto->GetVerificationStatus() == NUserDocument::EVerificationStatus::NotYetProcessed) {
                    licenseFront["confidence"] = licenseFrontPhoto->GetRecognizerMetaRef().SerializeDrivingLicenseFront();
                }
                g.AddReportElement("license_back", std::move(licenseBack));
                g.AddReportElement("license_front", std::move(licenseFront));
            } else {
                g.AddReportElement("license_error", NThreading::GetExceptionInfo(drivingLicense));
            }
        }
        if (passport.Initialized()) {
            if (passport.HasValue()) {
                auto passportBiographical = passport.GetValue().SerializeBioToYang(pbStatus.GetOrElse(false));
                auto passportRegistration = passport.GetValue().SerializeRegToYang(prStatus.GetOrElse(false));
                auto passportBiographicalPhoto = recentPhotos.FindPtr(NUserDocument::EType::PassportBiographical);
                if (passportBiographicalPhoto && passportBiographicalPhoto->HasRecognizerMeta() && passportBiographicalPhoto->GetVerificationStatus() == NUserDocument::EVerificationStatus::NotYetProcessed) {
                    passportBiographical["confidence"] = passportBiographicalPhoto->GetRecognizerMetaRef().SerializePassportBiographical();
                }
                g.AddReportElement("passport_biographical", std::move(passportBiographical));
                g.AddReportElement("passport_registration", std::move(passportRegistration));
            } else {
                g.AddReportElement("passport_error", NThreading::GetExceptionInfo(passport));
            }
        }
        {
            NJson::TJsonValue passportSelfie;
            passportSelfie["is_verified"] = psStatus.GetOrElse(false);
            g.AddReportElement("passport_selfie", std::move(passportSelfie));
        }
        {
            NJson::TJsonValue licenseSelfie;
            licenseSelfie["is_verified"] = NJson::ToJson(lsStatus);
            g.AddReportElement("license_selfie", std::move(licenseSelfie));
        }
    });
    g.Release();
}

void TYangUserTextdataProcessorTraits::ProcessGet(TJsonReport::TGuard& g, const TDriveUserData& user, NDrive::TEntitySession& session) {
    auto newReport = GetHandlerSetting<bool>("new_report").GetOrElse(true);
    if (newReport) {
        ProcessGetNewReport(g, user, session);
        return;
    }

    const auto& photoManager = DriveApi->GetDocumentPhotosManager();
    auto userId = user.GetUserId();
    const auto& uid = user.GetUid();
    R_ENSURE(uid, ConfigHttpStatus.UnknownErrorStatus, "malformed uid for user " << userId << ": " << user.GetUid());

    if (!user.GetPassportDatasyncRevision() || !user.GetDrivingLicenseDatasyncRevision()) {
        NJson::TJsonValue result;
        TUserPassportData defaultPassportData;
        TUserDrivingLicenseData defaultDrivingLicenseData;
        result["license_back"] = defaultDrivingLicenseData.SerializeBackToYang(false);
        result["license_front"] = defaultDrivingLicenseData.SerializeFrontToYang(false);
        result["passport_biographical"] = defaultPassportData.SerializeBioToYang(false);
        result["passport_registration"] = defaultPassportData.SerializeRegToYang(false);
        result["passport_selfie"] = NJson::JSON_MAP;
        result["passport_selfie"]["is_verified"] = false;
        g.MutableReport().SetExternalReport(std::move(result));
        g.SetCode(HTTP_OK);
        return;
    }

    TSet<NUserDocument::EType> documentTypes = {
        NUserDocument::PassportBiographical,
        NUserDocument::PassportRegistration,
        NUserDocument::PassportSelfie,
        NUserDocument::LicenseBack,
        NUserDocument::LicenseFront,
    };
    auto allAssignments = photoManager.GetDocumentVerificationAssignments().GetAllAssignments({userId}, documentTypes);
    std::reverse(allAssignments.begin(), allAssignments.end());
    std::sort(allAssignments.begin(), allAssignments.end(), [](const TYangDocumentVerificationAssignment& lhs, const TYangDocumentVerificationAssignment& rhs) {
        return lhs.GetCreatedAt() < rhs.GetCreatedAt();
    });

    bool lbStatus = false;
    bool lfStatus = false;
    bool pbStatus = false;
    bool prStatus = false;
    bool psStatus = false;
    if (allAssignments.size() > 1) {
        auto prevAssignment = allAssignments[allAssignments.size() - 2];
        TSet<TString> photoIds = TSet<TString>(
            {
                prevAssignment.GetPassportBiographicalId(),
                prevAssignment.GetPassportRegistrationId(),
                prevAssignment.GetPassportSelfieId(),
                prevAssignment.GetLicenseBackId(),
                prevAssignment.GetLicenseFrontId()
            }
        );
        auto fetchPhotos = photoManager.GetUserPhotosDB().FetchInfo(photoIds, session);
        const auto& fetchPhotosResult = fetchPhotos.GetResult();
        R_ENSURE(fetchPhotosResult.size() == 5, ConfigHttpStatus.UnknownErrorStatus, "can't restore user by photoId");

        pbStatus = fetchPhotosResult.at(prevAssignment.GetPassportBiographicalId()).GetVerificationStatus() == NUserDocument::EVerificationStatus::Ok;
        prStatus = fetchPhotosResult.at(prevAssignment.GetPassportRegistrationId()).GetVerificationStatus() == NUserDocument::EVerificationStatus::Ok;
        psStatus = fetchPhotosResult.at(prevAssignment.GetPassportSelfieId()).GetVerificationStatus() == NUserDocument::EVerificationStatus::Ok;
        lbStatus = fetchPhotosResult.at(prevAssignment.GetLicenseBackId()).GetVerificationStatus() == NUserDocument::EVerificationStatus::Ok;
        lfStatus = fetchPhotosResult.at(prevAssignment.GetLicenseFrontId()).GetVerificationStatus() == NUserDocument::EVerificationStatus::Ok;
    }

    auto callbackTask = MakeAtomicShared<TPrefilledDataAcquisionCallback>(Server, g.GetReport(), pbStatus, prStatus, psStatus, lbStatus, lfStatus, user);
    DriveApi->GetPrivateDataClient().GetPassport(user, user.GetPassportDatasyncRevision(), callbackTask);
    DriveApi->GetPrivateDataClient().GetDrivingLicense(user, user.GetDrivingLicenseDatasyncRevision(), callbackTask);
    g.Release();
}

void TYangUserTextdataProcessorTraits::FillDocumentsData(TJsonReport::TGuard& g, const NJson::TJsonValue& requestData, const TString& assignmentId, TDriveUserData& user, TMaybe<TUserPassportData>& passportData,  TMaybe<TUserDrivingLicenseData>& drivingLicenseData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();

    auto userId = user.GetUserId();

    bool strict = GetHandlerSetting<bool>("strict").GetOrElse(false);
    bool patch = GetValue<bool>(cgi, "patch", false).GetOrElse(false);
    bool setDocumentHash = GetHandlerSetting<bool>("set_document_hash").GetOrElse(true);

    const auto& uid = user.GetUid();
    R_ENSURE(uid, ConfigHttpStatus.UnknownErrorStatus, "malformed uid for user " << userId << ": " << user.GetUid());

    NJson::TJsonValue passportBiographical = requestData["passport_biographical"]["data"];
    NJson::TJsonValue passportRegistration = requestData["passport_registration"]["data"];
    NJson::TJsonValue licenseBack = requestData["license_back"]["data"];
    NJson::TJsonValue licenseFront = requestData["license_front"]["data"];

    if (licenseBack.IsMap() && licenseBack["has_at_mark"].IsBoolean()) {
        user.SetHasATMark(licenseBack["has_at_mark"].GetBoolean());
    }

    if (patch) {
        NThreading::TFuture<TUserPassportData> asyncPassportData;
        NThreading::TFuture<TUserDrivingLicenseData> asyncLicenseData;
        if (user.GetDrivingLicenseDatasyncRevision() && (licenseBack.IsDefined() || licenseFront.IsDefined())) {
            asyncLicenseData = DriveApi->GetPrivateDataClient().GetDrivingLicense(user, user.GetDrivingLicenseDatasyncRevision());
        }
        if (user.GetPassportDatasyncRevision() && (passportBiographical.IsDefined() || passportRegistration.IsDefined())) {
            asyncPassportData = DriveApi->GetPrivateDataClient().GetPassport(user, user.GetPassportDatasyncRevision());
        }
        if (asyncLicenseData.Initialized()) {
            auto eg = g.BuildEventGuard("WaitAsyncLicenseData");
            R_ENSURE(asyncLicenseData.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "DataSync LicenseData wait timeout");
            R_ENSURE(asyncLicenseData.HasValue(), ConfigHttpStatus.UnknownErrorStatus, "DataSync LicenseData exception: " << NThreading::GetExceptionMessage(asyncLicenseData));
            drivingLicenseData = asyncLicenseData.ExtractValue();
        }
        if (asyncPassportData.Initialized()) {
            auto eg = g.BuildEventGuard("WaitAsyncPassportData");
            R_ENSURE(asyncPassportData.Wait(Context->GetRequestDeadline()), ConfigHttpStatus.TimeoutStatus, "DataSync PassportData wait timeout");
            R_ENSURE(asyncPassportData.HasValue(), ConfigHttpStatus.UnknownErrorStatus, "DataSync PassportData exception: " << NThreading::GetExceptionMessage(asyncPassportData));
            passportData = asyncPassportData.ExtractValue();
        }

        TMessagesCollector errors;
        if (licenseBack.IsDefined() || licenseFront.IsDefined()) {
            TUserDrivingLicenseData drivingLicenseDataPatch;
            R_ENSURE(
                drivingLicenseDataPatch.ParseFromYang(licenseBack, licenseFront, errors, patch),
                ConfigHttpStatus.UserErrorState,
                "cannot ParseFromYang license data " + errors.GetStringReport()
            );
            if (!drivingLicenseData) {
                drivingLicenseData = drivingLicenseDataPatch;
            } else {
                drivingLicenseData->Patch(drivingLicenseDataPatch);
            }
            user.SetDrivingLicenseDatasyncRevision(assignmentId);
        }

        if (passportBiographical.IsDefined() || passportRegistration.IsDefined()) {
            TUserPassportData passportDataPatch;
            R_ENSURE(
                passportDataPatch.ParseFromYang(passportBiographical, passportRegistration, errors, patch),
                ConfigHttpStatus.UserErrorState,
                "cannot ParseFromYang passport data " + errors.GetStringReport()
            );
            if (!passportData) {
                passportData = passportDataPatch;
            } else {
                passportData->Patch(passportDataPatch);
            }
            user.SetPassportDatasyncRevision(assignmentId);
        }
    } else {
        TMessagesCollector errors;
        passportData.ConstructInPlace();
        R_ENSURE(
            passportData->ParseFromYang(passportBiographical, passportRegistration, errors, patch) || !strict,
            ConfigHttpStatus.UserErrorState,
            "cannot ParseFromYang passport data " + errors.GetStringReport()
        );
        drivingLicenseData.ConstructInPlace();
        R_ENSURE(
            drivingLicenseData->ParseFromYang(licenseBack, licenseFront, errors, patch) || !strict,
            ConfigHttpStatus.UserErrorState,
            "cannot ParseFromYang license data " + errors.GetStringReport()
        );
        user.SetPassportDatasyncRevision(assignmentId);
        user.SetDrivingLicenseDatasyncRevision(assignmentId);
    }

    auto userRegistrationManager = Server->GetUserRegistrationManager();
    if (patch && setDocumentHash && passportData && passportData->HasBirthDate()) {
        R_ENSURE(userRegistrationManager, ConfigHttpStatus.UnknownErrorStatus, "UserRegistrationManager is missing");
        user.SetPassportNamesHash(userRegistrationManager->GetDocumentNumberHash(passportData->GetNamesForHash()));
    }
    if (patch && setDocumentHash && passportData && passportData->GetNumber()) {
        R_ENSURE(userRegistrationManager, ConfigHttpStatus.UnknownErrorStatus, "UserRegistrationManager is missing");
        user.SetPassportNumberHash(userRegistrationManager->GetDocumentNumberHash(passportData->GetNumber()));
    }
    if (patch && setDocumentHash && drivingLicenseData && drivingLicenseData->GetNumber()) {
        R_ENSURE(userRegistrationManager, ConfigHttpStatus.UnknownErrorStatus, "UserRegistrationManager is missing");
        user.SetDrivingLicenseNumberHash(userRegistrationManager->GetDocumentNumberHash(drivingLicenseData->GetNumber()));
    }
    if (patch) {
        if (passportData && passportData->HasFirstName()) {
            user.SetFirstName(NRegistrarUtil::ToTitle(passportData->GetFirstName()));
            user.SetLastName(NRegistrarUtil::ToTitle(passportData->GetLastName()));
            user.SetPName(NRegistrarUtil::ToTitle(passportData->GetMiddleName()));
        } else if (drivingLicenseData && drivingLicenseData->HasFirstName()) {
            user.SetFirstName(NRegistrarUtil::ToTitle(drivingLicenseData->GetFirstName()));
            user.SetLastName(NRegistrarUtil::ToTitle(drivingLicenseData->GetLastName()));
            user.SetPName(NRegistrarUtil::ToTitle(drivingLicenseData->GetMiddleName()));
        }
    }
}

void TYangUserTextdataProcessor::ProcessRequest(TJsonReport::TGuard& g, const NJson::TJsonValue& requestData) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    TString secretId;
    TString assignmentId;
    if (requestData.Has("secretId") && requestData["secretId"].IsString() && requestData.Has("assignmentId") && requestData["assignmentId"].IsString()) {
        secretId = requestData["secretId"].GetString();
        assignmentId = requestData["assignmentId"].GetString();
    } else {
        secretId = GetString(cgi, "secretId");
        assignmentId = GetString(cgi, "assignmentId");
    }

    auto session = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();

    // get userId by secretId
    R_ENSURE(DriveApi->HasDocumentPhotosManager(), ConfigHttpStatus.UnknownErrorStatus, "user photo manager not defined");
    const auto& photoManager = DriveApi->GetDocumentPhotosManager();
    auto fetchAssignmentResult = photoManager.GetDocumentVerificationAssignments().FetchInfo(secretId, session);
    R_ENSURE(fetchAssignmentResult.size() == 1, ConfigHttpStatus.UnknownErrorStatus, "can't restore assignment by secretId");

    auto assignment = fetchAssignmentResult.begin()->second;
    auto userId = assignment.GetUserId();

    auto eventLogState = NDrive::TEventLog::CaptureState();
    auto operatorUserId = eventLogState.UserId ? eventLogState.UserId : "yang";
    NDrive::TEventLog::TUserIdGuard userIdGuard(userId);

    auto fetchUserResult = DriveApi->GetUsersData()->FetchInfo(userId, session);
    R_ENSURE(fetchUserResult, {}, "cannot FetchInfo " << userId, session);

    R_ENSURE(fetchUserResult.size() == 1, ConfigHttpStatus.UnknownErrorStatus, "can't find user by userId");
    auto user = fetchUserResult.begin()->second;
    const auto& uid = user.GetUid();
    R_ENSURE(uid, ConfigHttpStatus.UnknownErrorStatus, "malformed uid for user " << userId << ": " << user.GetUid());

    // No requestData (aka body) means GET request type
    if (requestData == NJson::JSON_NULL) {
        ProcessGet(g, user, session);
        return;
    }

    // process POST request
    TMaybe<TUserPassportData> passportData;
    TMaybe<TUserDrivingLicenseData> drivingLicenseData;
    FillDocumentsData(g, requestData, assignmentId, user, passportData, drivingLicenseData);

    assignment.MutableAssignmentIds().push_back(assignmentId);

    auto updatePassportFuture = passportData ? DriveApi->GetPrivateDataClient().UpdatePassport(user, assignmentId, *passportData) : NThreading::MakeFuture();
    auto updateDrivingLicenseFuture = drivingLicenseData ? DriveApi->GetPrivateDataClient().UpdateDrivingLicense(user, assignmentId, *drivingLicenseData) : NThreading::MakeFuture();

    auto waiter = NThreading::WaitAll(updatePassportFuture, updateDrivingLicenseFuture);
    auto report = g.GetReport();
    eventLogState = NDrive::TEventLog::CaptureState();
    bool patch = GetValue<bool>(cgi, "patch", false).GetOrElse(false);
    g.Release();
    waiter.Subscribe([
        eventLogState = std::move(eventLogState),
        drivingLicense = std::move(drivingLicenseData),
        passport = std::move(passportData),
        drivingLicenseFuture = std::move(updateDrivingLicenseFuture),
        passportFuture = std::move(updatePassportFuture),
        report = std::move(report),
        user = std::move(user),
        assignment = std::move(assignment),
        skipError = !patch,
        server = Server,
        operatorUserId = operatorUserId,
        verificationStatuses = requestData["verification_statuses"],
        isExperimental = GetValue<bool>(cgi, "isExperimental", false).GetOrElse(true)
    ](const NThreading::TFuture<void>& /*w*/) mutable {
        NDrive::TEventLog::TStateGuard stateGuard(eventLogState);
        TJsonReport::TGuard g(report, HTTP_INTERNAL_SERVER_ERROR);
        if (!drivingLicenseFuture.HasValue()) {
            g.AddReportElement("license_error", NThreading::GetExceptionInfo(drivingLicenseFuture));
        }
        if (!passportFuture.HasValue()) {
            g.AddReportElement("passport_error", NThreading::GetExceptionInfo(passportFuture));
        }
        auto hasError = !drivingLicenseFuture.HasValue() || !passportFuture.HasValue();
        if (!skipError && hasError) {
            return;
        }
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        if (!server->GetDriveAPI()->GetUsersData()->UpdateUser(user, operatorUserId ? operatorUserId : "yang", session)) {
            g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                .AddInfo("session_info", session.GetReport())
                .SetErrorCode("cannot_update_user")
                << "cannot update user"
            );
            return;
        }

        if (verificationStatuses.IsMap()) {
            TryFromString(ToLowerUTF8(verificationStatuses["is_fraud"].GetString()), assignment.MutableIsFraud());
            assignment.SetProcessedAt(Now());
            assignment.SetIsExperimental(isExperimental);

            NYangAssignment::FillFraudReasons(assignment, verificationStatuses);
            if (verificationStatuses.Has("comment") && verificationStatuses["comment"].IsString()) {
                assignment.MutableComments().push_back(verificationStatuses["comment"].GetString());
            }

            TSet<TString> photosForFetch;
            if (verificationStatuses["license_back_status"].IsDefined()) {
                photosForFetch.insert(assignment.GetLicenseBackId());
            }
            if (verificationStatuses["license_front_status"].IsDefined()) {
                photosForFetch.insert(assignment.GetLicenseFrontId());
            }
            if (verificationStatuses["license_selfie_status"].IsDefined()) {
                photosForFetch.insert(assignment.GetLicenseSelfieId());
            }
            if (verificationStatuses["passport_biographical_status"].IsDefined()) {
                photosForFetch.insert(assignment.GetPassportBiographicalId());
            }
            if (verificationStatuses["passport_registration_status"].IsDefined()) {
                photosForFetch.insert(assignment.GetPassportRegistrationId());
            }
            if (verificationStatuses["passport_selfie_status"].IsDefined()) {
                photosForFetch.insert(assignment.GetPassportSelfieId());
            }
            TUserDocumentPhotosDB::TFetchResult photosFR;
            if (!photosForFetch.empty()) {
                photosFR = server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().FetchInfo(photosForFetch, session);
            }
            if (!photosFR) {
                g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                    .AddInfo("session_info", session.GetReport())
                    .SetErrorCode("cannot_fetch_photos")
                    << "cannot FetchInfo: " << JoinStrings(photosForFetch.begin(), photosForFetch.end(), ",")
                );
                return;
            }
            for (auto&& [_, photo] : photosFR.MutableResult()) {
                switch (photo.GetType()) {
                case NUserDocument::EType::LicenseFront:
                    photo.SetVerificationStatus(NYangAssignment::ParsePhotoStatus(verificationStatuses["license_front_status"]));
                    break;
                case NUserDocument::EType::LicenseBack:
                    photo.SetVerificationStatus(NYangAssignment::ParsePhotoStatus(verificationStatuses["license_back_status"]));
                    break;
                case NUserDocument::EType::LicenseSelfie:
                    photo.SetVerificationStatus(NYangAssignment::ParsePhotoStatus(verificationStatuses["license_selfie_status"]));
                    break;
                case NUserDocument::EType::PassportBiographical:
                    photo.SetVerificationStatus(NYangAssignment::ParsePhotoStatus(verificationStatuses["passport_biographical_status"]));
                    break;
                case NUserDocument::EType::PassportRegistration:
                    photo.SetVerificationStatus(NYangAssignment::ParsePhotoStatus(verificationStatuses["passport_registration_status"]));
                    break;
                case NUserDocument::EType::PassportSelfie: {
                    photo.SetVerificationStatus(NYangAssignment::ParsePhotoStatus(verificationStatuses["passport_selfie_status"]));
                    TString problemTagName = "";
                    if (photo.GetVerificationStatus() == NUserDocument::EVerificationStatus::VideoAnotherPerson) {
                        problemTagName = "blocked_fraud_selfie_another_p";
                    } else if (photo.GetVerificationStatus() == NUserDocument::EVerificationStatus::VideoScreencap) {
                        problemTagName = "blocked_fraud_selfie_screencap";
                    }
                    if (problemTagName) {
                        auto tag = IJsonSerializableTag::BuildWithComment<TUniqueUserTag>(problemTagName, "Что-то не так с селфи");;
                        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, operatorUserId, user.GetUserId(), server, session)) {
                            g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                                .AddInfo("session_info", session.GetReport())
                                .SetErrorCode("cannot_add_problem_tag")
                                << "cannot add tag " << tag->GetName()
                            );
                            return;
                        }
                    }
                    break;
                }
                default:
                    break;
                }
                photo.SetVerifiedAt(Now());
                if (!server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().Upsert(photo, session)) {
                    g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                        .AddInfo("session_info", session.GetReport())
                        .SetErrorCode("cannot_update_photo")
                        << "cannot update photo: " << photo.GetId()
                    );
                    return;
                }
            }

            if (hasError) {
                if (!session.Commit()) {
                    g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                        .AddInfo("session_info", session.GetReport())
                        .SetErrorCode("cannot_commit_and_upload")
                        << "cannot commit session"
                    );
                    return;
                }
                if (!server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, operatorUserId, Now(), passport.Get(), drivingLicense.Get())) {
                    g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                        .SetErrorCode("cannot_handle_assignment")
                        << "cannot HandleVerifiedAssignment"
                    );
                    return;
                } else {
                    assignment.SetFinalizedAt(Now());
                }

                session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto tag = IJsonSerializableTag::BuildWithComment<TUniqueUserTag>("datasync_problems", "Не могу сохранить данные в DataSync");
                if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, operatorUserId, user.GetUserId(), server, session)) {
                    g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                        .AddInfo("session_info", session.GetReport())
                        .SetErrorCode("cannot_add_tag")
                        << "cannot add tag " << tag->GetName()
                    );
                    return;
                }
            } else {
                if (assignment.GetFinalizedAt() != TInstant::Zero()) {
                    assignment.SetNeedReprocessing(true);
                }
            }
            if (!server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentVerificationAssignments().Upsert(assignment, session)) {
                g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                    .AddInfo("session_info", session.GetReport())
                    .SetErrorCode("cannot_update_assignment")
                    << "cannot upsert assignment " << assignment.GetId()
                );
                return;
            }
        }
        if (!session.Commit()) {
            g.SetCode(TCodedException(HTTP_INTERNAL_SERVER_ERROR)
                .AddInfo("session_info", session.GetReport())
                .SetErrorCode("cannot_commit")
                << "cannot commit update session"
            );
            return;
        }
        g.SetCode(HTTP_OK);
    });
}

void TYangUserCompletionHistoryProcessor::ProcessRequest(TJsonReport::TGuard& g, const NJson::TJsonValue& /*requestData*/) {
    TString secretId = GetString(Context->GetCgiParameters(), "secretId");
    TString assignmentId = GetString(Context->GetCgiParameters(), "assignmentId");
    TString reqAssignmentId = GetString(Context->GetCgiParameters(), "requestAssignmentId");

    auto session = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();

    // get userId by secretId
    R_ENSURE(DriveApi->HasDocumentPhotosManager(), ConfigHttpStatus.UnknownErrorStatus, "user photo manager not defined");
    const auto& photoManager = DriveApi->GetDocumentPhotosManager();
    auto fetchAssignmentResult = photoManager.GetDocumentVerificationAssignments().FetchInfo(secretId, session);
    R_ENSURE(fetchAssignmentResult.size() == 1, ConfigHttpStatus.UnknownErrorStatus, "can't restore assignment by secretId");
    auto userId = fetchAssignmentResult.begin()->second.GetUserId();

    auto userFetchResult = DriveApi->GetUsersData()->FetchInfo(userId, session);
    auto userPtr = userFetchResult.GetResultPtr(userId);
    R_ENSURE(userPtr, ConfigHttpStatus.UnknownErrorStatus, "no user with such id");

    const auto& uid = userPtr->GetUid();
    R_ENSURE(uid, ConfigHttpStatus.UnknownErrorStatus, "malformed uid for user " << userId << ": " << userPtr->GetUid());

    auto callbackTask = MakeAtomicShared<TPrefilledDataAcquisionCallback>(Server, g.GetReport(), false, false, false, false, false, *userPtr);
    DriveApi->GetPrivateDataClient().GetPassport(*userPtr, reqAssignmentId, callbackTask);
    DriveApi->GetPrivateDataClient().GetDrivingLicense(*userPtr, reqAssignmentId, callbackTask);
    g.Release();
}

void TYangFaceMatchingResultsProcessor::ProcessRequest(TJsonReport::TGuard& g, const NJson::TJsonValue& requestData) {
    TString secretId = GetString(Context->GetCgiParameters(), "secretId");
    TString assignmentId = GetString(Context->GetCgiParameters(), "assignmentId");

    auto session = Server->GetDriveAPI()->template BuildTx<NSQL::Writable>();

    // get userId by secretId
    R_ENSURE(DriveApi->HasDocumentPhotosManager(), ConfigHttpStatus.UnknownErrorStatus, "user photo manager not defined");
    const auto& photoManager = DriveApi->GetDocumentPhotosManager();
    auto fetchAssignmentResult = photoManager.GetDocumentVerificationAssignments().FetchInfo(secretId, session);
    R_ENSURE(fetchAssignmentResult.size() == 1, ConfigHttpStatus.UnknownErrorStatus, "can't restore assignment by secretId");

    auto assignment = std::move(fetchAssignmentResult.begin()->second);
    R_ENSURE(requestData.IsMap() && requestData["verification_statuses"].IsMap(), ConfigHttpStatus.SyntaxErrorStatus, "no verification_statuses");
    auto verificationStatuses = requestData["verification_statuses"];
    R_ENSURE(verificationStatuses.Has("selfie_status") && verificationStatuses["selfie_status"].IsString(), ConfigHttpStatus.SyntaxErrorStatus, "no selfie status");

    auto fetchSelfieResult = photoManager.GetUserPhotosDB().FetchInfo(assignment.GetSelfieId(), session);
    R_ENSURE(fetchSelfieResult.size() == 1, ConfigHttpStatus.UnknownErrorStatus, "no selfie in assignment");
    auto selfie = std::move(fetchSelfieResult.begin()->second);

    selfie.SetVerificationStatus(NYangAssignment::ParsePhotoStatus(verificationStatuses["selfie_status"]));
    selfie.SetVerifiedAt(Context->GetRequestStartTime());
    assignment.SetProcessedAt(Context->GetRequestStartTime());
    R_ENSURE(TryFromString(ToLowerUTF8(verificationStatuses["is_fraud"].GetString()), assignment.MutableIsFraud()), ConfigHttpStatus.SyntaxErrorStatus, "can't parse overall status");
    NYangAssignment::FillFraudReasons(assignment, verificationStatuses);
    if (verificationStatuses.Has("comment") && verificationStatuses["comment"].IsString()) {
        assignment.MutableComments().push_back(verificationStatuses["comment"].GetString());
    }

    R_ENSURE(photoManager.GetDocumentVerificationAssignments().Upsert(assignment, session), ConfigHttpStatus.UnknownErrorStatus, "can't upsert assignment");
    R_ENSURE(photoManager.GetUserPhotosDB().Upsert(selfie, session), ConfigHttpStatus.UnknownErrorStatus, "can't upsert selfie");
    R_ENSURE(session.Commit(), ConfigHttpStatus.UnknownErrorStatus, "can't commit session");

    g.SetCode(HTTP_OK);
}
