#include "zones.h"

#include <drive/backend/parking_zones/manager.h>
#include <drive/backend/saas/api.h>

#include <drive/library/cpp/parking/inhouse/client.h>

void TParkingProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    const TCgiParameters& cgi = Context->GetCgiParameters();
    const auto& configHttpStatus = Server->GetHttpStatusManagerConfig();

    const bool useRawLocation = (Server->GetSettings().GetValueDef("parking_use_raw_location", true) && GetValue<bool>(cgi, "use_raw_location", false).GetOrElse(true));
    TMaybe<double> precision = GetValue<double>(cgi, "precision", false);
    const auto& parkingManager =  Server->GetParkingZonesManager();
    TExpected<TParkingInfo, TString> res;
    if (useRawLocation) {
        res = parkingManager.IsParkingAllowed(permissions->GetUserId(), *Context, precision);
    } else {
        auto position = FromString<TGeoCoord>(cgi.Get("position"));
        res = parkingManager.IsParkingAllowed(position, precision);
    }
    R_ENSURE(res, configHttpStatus.UnknownErrorStatus, res.GetError());

    TJsonReport& report = g.MutableReport();
    report.AddReportElement("data.signal_issues", res->SignalIssues);
    report.AddReportElement("data.parking_ability", ::ToString(res->Ability));
    report.AddReportElement("data.details", ::ToString(res->DenyDetails));
    if (res->Tags) {
        report.AddReportElement("data.tags", JoinStrings(res->Tags.begin(), res->Tags.end(), ","));
    }
    g.SetCode(HTTP_OK);
}

void TParkingListProcessor::ProcessServiceRequest(TJsonReport::TGuard& g, TUserPermissions::TPtr permissions, const NJson::TJsonValue& /*requestData*/) {
    Y_UNUSED(permissions);
    const TCgiParameters& cgi = Context->GetCgiParameters();
    auto coordinate = GetValue<TGeoCoord>(cgi, "coordinate");

    auto service = GetHandlerSetting<TString>("parking_list_client.service").GetOrElse("drive_router");
    auto api = Server->GetRTLineAPI(service);
    R_ENSURE(api, ConfigHttpStatus.UnknownErrorStatus, "cannot find service " << service);

    NDrive::TParkingListClient client(api->GetSearchClient());
    auto asyncParkings = client.GetParkings(coordinate->Y, coordinate->X);

    TJsonReport& report = g.MutableReport();
    report.AddReportElement("parkings", NJson::ToJson(asyncParkings.GetValueSync()));
    g.SetCode(HTTP_OK);
}
