#include "actions.h"

#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/account_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

TPromoTagProfit::TFactory::TRegistrator<TPromoTagProfit> TPromoTagProfit::Registrator(TPromoTagProfit::GetTypeName());

NDrive::TScheme TPromoTagProfit::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result;
    result.Add<TFSBoolean>("is_disposable", "Активировать можно только один раз").SetDefault(true);
    result.Add<TFSBoolean>("is_duplicate", "Несколько экземпляров одновременно").SetDefault(false);

    ITagsMeta::TTagDescriptionsByName tags = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();
    result.Add<TFSVariants>("tag_name", "Тег").SetVariants(MakeVector(NContainer::Keys(tags))).SetRequired(true);
    return result;
}

bool TPromoTagProfit::DoApply(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatSession*/) const {
    auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(TagName, GetType());
    return tag && FillTagInfo(tag, meta, server, session) && server.GetDriveAPI()->GetEntityTagsManager(GetEntityType(server)).AddTag(tag, historyUserId, objectId, &server, session);
}

bool TPromoTagProfit::AvailableForObject(const IPromoCodesManager::TApplyContext& context, const TString& objectId, const TPromoCodeMeta& meta, const NDrive::IServer& server, const TString& generatorName, NDrive::TEntitySession& session) const {
    if (!TBase::AvailableForObject(context, objectId, meta, server, generatorName, session)) {
        return false;
    }

    if (IsDisposable() || !IsDuplicate()) {
        auto optionalEvents = server.GetDriveAPI()->GetEntityTagsManager(GetEntityType(server)).GetEventsByObject(objectId, session);
        if (!optionalEvents) {
            return false;
        }
        for (auto&& ev : Reversed(*optionalEvents)) {
            if (ev && ev->GetName() == TagName) {
                if (IsDisposable() || (!IsDuplicate() && ev.GetHistoryAction() != EObjectHistoryAction::Remove)) {
                    auto landingId = server.GetSettings().GetValueDef<TString>("landing." + generatorName + ".promo_type_already_used", "");
                    if (!landingId || !server.GetDriveAPI()->GetLandingsDB()->InitLandingUserRequest({ landingId }, "TPromoTagProfit::AvailableForUser", session, nullptr, EDriveSessionResult::DuplicatePromoCode)) {
                        session.SetErrorInfo("TPromoTagProfit::AvailableForUser", "with type of promo already been used for:" + objectId, EDriveSessionResult::DuplicatePromoCode);
                    }
                    session.SetLocalizedMessageKey(server.GetSettings().GetValueDef<TString>("error." + generatorName + ".promo_type_already_used", "promo_type_already_used"));
                    return false;
                } else {
                    return true;
                }
            }
        }
    }
    return true;
}

NEntityTagsManager::EEntityType TPromoTagProfit::GetEntityType(const NDrive::IServer& server) const {
    auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(TagName);
    if (!tag || tag->GetObjectType().empty()) {
        return NEntityTagsManager::EEntityType::Undefined;
    }
    return *tag->GetObjectType().begin();
}

TPromoAttemptsDiscount::TFactory::TRegistrator<TPromoAttemptsDiscount> TPromoAttemptsDiscount::Registrator(TPromoAttemptsDiscount::GetTypeName());

NDrive::TScheme TPromoAttemptsDiscount::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);

    ITagsMeta::TTagDescriptionsByName tags = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();
    TSet<TString> tagNames;
    for (auto&& i : tags) {
        const ITemporaryActionTag::TDescription* descr = dynamic_cast<const ITemporaryActionTag::TDescription*>(i.second.Get());
        if (descr) {
            tagNames.emplace(descr->GetName());
        }
    }
    result.Remove("tag_name").Add<TFSVariants>("tag_name", "Тег для скидки").SetVariants(tagNames).SetRequired(true);

    result.Add<TFSNumeric>("attempts_count", "Количество поездок").SetMin(1).SetRequired(true);
    result.Add<TFSNumeric>("since", "Дата старта").SetVisual(TFSNumeric::EVisualType::DateTime);
    result.Add<TFSNumeric>("until", "Дата окончания").SetVisual(TFSNumeric::EVisualType::DateTime);
    result.Add<TFSDuration>("duration", "Продолжительность");
    result.Add<TFSVariants>("lifetime_policy", "Политика сгорания").InitVariants<EDeadlinePolicy>().SetDefault(::ToString(EDeadlinePolicy::None));
    result.Add<TFSDuration>("timezone", "Сдвиг временной зоны пользователя");
    return result;
}

bool TPromoAttemptsDiscount::DoApply(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const {
    if (Until < ModelingNow()) {
        session.SetErrorInfo("apply promo code", "expired code", EDriveSessionResult::IncorrectRequest);
        session.SetLocalizedMessageKey("promotional_completed");
        return false;
    }
    return TBase::DoApply(objectId, historyUserId, meta, server, session, chatSession);
}

bool TPromoAttemptsDiscount::FillTagInfo(ITag::TPtr tag, const TPromoCodeMeta& meta, const NDrive::IServer& /*server*/, NDrive::TEntitySession& session) const {
    auto tagImpl = dynamic_cast<ITemporaryActionTag*>(tag.Get());
    if (!tagImpl) {
        session.SetErrorInfo("TPromoAttemptsDiscount::FillTagInfo", "incorrect cast to ITemporaryActionTag", EDriveSessionResult::InternalError);
        return false;
    }
    tagImpl->SetAttempts(AttemptsCount);
    tagImpl->SetMaxAttempts(AttemptsCount);
    tagImpl->SetPromoCode(true);
    tagImpl->SetPromoIdentifier(meta.GetCode());

    if (auto accountTag = dynamic_cast<TAccountTemporaryActionTag*>(tag.Get()); accountTag && GetActivateSum()) {
        accountTag->SetRequiredSum(GetActivateSum());
        accountTag->SetActive(false);
    }

    if (Duration != TDuration::Zero()) {
        TInstant minimalTime = ModelingNow() + Duration;
        TInstant realDeadlineTime = minimalTime;
        {
            struct tm tmNow;
            minimalTime.LocalTime(&tmNow);
            TDateTimeFields timeParts;
            switch (LifetimePolicy) {
            case EDeadlinePolicy::Month:
                timeParts.Hour = 0;
                timeParts.Day = 1;
                timeParts.Month = (tmNow.tm_mon + 1) % 12 + 1;
                timeParts.SetLooseYear((tmNow.tm_mon + 1) > 11 ? tmNow.tm_year + 1 : tmNow.tm_year);
                realDeadlineTime = timeParts.ToInstant(TInstant::Max()) - Timezone;
                break;
            case EDeadlinePolicy::Day:
                realDeadlineTime = TInstant::Days(minimalTime.Days() + 1) - Timezone;
                break;
            case EDeadlinePolicy::Week:
                realDeadlineTime = TInstant::Days(minimalTime.Days()) + TDuration::Days(7 - tmNow.tm_wday) + Timezone;
                break;
            case EDeadlinePolicy::None:
                realDeadlineTime = minimalTime;
                break;
            }
        }

        tagImpl->SetUntil(realDeadlineTime);
    } else {
        tagImpl->SetSince(Since);
        tagImpl->SetUntil(Until);
    }
    return true;
}

TPromoBillingTag::TFactory::TRegistrator<TPromoBillingTag> TPromoBillingTag::Registrator(TPromoBillingTag::GetTypeName());

NDrive::TScheme TPromoBillingTag::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("robot_id", "Исполняющий робот").SetDefault("ec65f4f0-8fa8-4887-bfdc-ca01c9906696").SetRequired(true);

    ITagsMeta::TTagDescriptionsByName tags = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();
    TSet<TString> tagNames;
    for (auto&& i : tags) {
        const TBillingTagDescription* descr = dynamic_cast<const TBillingTagDescription*>(i.second.Get());
        if (descr) {
            tagNames.emplace(descr->GetName());
        }
    }
    result.Remove("tag_name").Add<TFSVariants>("tag_name", "Тег для начисления").SetVariants(tagNames).SetRequired(true);
    return result;
}

bool TPromoBillingTag::DoApply(const TString& objectId, const TString& /*historyUserId*/, const TPromoCodeMeta& /*meta*/, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const {
    const TBillingTagDescription* billingDesc = dynamic_cast<const TBillingTagDescription*>(server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetTagName()).Get());
    if (!billingDesc) {
        session.SetErrorInfo("TPromoBilling::Apply", "incorrect billing description " + GetTagName(), EDriveSessionResult::InternalError);
        return false;
    }
    return billingDesc->ProcessBillingOperation(objectId, RobotId, server, session, chatSession);
}

TPromoReferralDiscount::TFactory::TRegistrator<TPromoReferralDiscount> TPromoReferralDiscount::Registrator(TPromoReferralDiscount::GetTypeName());

NDrive::TScheme TPromoReferralDiscount::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("connection_tag", "Тег связи(потребитель)").SetRequired(true);
    result.Add<TFSString>("connection_friend_tag", "Тег связи(источник)").SetRequired(true);
    return result;
}

bool TPromoReferralDiscount::DoApply(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const {
    {
        auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(ConnectionTag);
        {
            TConnectionUserTag* connectionTag = dynamic_cast<TConnectionUserTag*>(tag.Get());
            if (!connectionTag) {
                session.SetErrorInfo("TPromoReferralDiscount::DoExecute", "incorrect cast " + ConnectionTag + " to TConnectionUserTag", EDriveSessionResult::InternalError);
                return false;
            }
            connectionTag->SetConnectedUserId(meta.GetGivenOut());
        }

        if (!server.GetDriveAPI()->GetEntityTagsManager(GetEntityType(server)).AddTag(tag, historyUserId, objectId, &server, session)) {
            return false;
        }
    }
    if (ConnectionFriendTag)  {
        auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(ConnectionFriendTag);
        {
            TConnectionUserTag* connectionTag = dynamic_cast<TConnectionUserTag*>(tag.Get());
            if (!connectionTag) {
                session.SetErrorInfo("TPromoReferralDiscount::DoExecute", "incorrect cast " + ConnectionTag + " to TConnectionUserTag", EDriveSessionResult::InternalError);
                return false;
            }
            connectionTag->SetConnectedUserId(objectId);
        }

        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, historyUserId, meta.GetGivenOut(), &server, session)) {
            return false;
        }
    }
    return TBase::DoApply(objectId, historyUserId, meta, server, session, chatSession);
}

bool TPromoReferralDiscount::AvailableForObject(const IPromoCodesManager::TApplyContext& context, const TString& objectId, const TPromoCodeMeta& meta, const NDrive::IServer& server, const TString& generatorName, NDrive::TEntitySession& session) const {
    TString sourceUuid = meta.GetGivenOut();
    auto fetchResult = server.GetDriveAPI()->GetUsersData()->RestoreUser(sourceUuid, session);
    if (!fetchResult) {
        session.AddErrorMessage("TPromoReferralDiscount::DoExecute", "incorrect given out " + sourceUuid);
        session.SetLocalizedMessageKey(server.GetSettings().GetValueDef<TString>("error." + generatorName + ".incorrect_promo_code", "incorrect_promo_code"));
        session.SetResult(EDriveSessionResult::IncorrectRequest);
        return false;
    }
    if (sourceUuid == objectId) {
        session.SetErrorInfo("TPromoReferralDiscount::DoExecute", "apply a referral code to yourself", EDriveSessionResult::NoUserPermissions);
        session.SetLocalizedMessageKey(server.GetSettings().GetValueDef<TString>("error." + generatorName + ".no_promo_code_permissions", "no_promo_code_permissions"));
        return false;
    }

    if (!TBase::AvailableForObject(context, objectId, meta, server, generatorName, session)) {
        return false;
    }

    TVector<TDBTag> tags;
    if (!server.GetDriveAPI()->GetEntityTagsManager(GetEntityType(server)).RestoreTags({ objectId }, { ConnectionTag }, tags, session)) {
        return false;
    }
    if (tags.size() != 0) {
        session.SetErrorInfo("TPromoReferralDiscount::AvailableForUser", "with type of promo already been used for:" + objectId, EDriveSessionResult::DuplicatePromoCode);
        session.SetLocalizedMessageKey(server.GetSettings().GetValueDef<TString>("error." + generatorName + ".promo_type_already_used", "promo_type_already_used"));
        return false;
    }
    return true;
}

TPCGRidesDiscount::TFactory::TRegistrator<TPCGRidesDiscount> TPCGRidesDiscount::Registrator(TPCGRidesDiscount::GetTypeName());
TUserTagProfitGenerator::TFactory::TRegistrator<TUserTagProfitGenerator> TUserTagProfitGenerator::Registrator(TUserTagProfitGenerator::GetTypeName());
TBillingTagProfitGenerator::TFactory::TRegistrator<TBillingTagProfitGenerator> TBillingTagProfitGenerator::Registrator(TBillingTagProfitGenerator::GetTypeName());
TReferralCodeGenerator::TFactory::TRegistrator<TReferralCodeGenerator> TReferralCodeGenerator::Registrator(TReferralCodeGenerator::GetTypeName());
TPromoCodeUsageActionImpl::TFactory::TRegistrator<TPromoCodeUsageActionImpl> TPromoCodeUsageActionImpl::Registrator(TPromoCodeUsageActionImpl::GetTypeName());

NDrive::TScheme TPromoCodeUsageActionImpl::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    const TVector<TString> actions = server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionNamesWithType<IPromoCodeGenerator>(/*reportDeprecated=*/false);
    result.Add<TFSVariants>("available_promo_types", "Допустимые типы промо кодов для использования").SetVariants(actions).SetMultiSelect(true);
    return result;
}
