#pragma once
#include "entities.h"

#include <drive/backend/promo_codes/common/action.h>

#include <drive/backend/actions/abstract/action.h>

#include <drive/library/cpp/scheme/scheme.h>

class TPromoTagProfit : public IPromoProfit, public IPromoTagProfit {
    using TBase = IPromoProfit;
    R_READONLY(bool, Disposable, true);
    R_READONLY(bool, Duplicate, false);
    static TFactory::TRegistrator<TPromoTagProfit> Registrator;
public:
    static TString GetTypeName() {
        return "user_tag_profit";
    }

    void SetTagName(const TString& tagName) {
        TagName = tagName;
    }

protected:
    const TString& GetTagName() const override {
        return TagName;
    }

    virtual TString GetType() const override {
        return TPromoTagProfit::GetTypeName();
    }

    virtual NEntityTagsManager::EEntityType GetEntityType(const NDrive::IServer& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonValue) override {
        JREAD_STRING(jsonValue, "tag_name", TagName);
        JREAD_BOOL_OPT(jsonValue, "is_disposable", Disposable);
        JREAD_BOOL_OPT(jsonValue, "is_duplicate", Duplicate);
        return true;
    }

    virtual NJson::TJsonValue DoSerializeToJson() const override {
        NJson::TJsonValue result;
        JWRITE(result, "tag_name", TagName);
        JWRITE(result, "is_disposable", Disposable);
        JWRITE(result, "is_duplicate", Duplicate);
        return result;
    }
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    virtual bool DoApply(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& /*chatSession*/) const override;
    virtual bool AvailableForObject(const IPromoCodesManager::TApplyContext& context, const TString& userId, const TPromoCodeMeta& meta, const NDrive::IServer& server, const TString& generatorName, NDrive::TEntitySession& session) const override;

private:
    virtual bool FillTagInfo(ITag::TPtr /*tag*/, const TPromoCodeMeta& /*meta*/, const NDrive::IServer& /*server*/, NDrive::TEntitySession& /*session*/) const {
        return true;
    }

protected:
    TString TagName;
};

class TPromoAttemptsDiscount: public TPromoTagProfit {
public:
    enum EDeadlinePolicy {
        None,
        Day,
        Week,
        Month,
    };

private:
    using TBase = TPromoTagProfit;
    R_READONLY(ui32, AttemptsCount, 1);
    R_READONLY(TInstant, Since, TInstant::Zero());
    R_READONLY(TInstant, Until, TInstant::Max());
    R_READONLY(TDuration, Duration, TDuration::Zero());
    R_READONLY(EDeadlinePolicy, LifetimePolicy, EDeadlinePolicy::None);
    R_READONLY(TDuration, Timezone, TDuration::Hours(3));
    static TFactory::TRegistrator<TPromoAttemptsDiscount> Registrator;

public:
    static TString GetTypeName() {
        return "attempts_discount";
    }

protected:
    virtual TString GetType() const override {
        return TPromoAttemptsDiscount::GetTypeName();
    }

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonValue) override {
        JREAD_INT_OPT(jsonValue, "attempts_count", AttemptsCount);
        JREAD_INSTANT_OPT(jsonValue, "since", Since);
        JREAD_DURATION_OPT(jsonValue, "duration", Duration);
        JREAD_INSTANT_OPT(jsonValue, "until", Until);
        JREAD_FROM_STRING_OPT(jsonValue, "lifetime_policy", LifetimePolicy);
        JREAD_DURATION_OPT(jsonValue, "timezone", Timezone);
        return TBase::DoDeserializeFromJson(jsonValue);
    }

    virtual NJson::TJsonValue DoSerializeToJson() const override {
        NJson::TJsonValue result = TBase::DoSerializeToJson();
        JWRITE(result, "attempts_count", AttemptsCount);
        JWRITE_INSTANT(result, "since", Since);
        TJsonProcessor::WriteInstant(result, "until", Until, TInstant::Max());
        TJsonProcessor::WriteDurationInt(result, "duration", Duration, TDuration::Zero());
        JWRITE_ENUM_DEF(result, "lifetime_policy", LifetimePolicy, EDeadlinePolicy::None);
        TJsonProcessor::WriteDurationString(result, "timezone", Timezone, TDuration::Hours(3));
        return result;
    }
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    virtual bool DoApply(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const override;
    virtual bool FillTagInfo(ITag::TPtr tag, const TPromoCodeMeta& meta, const NDrive::IServer& /*server*/, NDrive::TEntitySession& session) const override;
};

class TPromoBillingTag : public TPromoTagProfit {
    using TBase = TPromoTagProfit;
    R_READONLY(TString, RobotId);
    static TFactory::TRegistrator<TPromoBillingTag> Registrator;

public:
    static TString GetTypeName() {
        return "promo_billing_tag";
    }

protected:
    virtual TString GetType() const override {
        return TPromoBillingTag::GetTypeName();
    }

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonValue) override {
        JREAD_STRING(jsonValue, "robot_id", RobotId);
        return TBase::DoDeserializeFromJson(jsonValue);
    }

    virtual NJson::TJsonValue DoSerializeToJson() const override {
        NJson::TJsonValue result = TBase::DoSerializeToJson();
        JWRITE(result, "robot_id", RobotId);
        return result;
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    virtual bool DoApply(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& /*meta*/, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const override;
};

class TPromoReferralDiscount : public TPromoAttemptsDiscount {
    using TBase = TPromoAttemptsDiscount;
    R_READONLY(TString, ConnectionTag);
    R_READONLY(TString, ConnectionFriendTag);
    static TFactory::TRegistrator<TPromoReferralDiscount> Registrator;

public:
    static TString GetTypeName() {
        return "referral_discount";
    }

protected:
    virtual TString GetType() const override {
        return TPromoReferralDiscount::GetTypeName();
    }

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonValue) override {
        JREAD_STRING(jsonValue, "connection_tag", ConnectionTag);
        JREAD_STRING_OPT(jsonValue, "connection_friend_tag", ConnectionFriendTag);
        return TBase::DoDeserializeFromJson(jsonValue);
    }

    virtual NJson::TJsonValue DoSerializeToJson() const override {
        NJson::TJsonValue result = TBase::DoSerializeToJson();
        JWRITE(result, "connection_tag", ConnectionTag);
        JWRITE(result, "connection_friend_tag", ConnectionFriendTag);
        return result;
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    virtual bool DoApply(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const override;
    virtual bool AvailableForObject(const IPromoCodesManager::TApplyContext& context, const TString& objectId, const TPromoCodeMeta& meta, const NDrive::IServer& server, const TString& generatorName, NDrive::TEntitySession& session) const override;
};

class TPromoCodeUsageActionImpl : public IPromoCodeUsage, public TUserAction {
    using TBase = TUserAction;

protected:
    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override {
        if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
            return false;
        }
        if (!TJsonProcessor::ReadContainer(jsonValue, "available_promo_types", AvailableTypes)) {
            return false;
        }
        return true;
    }
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override {
        NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
        TJsonProcessor::WriteContainerArray(result, "available_promo_types", AvailableTypes);
        return result;
    }
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

public:
    void SetAvailableTypes(const TSet<TString>& availableTypes) {
        AvailableTypes = availableTypes;
    }

    virtual const TSet<TString>& GetAvailableTypes() const override {
        return AvailableTypes;
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "promo_types_usage";
    }

private:
    TSet<TString> AvailableTypes;
    static TFactory::TRegistrator<TPromoCodeUsageActionImpl> Registrator;
};

template <class TPromoProfitClass>
class TCustomPromoInfoAction : public IPromoCodeGenerator, public TUserAction {
private:
    using TBase = TUserAction;
    R_FIELD(TPromoProfitClass, PromoProfit);
protected:
    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override {
        if (!TBase::DeserializeSpecialsFromJson(jsonValue)) {
            return false;
        }
        if (!jsonValue.Has("promo_info")) {
            return false;
        }
        if (!PromoProfit.DeserializeFromJson(jsonValue["promo_info"])) {
            return false;
        }
        return true;
    }

    virtual NJson::TJsonValue SerializeSpecialsToJson() const override {
        NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
        result.InsertValue("promo_info", PromoProfit.SerializeToJson());
        return result;
    }

public:
    virtual const TString& GetName() const override {
        return TBase::GetName();
    }

    virtual IPromoProfit::TPtr BuildPromoProfit() const override {
        return new TPromoProfitClass(PromoProfit);
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::DoGetScheme(server);
        result.Add<TFSStructure>("promo_info").SetStructure<NDrive::TScheme>(PromoProfit.GetScheme(server));
        return result;
    }
};

class TPCGRidesDiscount : public TCustomPromoInfoAction<TPromoAttemptsDiscount> {
private:
    static TFactory::TRegistrator<TPCGRidesDiscount> Registrator;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "promo_generator_rides";
    }
};

class TUserTagProfitGenerator : public TCustomPromoInfoAction<TPromoTagProfit> {
private:
    static TFactory::TRegistrator<TUserTagProfitGenerator> Registrator;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "promo_generator_simple_tag";
    }
};

class TReferralCodeGenerator : public TCustomPromoInfoAction<TPromoReferralDiscount> {
private:
    static TFactory::TRegistrator<TReferralCodeGenerator> Registrator;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "promo_generator_referral_code";
    }
};

class TBillingTagProfitGenerator : public TCustomPromoInfoAction<TPromoBillingTag> {
private:
    static TFactory::TRegistrator<TBillingTagProfitGenerator> Registrator;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "promo_generator_billing_tag";
    }
};
