#include "entities.h"

#include <drive/backend/abstract/drive_database.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/notifications/native_chat/chat.h>
#include <drive/backend/tags/tags.h>

template <>
NJson::TJsonValue NJson::ToJson<TTagDictionaryContext::TElement>(const TTagDictionaryContext::TElement& object) {
    return object.ToJson();
}

template <>
bool NJson::TryFromJson<TTagDictionaryContext::TElement>(const TJsonValue& value, TTagDictionaryContext::TElement& result) {
    return result.FromJson(value);
}

template <>
NJson::TJsonValue NJson::ToJson<TTagDictionaryContext>(const TTagDictionaryContext& object) {
    return object.ToJson();
}

template <>
bool NJson::TryFromJson<TTagDictionaryContext>(const TJsonValue& value, TTagDictionaryContext& result) {
    return result.FromJson(value);
}

NDrive::TScheme TTagDictionaryContext::GetScheme(const NDrive::IServer* server) {
    NDrive::TScheme scheme;
    scheme.Add<TFSVariants>("tag_name", "Название тега").SetVariants(MakeSet(NContainer::Keys(server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User, { TUserDictionaryTag::TypeName }))));
    scheme.Add<TFSArray>("parameters", "Параметры").SetElement(TElement::GetScheme(server));
    return scheme;
}

NSQL::TQueryOptions TPromoCodesSearchContext::BuildCondition() const {
    NSQL::TQueryOptions result;
    if (Ids) {
        result.SetGenericCondition("id", *Ids);
    }
    if (GivenOut && *GivenOut != "__ALL") {
        if (*GivenOut == "__FILLED") {
            result.SetGenericCondition("given_out", NSQL::Not(NSQL::TEmptyString()));
        } else if (*GivenOut == "__EMPTY" || GivenOut->empty()) {
            result.SetGenericCondition("given_out", NSQL::TEmptyString());
        } else {
            result.AddGenericCondition("given_out", *GivenOut);
        }
    }
    const bool activeOnly = ActiveOnly && *ActiveOnly;
    if (activeOnly) {
        result.SetGenericCondition("activity_start", MakeRange<ui64>(Nothing(), ModelingNow().Seconds() + 1));
    }
    if (activeOnly || Since || Until) {
        TRange<ui64> range;
        if ((activeOnly) || Since) {
            const ui64 sSince = Since ? Since->Seconds() : 0;
            const ui64 aSince = activeOnly ? ModelingNow().Seconds() : 0;
            range.From = Max(sSince, aSince);
        }
        if (Until) {
            range.To = Until->Seconds();
        }
        result.SetGenericCondition("activity_deadline", range);
    }
    if (Prefix) {
        if (Prefix->empty()) {
            result.SetGenericCondition("prefix", NSQL::TEmptyString());
        } else {
            result.AddGenericCondition("prefix", *Prefix);
        }
    }
    return result;
}

bool TPromoCodesSearchContext::ReadFrom(const TCgiParameters& cgi) {
    if (cgi.Has("given_out")) {
        GivenOut = cgi.Get("given_out");
    }
    if (cgi.Has("prefix")) {
        Prefix = cgi.Get("prefix");
    }
    if (cgi.Has("generator")) {
        Generator = cgi.Get("generator");
    }
    ui32 count = 0;
    if (cgi.Has("count")) {
        if (!TryFromString<ui32>(cgi.Get("count"), count)) {
            return false;
        }
        Count = count;
    }
    if (cgi.Has("since")) {
        const TString sinceStr = cgi.Get("since");
        ui32 sinceInt;
        if (!TryFromString(sinceStr, sinceInt)) {
            return false;
        }
        Since = TInstant::Seconds(sinceInt);
    }
    if (cgi.Has("until")) {
        const TString untilStr = cgi.Get("until");
        ui32 untilInt;
        if (!TryFromString(untilStr, untilInt)) {
            return false;
        }
        Until = TInstant::Seconds(untilInt);
    }
    if (cgi.Has("active_only")) {
        ActiveOnly = IsTrue(cgi.Get("active_only"));
    }
    if (cgi.Has("ids")) {
        Ids.ConstructInPlace();
        StringSplitter(cgi.Get("ids")).SplitBySet(", ").SkipEmpty().Collect(Ids.Get());
    }
    if (cgi.Has("usage_history")) {
        UsageHistory = IsTrue(cgi.Get("usage_history"));
    }
    return !IsEmpty();
}

bool AddSymbolImpl(const TString& charPool, const ui32 sym, TString& result, ui32& delta) {
    if (sym - delta < charPool.size()) {
        result += charPool[sym - delta];
        return true;
    }
    delta += charPool.size();
    return false;
}

bool TPromoCodeBuilder::AddSymbol(const ui32 sym, TString& result) const {
    ui32 delta = 0;
    if (DigitsUsage && AddSymbolImpl(DigitLiterals, sym, result, delta)) {
        return true;
    }
    if (UpperLiteralsUsage && AddSymbolImpl(UpperLiterals, sym, result, delta)) {
        return true;
    }
    if (LowerLiteralsUsage && AddSymbolImpl(LowerLiterals, sym, result, delta)) {
        return true;
    }
    return false;
}

bool TPromoCodeBuilder::DeserializeFromJson(const NJson::TJsonValue& data) {
    JREAD_BOOL_OPT(data, "upper_literals_usage", UpperLiteralsUsage);
    JREAD_BOOL_OPT(data, "lower_literals_usage", LowerLiteralsUsage);
    JREAD_INT_OPT(data, "count_symbols", CountSymbols);
    JREAD_STRING_OPT(data, "prefix", Prefix);
    JREAD_BOOL_OPT(data, "digits_usage", DigitsUsage);

    JREAD_STRING_OPT(data, "upper_literals", UpperLiterals);
    JREAD_STRING_OPT(data, "lower_literals", LowerLiterals);
    JREAD_STRING_OPT(data, "digit_literals", DigitLiterals);
    return true;
}

bool TPromoCodeBuilder::Build(TString& result) const {
    result = Prefix;
    TGUID guid;
    CreateGuid(&guid);
    ui32 variantsCount = 0;
    if (DigitsUsage) {
        variantsCount += DigitLiterals.size();
    }
    if (UpperLiteralsUsage) {
        variantsCount += UpperLiterals.size();
    }
    if (LowerLiteralsUsage) {
        variantsCount += LowerLiterals.size();
    }
    if (!variantsCount) {
        return false;
    }
    for (ui32 i = 0; i < 4; ++i) {
        ui32 val = guid.dw[i];
        ui32 maxSymbols = Max<ui32>();
        while (maxSymbols) {
            if (result.size() >= Prefix.size() + CountSymbols) {
                return true;
            }
            const ui8 sym = val % variantsCount;
            maxSymbols /= variantsCount;
            if (!AddSymbol(sym, result)) {
                return false;
            }
            val = val / variantsCount;
        }
    }
    return false;
}

bool IPromoProfitBase::TChatBuilder::DeserializeFromJson(const NJson::TJsonValue& jsonValue) {
    JREAD_STRING_OPT(jsonValue, "chat_notifier", ChatNotifier);
    return true;
}

NJson::TJsonValue IPromoProfitBase::TChatBuilder::SerializeToJson() const {
    NJson::TJsonValue result;
    JWRITE(result, "chat_notifier", ChatNotifier);
    return result;
}

NDrive::TScheme IPromoProfitBase::TChatBuilder::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSVariants>("chat_notifier", "Нотификатор чата").SetVariants(server->GetNotifierNames());
    return scheme;
}

NJson::TJsonValue IPromoProfitBase::TChatBuilder::GetDescriptor(const TString& userId, const TString& topic, const NDrive::IServer& server) const {
    auto notifier = dynamic_cast<const TNativeChatNotifier*>(server.GetNotifier(ChatNotifier).Get());
    if (notifier) {
        return notifier->GetDescriptor(userId, topic);
    }
    return{};
}

NStorage::TTableRecord TBasePromoMeta::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("usage_limit", UsageCountLimit);
    result.Set("activity_start", ActivityStart.Seconds());
    result.Set("activity_deadline", ActivityDeadline.Seconds());
    result.Set("removing_deadline", RemovingDeadline.Seconds());
    result.Set("prefix", Prefix);
    result.Set("given_out", GivenOut);
    result.Set("meta", Meta);
    return result;
}

NJson::TJsonValue TBasePromoMeta::SerializeToJsonReport() const {
    NJson::TJsonValue json;
    JWRITE(json, "usage_limit", UsageCountLimit);
    JWRITE_INSTANT(json, "activity_start", ActivityStart);
    JWRITE_INSTANT(json, "activity_deadline", ActivityDeadline);
    JWRITE_INSTANT(json, "removing_deadline", RemovingDeadline);
    JWRITE(json, "prefix", Prefix);
    JWRITE(json, "given_out", GivenOut);
    JWRITE(json, "meta", Meta);
    return json;
}

bool TBasePromoMeta::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, UsageCountLimit);
    READ_DECODER_VALUE_INSTANT(decoder, values, ActivityStart);
    READ_DECODER_VALUE_INSTANT(decoder, values, ActivityDeadline);
    READ_DECODER_VALUE_INSTANT(decoder, values, RemovingDeadline);
    READ_DECODER_VALUE(decoder, values, Prefix);
    READ_DECODER_VALUE(decoder, values, GivenOut);
    if (decoder.GetMeta() != -1) {
        READ_DECODER_VALUE_JSON(decoder, values, Meta, Meta);
    }
    return NJson::ParseField(Meta, "dictionary_context", TagDictionaryContext, false);
}

bool TBasePromoMeta::DeserializeFromJson(const NJson::TJsonValue& json) {
    if (!NJson::ParseField(json, "activity_start", ActivityStart, false)) {
        return false;
    }
    {
        TDuration duration;
        if (!NJson::ParseField(json, "duration", duration, false)) {
            return false;
        }
        if (duration) {
            ActivityDeadline = ActivityStart + duration;
        }
    }
    {
        TDuration duration;
        if (!NJson::ParseField(json, "removing_duration", duration, false)) {
            return false;
        }
        if (duration) {
            RemovingDeadline = ActivityStart + duration;
        }
    }
    return NJson::ParseField(json, "usage_limit", UsageCountLimit, false)
        && NJson::ParseField(json, "activity_start", ActivityStart, false)
        && NJson::ParseField(json, "activity_deadline", ActivityDeadline, false)
        && NJson::ParseField(json, "removing_deadline", RemovingDeadline, false)
        && NJson::ParseField(json, "prefix", Prefix, true)
        && NJson::ParseField(json, "given_out", GivenOut, false)
        && NJson::ParseField(json, "meta", Meta, false)
        && NJson::ParseField(Meta, "dictionary_context", TagDictionaryContext, false);
}
