#pragma once
#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/cgiparam/cgiparam.h>

#include <rtline/library/json/parse.h>
#include <rtline/library/storage/structured.h>
#include <rtline/library/storage/sql/query.h>
#include <rtline/util/types/accessor.h>

namespace NDrive {
    class IServer;
}

class TTagDictionaryContext {
public:
    class TElement {
        R_READONLY(TString, Key);
        R_READONLY(TString, Value);
    public:
        static NDrive::TScheme GetScheme(const NDrive::IServer* /*server*/) {
            NDrive::TScheme scheme;
            scheme.Add<TFSString>("key", "Ключ").SetRequired(true);
            scheme.Add<TFSString>("value", "Значение");
            return scheme;
        }

        bool FromJson(const NJson::TJsonValue& json) {
            return NJson::ParseField(json, "key", Key, true)
                && NJson::ParseField(json, "value", Value, true);
        }

        NJson::TJsonValue ToJson() const {
            NJson::TJsonValue json;
            NJson::InsertField(json, "key", Key);
            NJson::InsertField(json, "value", Value);
            return json;
        }
    };

    R_OPTIONAL(TString, TagName);
    R_READONLY(TVector<TElement>, Dictionary);

public:
    TTagDictionaryContext() = default;

    static NDrive::TScheme GetScheme(const NDrive::IServer* server);

    bool FromJson(const NJson::TJsonValue& json) {
        return NJson::ParseField(json, "tag_name", TagName, false)
            && NJson::ParseField(json, "parameters", Dictionary, false);
    }

    NJson::TJsonValue ToJson() const {
        NJson::TJsonValue json;
        NJson::InsertNonNull(json, "tag_name", TagName);
        NJson::InsertNonNull(json, "parameters", Dictionary);
        return json;
    }
};

class TBasePromoMeta {
    R_READONLY(ui32, UsageCountLimit, 1);
    R_READONLY(TInstant, ActivityStart, TInstant::Now());
    R_READONLY(TInstant, ActivityDeadline, TInstant::Max());
    R_READONLY(TString, Prefix);
    R_READONLY(TInstant, RemovingDeadline, TInstant::Max());
    R_READONLY(TString, GivenOut);
    R_READONLY(NJson::TJsonValue, Meta, NJson::JSON_MAP);
    R_READONLY(TTagDictionaryContext, TagDictionaryContext);

public:
    using TId = TString;
    class TDecoder : public TBaseDecoder {
        R_FIELD(i32, UsageCountLimit, -1);
        R_FIELD(i32, ActivityStart, -1);
        R_FIELD(i32, ActivityDeadline, -1);
        R_FIELD(i32, RemovingDeadline, -1);
        R_FIELD(i32, Prefix, -1);
        R_FIELD(i32, GivenOut, -1);
        R_FIELD(i32, Meta, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            UsageCountLimit = GetFieldDecodeIndex("usage_limit", decoderBase);
            ActivityStart = GetFieldDecodeIndex("activity_start", decoderBase);
            ActivityDeadline = GetFieldDecodeIndex("activity_deadline", decoderBase);
            RemovingDeadline = GetFieldDecodeIndex("removing_deadline", decoderBase);
            Prefix = GetFieldDecodeIndex("prefix", decoderBase);
            GivenOut = GetFieldDecodeIndex("given_out", decoderBase);
            Meta = GetFieldDecodeIndex("meta", decoderBase);
        }
    };

    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue SerializeToJsonReport() const;
    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    bool DeserializeFromJson(const NJson::TJsonValue& json);

    bool SetTagDictionaryContext(const NJson::TJsonValue& context) {
        Meta["dictionary_context"].InsertValue("parameters", context);
        return TagDictionaryContext.FromJson(Meta["dictionary_context"]);
    }
};

class TPromoCodeBuilder {
    R_FIELD(TString, Prefix);
    R_FIELD(bool, DigitsUsage, false);
    R_FIELD(bool, UpperLiteralsUsage, true);
    R_FIELD(bool, LowerLiteralsUsage, false);
    R_FIELD(ui32, CountSymbols, 10);
    R_READONLY(TString, UpperLiterals, "ABCDEFGHJKLMNPQRSTUVWXYZ");
    R_READONLY(TString, LowerLiterals, "abcdefghijkmnopqrstuvwxyz");
    R_READONLY(TString, DigitLiterals, "123456789");

protected:
    bool AddSymbol(const ui32 sym, TString& result) const;

public:
    bool DeserializeFromJson(const NJson::TJsonValue& data);
    bool Build(TString& result) const;
};


class TPromoCodeMeta : public TBasePromoMeta {
private:
    using TBase = TBasePromoMeta;
    R_FIELD(TString, Id);
    R_FIELD(ui32, ProfitType, 0);
    mutable TString Code;
public:

    const TString& GetCode() const {
        return Code;
    }

    class TDecoder : public TBasePromoMeta::TDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, Code, -1);
        R_FIELD(i32, ProfitType, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase)
            : TBasePromoMeta::TDecoder(decoderBase)
        {
            Id = GetFieldDecodeIndex("id", decoderBase);
            Code = GetFieldDecodeIndex("code", decoderBase);
            ProfitType = GetFieldDecodeIndex("type", decoderBase);
        }
    };

    bool RebuildCode(const TPromoCodeBuilder& builder) {
        return builder.Build(Code);
    }

    TPromoCodeMeta() = default;

    TPromoCodeMeta(const TBasePromoMeta& baseData)
        : TBasePromoMeta(baseData)
    {

    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
        if (!TBase::DeserializeWithDecoder(decoder, values, hContext)) {
            return false;
        }
        READ_DECODER_VALUE(decoder, values, Id);
        READ_DECODER_VALUE(decoder, values, Code);
        if (!Code) {
            return false;
        }
        READ_DECODER_VALUE(decoder, values, ProfitType);

        return true;
    }

    NJson::TJsonValue SerializeToJsonReport() const {
        NJson::TJsonValue json = TBase::SerializeToJsonReport();
        JWRITE(json, "id", Id);
        JWRITE(json, "type", ProfitType);
        JWRITE(json, "code", Code);
        return json;
    }

    NJson::TJsonValue SerializeToPublicReport() const {
        NJson::TJsonValue result = SerializeToJsonReport();
        result.EraseValue("code");
        return result;
    }

    NStorage::TTableRecord SerializeToTableRecord() const {
        NStorage::TTableRecord result = TBase::SerializeToTableRecord();
        if (Id == "") {
            result.Set("id", "uuid_generate_v4()");
        }
        else {
            result.Set("id", Id);
        }
        result.Set("type", ProfitType);
        result.Set("code", Code);
        return result;
    }

    const TString& GetInternalId() const {
        return Id;
    }

    static TString GetTableName() {
        return "promo_codes_meta";
    }

    static TString GetHistoryTableName() {
        return "promo_codes_meta_history";
    }

    bool operator!() const {
        return false;
    }
};

class TGeneratorContext : public TBasePromoMeta {
private:
    TPromoCodeBuilder CodeBuilder;
public:
    TPromoCodeBuilder& MutableCodeBuilder() {
        return CodeBuilder;
    }
    const TPromoCodeBuilder& GetCodeBuilder() const {
        return CodeBuilder;
    }

    void SetGivenOut(const TString& value) {
        SetProtectedGivenOut(value);
    }

    bool DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!CodeBuilder.DeserializeFromJson(data)) {
            return false;
        }
        return TBasePromoMeta::DeserializeFromJson(data);
    }
};

class TPromoCodesSearchContext {
    R_OPTIONAL(TInstant, Since);
    R_OPTIONAL(TInstant, Until);
    R_OPTIONAL(bool, ActiveOnly);
    R_OPTIONAL(TString, Prefix);
    R_OPTIONAL(TString, GivenOut);
    R_OPTIONAL(ui32, Count);
    R_OPTIONAL(TSet<TString>, Ids);
    R_FIELD(bool, UsageHistory, false);
    R_OPTIONAL(TString, Generator);
public:

    NSQL::TQueryOptions BuildCondition() const;

    bool IsEmpty() const {
        return !Since && !Until && !ActiveOnly && !Prefix && !GivenOut && !Ids && !Generator;
    }

    bool ReadFrom(const TCgiParameters& cgi);
};

class IPromoCodeMetaReport {
public:
    using TPtr = TAtomicSharedPtr<IPromoCodeMetaReport>;

    virtual ~IPromoCodeMetaReport() = default;

    virtual NJson::TJsonValue GetReport(const NDrive::IServer& server, bool withCodes = false) const = 0;
};

class IPromoProfitBase {
private:
    class TChatBuilder {
        R_READONLY(TString, ChatNotifier);

    public:
        TChatBuilder() = default;

        bool DeserializeFromJson(const NJson::TJsonValue& jsonValue);
        NJson::TJsonValue SerializeToJson() const;
        NDrive::TScheme GetScheme(const NDrive::IServer* server) const;
        NJson::TJsonValue GetDescriptor(const TString& userId, const TString& code, const NDrive::IServer& server) const;
    };

private:
    R_READONLY(bool, FirstOnly, false);
    R_READONLY(ui32, ActivateSum, 0);
    R_READONLY(TChatBuilder, ChatBuilder);
    R_READONLY(NJson::TJsonValue, Meta, NJson::JSON_MAP);
    R_READONLY(TTagDictionaryContext, TagDictionaryContext);

public:
    virtual ~IPromoProfitBase() = default;

    virtual bool DeserializeFromJson(const NJson::TJsonValue& info) {
        return NJson::ParseField(info, "first_only", FirstOnly, false)
            && NJson::ParseField(info, "activate_sum", ActivateSum, false)
            && NJson::ParseField(info, "meta", Meta, false)
            && NJson::ParseField(Meta, "dictionary_context", TagDictionaryContext, false)
            && ChatBuilder.DeserializeFromJson(info["chat"])
            && DoDeserializeFromJson(info);
    }

    virtual NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue result = DoSerializeToJson();
        NJson::InsertField(result, "type", GetType());
        NJson::InsertField(result, "first_only", FirstOnly);
        NJson::InsertField(result, "activate_sum", ActivateSum);
        NJson::InsertField(result, "chat", ChatBuilder.SerializeToJson());
        NJson::InsertField(result, "meta", Meta);
        return result;
    }

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const {
        NDrive::TScheme scheme = DoGetScheme(server);
        scheme.Add<TFSBoolean>("first_only", "Только на первые поездки").SetDefault(false);
        scheme.Add<TFSNumeric>("activate_sum", "Сумма для активации").SetMin(0).SetDefault(0);
        scheme.Add<TFSStructure>("chat", "Чат").SetStructure(ChatBuilder.GetScheme(server));

        NDrive::TScheme schemeMeta;
        schemeMeta.Add<TFSStructure>("dictionary_context", "Контекст").SetStructure(TTagDictionaryContext::GetScheme(server));
        scheme.Add<TFSStructure>("meta", "Meta").SetStructure(schemeMeta);
        return scheme;
    }

protected:
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& info) = 0;
    virtual NJson::TJsonValue DoSerializeToJson() const = 0;
    virtual TString GetType() const = 0;
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* /*server*/) const {
        return NDrive::TScheme();
    }
};
