#include "entities.h"

#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/billing/manager.h>

#include <library/cpp/digest/md5/md5.h>

#include <util/generic/guid.h>
#include <util/string/type.h>

bool IPromoProfit::Execute(const IPromoCodesManager::TApplyContext& context, const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, NJson::TJsonValue& report, const NDrive::IServer& server, const TString& generatorName, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const {
    return AvailableForObject(context, objectId, meta, server, generatorName, session) && Apply(objectId, historyUserId, meta, server, session, chatSession) && ConstructReport(objectId, meta, report, server, session);
}

bool IPromoProfit::ConstructReport(const TString& objectId, const TPromoCodeMeta& meta, NJson::TJsonValue& report, const NDrive::IServer& server, NDrive::TEntitySession& /*session*/) const {
    if (GetEntityType(server) == NEntityTagsManager::EEntityType::User) {
        auto descriptor = GetChatBuilder().GetDescriptor(objectId, meta.GetCode(), server);
        if (descriptor.IsDefined()) {
            report.InsertValue("chat", descriptor);
        }
    }
    return true;
}

bool IPromoProfit::AvailableForObject(const IPromoCodesManager::TApplyContext& context, const TString& objectId, const TPromoCodeMeta& /*meta*/, const NDrive::IServer& server, const TString& /*generatorName*/, NDrive::TEntitySession& session) const {
    if (GetActivateSum() && context.PaymentSum < GetActivateSum()) {
        session.SetErrorInfo("IPromoProfit::Execute", "incorrect sum" + ToString(context.PaymentSum) + "/" + ToString(GetActivateSum()), EDriveSessionResult::IncorrectRequest);
        auto locale = ELocalization::Rus;
        auto localMessage = Yensured(server.GetLocalization())->GetLocalString(locale, "incorrect_sum_promo_error");
        SubstGlobal(localMessage, "_PaymentSum_", Yensured(server.GetLocalization())->FormatPrice(locale, GetActivateSum(), { "units.short.rub" }));
        session.SetLocalizedMessageKey(localMessage);
        return false;
    }
    if (GetEntityType(server) == NEntityTagsManager::EEntityType::User) {
        auto gResult = server.GetDriveAPI()->GetUsersData()->FetchInfo(objectId, session);
        const TDriveUserData* resultPtr = gResult.GetResultPtr(objectId);
        if (!resultPtr) {
            session.SetErrorInfo("IPromoProfit::Execute", "incorrect user id:" + objectId, EDriveSessionResult::InconsistencyUser);
            return false;
        }
        if (GetFirstOnly() && !resultPtr->IsFirstRiding()) {
            session.SetErrorInfo("IPromoProfit::Execute", "not first riding", EDriveSessionResult::IncorrectRequest);
            session.SetLocalizedMessageKey("not_first_riding_promo_error");
            return false;
        }
    }
    if (GetEntityType(server) == NEntityTagsManager::EEntityType::Account) {
        if (!server.GetDriveAPI()->HasBillingManager()) {
            session.SetErrorInfo("IPromoProfit::Execute", "undefined manager", EDriveSessionResult::InternalError);
            return false;
        }
        ui32 id = 0;
        if (!TryFromString(objectId, id)) {
            session.SetErrorInfo("IPromoProfit::Execute", "incorrect account id:" + objectId, EDriveSessionResult::IncorrectRequest);
            return false;
        }
        auto resultPtr = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetAccountById(id);
        if (!resultPtr || !resultPtr->GetRecord()) {
            session.SetErrorInfo("IPromoProfit::Execute", "incorrect account id:" + objectId, EDriveSessionResult::InconsistencyUser);
            return false;
        }
        if (GetFirstOnly() && (resultPtr->GetRecord()->GetSoftLimit() != 0 || resultPtr->GetSpent() != 0)) {
            session.SetErrorInfo("IPromoProfit::Execute", "not first riding", EDriveSessionResult::IncorrectRequest);
            session.SetLocalizedMessageKey("not_first_riding_promo_error");
            return false;
        }
    }
    return true;
}

bool IPromoProfit::ApplyTagsMeta(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    TVector<TDBTag> tags;
    if (GetTagDictionaryContext().HasTagName()) {
        if (!server.GetDriveAPI()->GetEntityTagsManager(GetEntityType(server)).RestoreTags({ objectId }, { GetTagDictionaryContext().GetTagNameUnsafe() }, tags, session)) {
            return false;
        }
        if (tags.empty()) {
            TDBTag tag;
            tag.SetObjectId(objectId).SetData(server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(GetTagDictionaryContext().GetTagNameUnsafe()));
            tags.emplace_back(std::move(tag));
        }
        for (auto&& tag : tags) {
            auto tagImpl = tag.MutableTagAs<IDictionaryTag>();
            if (!tagImpl) {
                session.SetErrorInfo("IPromoProfit::Apply", "cannot cast to TUserDictionaryTag", EDriveSessionResult::InternalError);
                return false;
            }
            for (const auto& parameter : GetTagDictionaryContext().GetDictionary()) {
                tagImpl->SetField(parameter.GetKey(), parameter.GetValue());
            }
        }
    }
    if (meta.GetTagDictionaryContext().HasTagName()) {
        if (!GetTagDictionaryContext().HasTagName() || GetTagDictionaryContext().GetTagNameUnsafe() != meta.GetTagDictionaryContext().GetTagNameUnsafe()) {
            TVector<TDBTag> metaTags;
            if (!server.GetDriveAPI()->GetEntityTagsManager(GetEntityType(server)).RestoreTags({ objectId }, { meta.GetTagDictionaryContext().GetTagNameUnsafe() }, metaTags, session)) {
                return false;
            }
            if (metaTags.empty()) {
                TDBTag tag;
                tag.SetObjectId(objectId).SetData(server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(meta.GetTagDictionaryContext().GetTagNameUnsafe()));
                metaTags.emplace_back(std::move(tag));
            }
            tags.insert(tags.end(), metaTags.begin(), metaTags.end());
        }
        for (auto&& tag : tags) {
            auto tagImpl = tag.MutableTagAs<IDictionaryTag>();
            if (!tagImpl) {
                session.SetErrorInfo("IPromoProfit::Apply", "cannot cast to TUserDictionaryTag", EDriveSessionResult::InternalError);
                return false;
            }
            for (const auto& parameter : meta.GetTagDictionaryContext().GetDictionary()) {
                tagImpl->SetField(parameter.GetKey(), parameter.GetValue());
            }
        }
    }
    for (auto&& tag : tags) {
        if (tag.GetTagId()) {
            if (!server.GetDriveAPI()->GetEntityTagsManager(GetEntityType(server)).UpdateTagData(tag, historyUserId, session)) {
                return false;
            }
        } else {
            if (!server.GetDriveAPI()->GetEntityTagsManager(GetEntityType(server)).AddTag(tag.GetData(), historyUserId, tag.GetObjectId(), &server, session)) {
                return false;
            }
        }
    }
    return true;
}

bool IPromoProfit::Apply(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const {
    if (!ApplyTagsMeta(objectId, historyUserId, meta, server, session) || !DoApply(objectId, historyUserId, meta, server, session, chatSession)) {
        return false;
    }
    if (GetEntityType(server) == NEntityTagsManager::EEntityType::User) {
        auto descriptor = GetChatBuilder().GetDescriptor(objectId, meta.GetCode(), server);
        if (descriptor.IsDefined()) {
            TString chatId;
            TString chatTopic;
            IChatRobot::ParseTopicLink(descriptor["id"].GetString(), chatId, chatTopic);
            IChatRobot::TPtr chatRobot = server.GetChatRobot(chatId);
            if (!!chatRobot) {
                return chatRobot->EnsureChat(objectId, chatTopic, chatSession, false);
            }
        }
    }
    return true;
}

NJson::TJsonValue TPromoCodeMetaReport::GetReport(const NDrive::IServer& server, bool withCodes) const {
    NJson::TJsonValue result = Meta.SerializeToPublicReport();
    if (!!Usage) {
        result.InsertValue("remaining", Usage->GetRemainingUsageCount());
    }
    JWRITE(result, "generator", Generator);
    if (Generator) {
        auto action = server.GetDriveAPI()->GetRolesManager()->GetAction(Generator);
        if (action.Defined()) {
            JWRITE(result, "description", (*action)->GetDescription());
        }
    }
    if (withCodes) {
        JWRITE(result, "code", Meta.GetCode());
    }
    if (UsageHistory.size()) {
        TSet<TString> userIds;
        for (const auto& ev : UsageHistory) {
            if (ev.GetHistoryUserId()) {
                userIds.insert(ev.GetHistoryUserId());
            }
            if (ev.GetHistoryOriginatorId()) {
                userIds.insert(ev.GetHistoryOriginatorId());
            }
        }
        auto usersData = server.GetDriveAPI()->GetUsersData()->FetchInfo(userIds);
        NJson::TJsonValue historyJson = NJson::JSON_ARRAY;
        for (const auto& ev : UsageHistory) {
            NJson::TJsonValue eventJson = ev.BuildReportItem();
            if (ev.GetHistoryUserId()) {
                const TDriveUserData* userData = usersData.GetResultPtr(ev.GetHistoryUserId());
                if (!!userData) {
                    eventJson.InsertValue("user_data_full", userData->GetReport());
                }
            }
            if (ev.GetHistoryOriginatorId()) {
                const TDriveUserData* userData = usersData.GetResultPtr(ev.GetHistoryUserId());
                if (!!userData) {
                    eventJson.InsertValue("originator_data", userData->GetReport());
                }
            }
            historyJson.AppendValue(std::move(eventJson));
        }
        JWRITE(result, "history", historyJson);
    }
    return result;
}

NDrive::TScheme NPromoCodes::GetGeneratorContextScheme(const IServerBase& baseServer) {
    NDrive::TScheme scheme;

    auto server = baseServer.GetAs<NDrive::IServer>();
    if (server) {
        scheme.Add<TFSVariants>("generator", "Генератор").SetVariants(server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionNamesWithType<IPromoCodeGenerator>(/*reportDeprecated=*/false)).SetRequired(true);
    } else {
        scheme.Add<TFSString>("generator", "Генератор").SetRequired(true);
    }
    scheme.Add<TFSString>("prefix", "Префикс").SetRequired(true);
    scheme.Add<TFSNumeric>("count", "Количество промокодов").SetRequired(true);
    scheme.Add<TFSNumeric>("usage_limit", "Количество использований").SetRequired(true);

    scheme.Add<TFSNumeric>("count_symbols", "Количество символов").SetRequired(true);
    scheme.Add<TFSBoolean>("upper_literals_usage", "Использовать буквы верхнего регистра").SetDefault(true);
    scheme.Add<TFSBoolean>("lower_literals_usage", "Использовать буквы нижнего регистра").SetDefault(true);
    scheme.Add<TFSBoolean>("digits_usage", "Использовать цифры").SetDefault(true);

    scheme.Add<TFSNumeric>("activity_start", "Начало использования").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
    scheme.Add<TFSNumeric>("activity_deadline", "Окончание использования").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
    scheme.Add<TFSNumeric>("removing_deadline", "Время удаления").SetVisual(TFSNumeric::EVisualType::DateTime).SetDefault((ModelingNow() + TDuration::Days(365)).Seconds());
    scheme.Add<TFSString>("given_out", "Кому отдано");
    scheme.Add<TFSJson>("meta", "Meta").SetDefault("{}");

    scheme.Add<TFSString>("upper_literals", "Буквы верхнего регистра").SetDefault("ABCDEFGHJKLMNPQRSTUVWXYZ");
    scheme.Add<TFSString>("lower_literals", "Буквы нижнего регистра").SetDefault("abcdefghijkmnopqrstuvwxyz");
    scheme.Add<TFSString>("digit_literals", "Используемые цифры").SetDefault("123456789");

    return scheme;
}
