#pragma once

#include <drive/backend/promo_codes/common/action.h>
#include <drive/backend/promo_codes/common/entities.h>
#include <drive/backend/promo_codes/common/manager.h>

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/tags/abstract.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/types/accessor.h>

class IPromoProfit : public IPromoProfitBase {
public:
    using TPtr = TAtomicSharedPtr<IPromoProfit>;
    using TFactory = NObjectFactory::TParametrizedObjectFactory<IPromoProfit, TString>;
public:
    bool Execute(const IPromoCodesManager::TApplyContext& context, const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, NJson::TJsonValue& report, const NDrive::IServer& server, const TString& generatorName, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const;

protected:
    virtual bool ConstructReport(const TString& objectId, const TPromoCodeMeta& /*meta*/, NJson::TJsonValue& report, const NDrive::IServer& server, NDrive::TEntitySession& /*session*/) const;
    virtual bool AvailableForObject(const IPromoCodesManager::TApplyContext& context, const TString& objectId, const TPromoCodeMeta& meta, const NDrive::IServer& server, const TString& generatorName, NDrive::TEntitySession& session) const;
    virtual bool DoApply(const TString& userId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const = 0;
    virtual NEntityTagsManager::EEntityType GetEntityType(const NDrive::IServer& server) const = 0;

private:
    bool Apply(const TString& userId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const;

    // deprecated
    bool ApplyTagsMeta(const TString& objectId, const TString& historyUserId, const TPromoCodeMeta& meta, const NDrive::IServer& server, NDrive::TEntitySession& session) const;
};

class TPromoProfitType {
private:
    R_FIELD(ui32, Id, Max<ui32>());
    R_FIELD(TString, IdentifierStr);
    R_FIELD(TString, Generator);
    R_FIELD(IPromoProfit::TPtr, ProfitDescription);
public:
    using TId = ui32;

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, IdentifierStr, -1);
        R_FIELD(i32, ProfitDescription, -1);
        R_FIELD(i32, Generator, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            Id = GetFieldDecodeIndex("id", decoderBase);
            IdentifierStr = GetFieldDecodeIndex("name", decoderBase);
            ProfitDescription = GetFieldDecodeIndex("profit_description", decoderBase);
            Generator = GetFieldDecodeIndex("generator", decoderBase);
        }
    };

    ui32 GetInternalId() const {
        return Id;
    }

    static TString GetTableName() {
        return "promo_code_types";
    }

    static TString GetHistoryTableName() {
        return "promo_code_types_history";
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, Id);
        READ_DECODER_VALUE(decoder, values, IdentifierStr);
        READ_DECODER_VALUE(decoder, values, Generator);
        NJson::TJsonValue jsonDescription;
        READ_DECODER_VALUE_JSON(decoder, values, jsonDescription, ProfitDescription);
        if (!jsonDescription.Has("type")) {
            return false;
        }
        ProfitDescription = IPromoProfit::TFactory::Construct(jsonDescription["type"].GetString());
        if (!ProfitDescription) {
            return false;
        }
        return ProfitDescription->DeserializeFromJson(jsonDescription);
    }

    NStorage::TTableRecord SerializeToTableRecord() const {
        NStorage::TTableRecord result;
        if (Id != Max<ui32>()) {
            result.Set("id", Id);
        }
        result.Set("name", IdentifierStr);
        result.Set("generator", Generator);
        result.Set("profit_description", ProfitDescription->SerializeToJson().GetStringRobust());
        return result;
    }

    bool operator!() const {
        return !Generator || !ProfitDescription;
    }
};

class TPromoCodeUsage {
private:
    R_FIELD(TString, Id);
    R_FIELD(i32, RemainingUsageCount, 0);
public:
    using TId = TString;

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, RemainingUsageCount, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            Id = GetFieldDecodeIndex("id", decoderBase);
            RemainingUsageCount = GetFieldDecodeIndex("remaining", decoderBase);
        }
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, Id);
        READ_DECODER_VALUE(decoder, values, RemainingUsageCount);
        return true;
    }

    NStorage::TTableRecord SerializeToTableRecord() const {
        NStorage::TTableRecord result;
        result.Set("id", Id);
        result.Set("remaining", RemainingUsageCount);
        return result;
    }

    const TString& GetInternalId() const {
        return Id;
    }

    static TString GetTableName() {
        return "promo_codes_usage";
    }

    static TString GetHistoryTableName() {
        return "promo_codes_usage_history";
    }

    bool operator!() const {
        return false;
    }

    void DoBuildReportItem(NJson::TJsonValue& json) const {
        json.InsertValue("promo_id", Id);
        json.InsertValue("remaining", RemainingUsageCount);
    }
};

class TPromoCodeMetaReport : public IPromoCodeMetaReport {
private:
    R_READONLY(TPromoCodeMeta, Meta);
    R_READONLY(TString, Generator);
    R_FIELD(TVector<TObjectEvent<TPromoCodeUsage>>, UsageHistory);
    THolder<TPromoCodeUsage> Usage;
public:

    TPromoCodeMetaReport(const TPromoCodeMeta& meta, const TString& generator)
        : Meta(meta)
        , Generator(generator)
    {

    }

    TPromoCodeMetaReport(const TPromoCodeMeta& meta, const TString& generator, const TPromoCodeUsage& usage)
        : Meta(meta)
        , Generator(generator)
    {
        Usage.Reset(new TPromoCodeUsage(usage));
    }

    virtual NJson::TJsonValue GetReport(const NDrive::IServer& server, bool withCodes = false) const override;
};

namespace NPromoCodes {
    NDrive::TScheme GetGeneratorContextScheme(const IServerBase& baseServer);
}
