#include "manager.h"

#include <drive/backend/database/drive_api.h>

#include <library/cpp/digest/md5/md5.h>

TPromoCodesUsageHistoryManager::TOptionalEvents TPromoCodesUsageHistoryManager::GetEventsByInternalId(const TString& keyId, const TInstant since, NDrive::TEntitySession& session) const {
    return TBase::GetEvents<TObjectContainer>({}, TRange<TInstant>(since), session, TQueryOptions().AddGenericCondition("id", keyId));
}

TPromoCodesUsageHistoryManager::TOptionalEvents TPromoCodesUsageHistoryManager::GetEventsByHistoryUserId(const TString& keyId, const TInstant since, NDrive::TEntitySession& session) const {
    return TBase::GetEvents<TObjectContainer>({}, TRange<TInstant>(since), session, TQueryOptions().AddGenericCondition("history_user_id", keyId));
}

bool TPromoCodesManager::CleanOld(const TString& userId, NDrive::TEntitySession& session) const {
    NSQL::TQueryOptions condition;
    condition.SetGenericCondition("removing_deadline", MakeRange<ui64>(Nothing(), ModelingNow().Seconds()));
    condition.SetLimit(Config.GetCleanPackSize());
    auto objects = Meta->GetObjects(session, condition);
    if (!objects) {
        return false;
    }
    TSet<TString> idsRemove;
    Transform(objects->begin(), objects->end(), std::inserter(idsRemove, idsRemove.begin()), [](const auto& meta) { return meta.GetId(); });

    auto usages = Usage->GetObjects(idsRemove, session);
    if (!usages) {
        return false;
    }
    TSet<TString> usageIdsRemove;
    Transform(usages->begin(), usages->end(), std::inserter(usageIdsRemove, usageIdsRemove.begin()), [](const auto& usage) { return usage.GetId(); });

    if (!Usage->RemoveObjects(usageIdsRemove, userId, session) ||
        !Meta->RemoveObjects(idsRemove, userId, session))
    {
        return false;
    }

    NStorage::TObjectRecordsSet<TPromoProfitType> records;
    auto tableTypes = Server.GetDriveAPI()->GetDatabase().GetTable(TPromoProfitType::GetTableName());
    if (!tableTypes->GetRows("id NOT IN (SELECT DISTINCT type FROM promo_codes_meta)", records, session.GetTransaction())->IsSucceed()) {
        return false;
    }
    TSet<ui32> idsRemoveTypes;
    for (auto&& i : records) {
        idsRemoveTypes.emplace(i.GetId());
    }
    if (!Types->RemoveObjects(idsRemoveTypes, userId, session)) {
        return false;
    } else if (idsRemove.size() || idsRemoveTypes.size()) {
        NOTICE_LOG << "Removed: " << idsRemove.size() << " from usage/meta AND " << idsRemoveTypes.size() << " from types" << Endl;
    }
    return true;
}

bool TPromoCodesManager::Process(IMessage* message) {
    TRegularDBServiceMessage* dbServiceMessage = dynamic_cast<TRegularDBServiceMessage*>(message);
    if (dbServiceMessage) {
        auto session = Server.GetDriveAPI()->template BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
        if (!CleanOld(dbServiceMessage->GetUserId(), session) || !session.Commit()) {
            dbServiceMessage->MutableReport().AddMessage<TLOG_ERR>("promo_cleaning", session.GetStringReport());
        }
        return true;
    }
    return false;
}

TPromoCodesManager::TPromoCodesManager(const IHistoryContext& context, const TPromoCodesManagerConfig& config, const NDrive::IServer& server)
    : Config(config)
    , Server(server)
{
    Types.Reset(new TPromoCodeTypes(context));
    Meta.Reset(new TPromoCodesMeta(context));
    Usage.Reset(new TPromoCodesUsage(context));
    RegisterGlobalMessageProcessor(this);
}

TPromoCodesManager::~TPromoCodesManager() {
    UnregisterGlobalMessageProcessor(this);
}

TPromoCodesManager::TSelectOptions::TSelectOptions(const TPromoCodesSearchContext& filter)
    : Options(filter.BuildCondition().SetOrderBy({"id"}))
    , Limit(filter.GetCountDef(Max<ui32>()))
    , ActiveOnly(filter.GetActiveOnlyDef(false))
    , UseHistory(filter.IsUsageHistory())
{
}

void TPromoCodesManager::TSelectOptions::UpdateLimits(const ui32 offset, const ui32 limit, const ui32 originLimit) {
    Options.SetOffset(offset).SetLimit(limit);
    if (originLimit) {
        Limit = originLimit;
    }
}

bool TPromoCodesManager::GetPromoCodes(const TPromoCodesSearchContext& filter, TUserPermissions::TPtr permissions, TVector<IPromoCodeMetaReport::TPtr>& result, NDrive::TEntitySession& session) const {
    TSelectOptions options(filter);
    if (filter.HasGenerator()) {
        auto types = Types->GetObjectsByField(filter.GetGeneratorRef(), "generator", session);
        if (!types) {
            return false;
        }
        if (types->empty()) {
            return true;
        }
        TSet<ui32> typeCondition;
        Transform(types->begin(), types->end(), std::inserter(typeCondition, typeCondition.begin()), [](const auto& type) { return type.GetId(); });
        options.SetGenericCondition("type", typeCondition);
    }
    ui32 offset = 0;
    const ui32 limit = filter.GetCountDef(Max<ui32>());
    const ui32 batch = Server.GetSettings().GetValueDef<ui32>("promo.select_batch_size", 1000);
    while (limit > result.size()) {
        options.UpdateLimits(offset, batch, Min<ui32>(batch, limit - result.size()));
        TVector<IPromoCodeMetaReport::TPtr> subResult;
        if (!GetPromoCodesImpl(options, permissions, subResult, session)) {
            return false;
        }
        if (subResult.empty()) {
            return true;
        }
        std::move(subResult.begin(), subResult.end(), std::back_inserter(result));
        offset += batch;
    }
    return true;
}

bool TPromoCodesManager::GetPromoCodesImpl(const TPromoCodesManager::TSelectOptions& options, TUserPermissions::TPtr permissions, TVector<IPromoCodeMetaReport::TPtr>& result, NDrive::TEntitySession& session) const {
    auto metaInfos = Meta->GetObjects(session, options.Options);
    if (!metaInfos) {
        return false;
    }
    TSet<ui32> typeIds;
    Transform(metaInfos->begin(), metaInfos->end(), std::inserter(typeIds, typeIds.begin()), [](const auto& meta) { return meta.GetProfitType(); });
    auto profitTypes = Types->GetObjects(typeIds, session);
    if (!profitTypes) {
        return false;
    }
    TMap<ui32, TString> generators;
    Transform(profitTypes->begin(), profitTypes->end(), std::inserter(generators, generators.begin()), [](const auto& type) { return std::make_pair(type.GetId(), type.GetGenerator()); });

    TSet<TString> ids;
    TVector<std::pair<TPromoCodeMeta, TString>> metaWithGenerators;
    for (auto&& meta : *metaInfos) {
        auto it = generators.find(meta.GetProfitType());
        if (it != generators.end() && !it->second.empty() &&
            permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Observe, TAdministrativeAction::EEntity::PromoCodes, {}, it->second))
        {
            ids.insert(meta.GetId());
            metaWithGenerators.emplace_back(std::move(meta), it->second);
        }
    }

    auto usageInfos = Usage->GetObjects(ids, session);
    if (!usageInfos) {
        return false;
    }
    TMap<TString, TPromoCodeUsage> usageInfo;
    Transform(usageInfos->begin(), usageInfos->end(), std::inserter(usageInfo, usageInfo.begin()), [](auto& item) { return std::make_pair(item.GetId(), std::move(item)); });
    auto itUsage = usageInfo.begin();
    for (auto&& meta : metaWithGenerators) {
        THolder<TPromoCodeMetaReport> metaReport;
        if (!Advance(itUsage, usageInfo.end(), meta.first.GetId())) {
            if (!options.ActiveOnly) {
                metaReport = MakeHolder<TPromoCodeMetaReport>(meta.first, meta.second, TPromoCodeUsage().SetId(meta.first.GetId()));
            }
        } else {
            metaReport = MakeHolder<TPromoCodeMetaReport>(meta.first, meta.second, itUsage->second);
        }
        if (metaReport
            && options.UseHistory
            && permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::ObserveStructure, TAdministrativeAction::EEntity::PromoCodes, {}, meta.second))
        {
            auto usageData = Usage->GetHistoryManager().GetEventsByInternalId(meta.first.GetId(), TInstant::Zero(), session);
            if (!usageData) {
                return false;
            }
            metaReport->MutableUsageHistory() = *usageData;
        }
        if (metaReport) {
            result.emplace_back(metaReport.Release());
        }

        if (result.size() >= options.Limit) {
            break;
        }
    }
    return true;
}

bool TPromoCodesManager::GiveOutCodes(const TSet<TString>& ids, const TString& givenOutInfo, TUserPermissions::TPtr permissions, TVector<TPromoCodeMeta>& codes, NDrive::TEntitySession& session) const {
    if (!givenOutInfo) {
        session.SetErrorInfo("GiveOutCodes", "empty given out info", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    auto metaInfos = Meta->GetObjects(ids, session);
    if (!metaInfos) {
        return false;
    }
    TSet<ui32> typeIds;
    Transform(metaInfos->begin(), metaInfos->end(), std::inserter(typeIds, typeIds.begin()), [](const auto& meta) { return meta.GetProfitType(); });
    auto profitTypes = Types->GetObjects(typeIds, session);
    if (!profitTypes) {
        return false;
    }
    TMap<ui32, TString> generators;
    Transform(profitTypes->begin(), profitTypes->end(), std::inserter(generators, generators.begin()), [](const auto& type) { return std::make_pair(type.GetId(), type.GetGenerator()); });

    for (const auto& meta : *metaInfos) {
        auto it = generators.find(meta.GetProfitType());
        if (it == generators.end() || !it->second) {
            session.SetErrorInfo("GenerateCodes", "incorrect promo info", EDriveSessionResult::InconsistencySystem);
            return false;
        }
        if (!permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::PromoCodes, {}, it->second)) {
            session.SetErrorInfo("GenerateCodes", "no permissions for given out promo type: " + it->second, EDriveSessionResult::NoUserPermissions);
            return false;
        }
    }

    return Meta->GiveOutCodes(ids, givenOutInfo, permissions->GetUserId(), codes, session);
}

bool TPromoCodesManager::GiveOutCodes(const TString& generator, const ui32 count, const TString& givenOutInfo, TUserPermissions::TPtr permissions, TVector<TPromoCodeMeta>& codes, NDrive::TEntitySession& session) const {
    if (!givenOutInfo) {
        session.SetErrorInfo("GiveOutCodes", "empty given out info", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    if (!permissions->CheckAdministrativeActions(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::PromoCodes, {}, generator)) {
        session.SetErrorInfo("GiveOutCodes", "no permissions for given out promo type or incorrect generator: " + generator, EDriveSessionResult::NoUserPermissions);
        session.SetLocalizedMessageKey("no_permissions_giveout");
        return false;
    }

    return Meta->GiveOutCodes(generator, count, givenOutInfo, permissions->GetUserId(), codes, session);
}

bool TPromoCodesManager::RemoveCodes(const TSet<TString>& ids, const bool force, const TString& userId, NDrive::TEntitySession& session) const {
    TMap<TString, TPromoCodeMeta> objects;
    {
        auto objectsMeta = Meta->GetObjects(ids, session);
        if (!objectsMeta) {
            return false;
        }
        Transform(objectsMeta->begin(), objectsMeta->end(), std::inserter(objects, objects.begin()), [](const auto& meta) { return std::make_pair(meta.GetId(), std::move(meta)); });
    }
    auto usages = Usage->GetObjects(ids, session);
    if (!usages) {
        return false;
    }
    TMap<TString, i32> objectUsage;
    Transform(usages->begin(), usages->end(), std::inserter(objectUsage, objectUsage.begin()), [](const auto& usage) { return std::make_pair(usage.GetId(), usage.GetRemainingUsageCount()); });

    if (!force) {
        TVector<TString> alreadyUsed;
        for (const auto& meta : objects) {
            auto itUsage = objectUsage.find(meta.second.GetId());
            if (itUsage == objectUsage.end() || itUsage->second != (const i32)meta.second.GetUsageCountLimit()) {
                alreadyUsed.emplace_back(meta.second.GetCode());
                break;
            }
        }
        if (alreadyUsed.size()) {
            session.SetErrorInfo("RemoveCodes", JoinSeq(", ", alreadyUsed) + " already been used", EDriveSessionResult::IncorrectRequest);
            return false;
        }
    }

    return Usage->RemoveObjects(MakeSet(NContainer::Keys(objectUsage)), userId, session) && Meta->RemoveObjects(MakeSet(NContainer::Keys(objects)), userId, session);
}

bool TPromoCodesManager::GenerateCodes(const ui32 count, const IPromoCodeGenerator& generator, const TGeneratorContext& context, const TString& userId, TVector<TPromoCodeMeta>& codes, NDrive::TEntitySession& session) const {
    IPromoProfit::TPtr profit = generator.BuildPromoProfit();
    if (!profit) {
        session.SetErrorInfo("GenerateCodes", "incorrect profit", EDriveSessionResult::InternalError);
        return false;
    }
    TString typeName = generator.GetName();
    const TString description = profit->SerializeToJson().GetStringRobust();
    const TString md5sum = MD5::Calc(description);
    auto expectedTypeInfo = Types->GetObjectsByField(typeName + "_" + md5sum, "name", session);
    if (!expectedTypeInfo) {
        return false;
    }
    TPromoProfitType typeInfo;
    if (expectedTypeInfo->empty()) {
        TPromoProfitType typeInfoImpl;
        typeInfoImpl.SetIdentifierStr(typeName + "_" + md5sum).SetProfitDescription(profit).SetGenerator(typeName);
        NStorage::TObjectRecordsSet<TPromoProfitType> typesNew;
        if (!Types->UpsertObject(typeInfoImpl, userId, session, &typesNew)) {
            return false;
        }
        if (typesNew.size() == 0) {
            session.SetErrorInfo("GenerateCodes", "cannot register type: " + typeInfoImpl.GetIdentifierStr(), EDriveSessionResult::InternalError);
            return false;
        }
        typeInfo = typesNew.front();
    } else {
        typeInfo = expectedTypeInfo->front();
    }
    for (ui32 i = 0; i < count;) {
        const ui32 iNext = Min(i + Config.GetGenerationPackSize(), count);
        TVector<TPromoCodeMeta> metaPack;
        for (ui32 j = i; j < iNext; ++j) {
            TPromoCodeMeta meta(context);
            if (!meta.RebuildCode(context.GetCodeBuilder())) {
                session.SetErrorInfo("GenerateCodes", "Cannot build code", EDriveSessionResult::InternalError);
                return false;
            }
            meta.SetProfitType(typeInfo.GetId());
            metaPack.emplace_back(std::move(meta));
        }
        NStorage::TObjectRecordsSet<TPromoCodeMeta> records;
        if (!Meta->AddObjects(metaPack, userId, session, &records)) {
            return false;
        }
        if (records.size() != iNext - i) {
            session.SetErrorInfo("GenerateCodes", "Cannot add new code", EDriveSessionResult::InternalError);
            session.SetLocalizedMessageKey("cannot_add_new_code");
            return false;
        }

        TVector<TPromoCodeUsage> usagePack;
        for (auto&& obj : records) {
            TPromoCodeUsage usage;
            usage.SetRemainingUsageCount(context.GetUsageCountLimit());
            usage.SetId(obj.GetId());
            usagePack.emplace_back(std::move(usage));
        }
        if (!Usage->AddObjects(usagePack, userId, session)) {
            return false;
        }
        i = iNext;
        codes.insert(codes.end(), records.begin(), records.end());
    }
    return true;
}

bool TPromoCodesManager::GetPromoProfit(const TString& code, TAtomicSharedPtr<IPromoProfitBase>& result, NDrive::TEntitySession& session) const {
    auto data = GetMetaByCode(code, session);
    if (!data) {
        return false;
    }
    if (data->empty()) {
        session.SetErrorInfo("GetPromoProfit", "Incorrect code", EDriveSessionResult::IncorrectRequest);
        ERROR_LOG << "apply promocode : incorrect code" << Endl;
        return false;
    }

    auto dataTypes = Types->GetObjects({data->front().GetProfitType()}, session);
    if (!dataTypes) {
        return false;
    }
    if (dataTypes->empty()) {
        session.SetErrorInfo("GetPromoProfit", "Incorrect promo type", EDriveSessionResult::InconsistencySystem);
        ERROR_LOG << "apply promocode : incorrect promo type" << Endl;
        return false;
    }

    result = dataTypes->front().GetProfitDescription();
    return true;
}

TMaybe<TVector<TPromoCodeMeta>> TPromoCodesManager::GetMetaByCode(const TString& code, NDrive::TEntitySession& session) const {
    return Meta->GetObjectsByField(code, "code", session);
}

bool TPromoCodesManager::ApplyCode(const IPromoCodesManager::TApplyContext& context, const TString& objectId, TUserPermissions::TPtr permissions, NJson::TJsonValue& jsonReport, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const {
    if (!permissions) {
        session.SetErrorInfo("ApplyCode", "no permissions for using with promo code", EDriveSessionResult::NoUserPermissions);
        session.SetLocalizedMessageKey("no_promo_code_permissions");
        return false;
    }

    auto meta = GetMetaByCode(context.Code, session);
    if (!meta) {
        return false;
    }
    if (meta->empty()) {
        session.SetErrorInfo("ApplyCode", "Incorrect code", EDriveSessionResult::IncorrectRequest);
        session.SetLocalizedMessageKey("incorrect_promo_code");
        return false;
    }
    auto& data = meta->front();
    if (data.GetActivityStart() > ModelingNow() || !data.GetGivenOut()) {
        session.SetErrorInfo("apply promo code", "incorrect code", EDriveSessionResult::IncorrectRequest);
        session.SetLocalizedMessageKey("incorrect_promo_code");
        return false;
    }

    if (data.GetActivityDeadline() < ModelingNow()) {
        session.SetErrorInfo("apply promo code", "expired code", EDriveSessionResult::IncorrectRequest);
        session.SetLocalizedMessageKey("promotional_completed");
        return false;
    }

    auto usages = Usage->GetObjects({data.GetId()}, session);
    if (!usages) {
        return false;
    }
    if (usages->empty()) {
        session.SetErrorInfo("apply promo code", "incorrect usage data", EDriveSessionResult::InconsistencySystem);
        session.SetLocalizedMessageKey("promotional_completed");
        return false;
    }

    auto& dataUsage = usages->front();
    if (dataUsage.GetRemainingUsageCount() <= 1) {
        if (!Usage->RemoveObjects({dataUsage.GetId()}, permissions->GetUserId(), session)) {
            session.SetLocalizedMessageKey("promotional_completed");
            return false;
        }
    } else {
        dataUsage.SetRemainingUsageCount(dataUsage.GetRemainingUsageCount() - 1);
        if (!Usage->UpsertObject(dataUsage, permissions->GetUserId(), session)) {
            return false;
        }
    }

    auto dataTypes = Types->GetObjects({data.GetProfitType()}, session);
    if (!dataTypes) {
        return false;
    }
    if (dataTypes->empty() || !dataTypes->front()) {
        session.SetErrorInfo("apply promo code", "incorrect promo type", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    auto& dataType = dataTypes->front();

    if (!permissions->GetAvailablePromoCodeTypes().contains(dataType.GetGenerator())) {
        session.SetErrorInfo("CheckUserPermissions", "no permissions for using with promo code", EDriveSessionResult::NoUserPermissions);
        session.SetLocalizedMessageKey("no_promo_code_permissions");
        return false;
    }

    return dataType.GetProfitDescription()->Execute(context, objectId, permissions->GetUserId(), data, jsonReport, Server, dataType.GetGenerator(), session, chatSession);
}

bool TPromoCodesMeta::GiveOutCodes(const TString& generator, const ui32 count, const TString& giveOutInfo, const TString& userId, TVector<TPromoCodeMeta>& result, NDrive::TEntitySession& session) const {
    ui32 typeId = 0;
    {
        TString typeCondition = "SELECT MAX(id) as type_id from " + TPromoProfitType::GetTableName() + " WHERE generator='" + generator + "'";
        TRecordsSet records;
        auto execResult = session->Exec(typeCondition, &records);
        if (!execResult || !execResult->IsSucceed() || records.size() != 1) {
            session.SetErrorInfo("PromoCodesMeta::GiveOutCodes", "SELECT type_id failed", EDriveSessionResult::IncorrectRequest);
            return false;
        }

        if (!records.begin()->TryGet("type_id", typeId)) {
            session.SetErrorInfo("max_promo_type", "unknown type_id", EDriveSessionResult::InternalError);
            return false;
        }
    }
    ui32 availableCount = 0;
    {
        TString countCondition = "SELECT count(1) as count from " + TPromoCodeMeta::GetTableName() + " WHERE given_out='' AND type=" + ::ToString(typeId);
        TRecordsSet records;
        auto execResult = session->Exec(countCondition, &records);
        if (!execResult || !execResult->IsSucceed() || records.size() != 1) {
            session.SetErrorInfo("PromoCodesMeta::GiveOutCodes", "SELECT count failed", EDriveSessionResult::InternalError);
            return false;
        }

        if (!records.begin()->TryGet("count", availableCount)) {
            session.SetErrorInfo("codes count", "incorrect codes count", EDriveSessionResult::InternalError);
            return false;
        }
    }
    if (count > availableCount) {
        session.SetErrorInfo("codes count", "not enough codes", EDriveSessionResult::IncorrectRequest);
        return false;
    }
    ui32 offset = RandomNumber<ui32>(availableCount - count + 1);
    return GiveOutCodes("id IN (SELECT id FROM " + TPromoCodeMeta::GetTableName() + " WHERE type = " + ::ToString(typeId) + " AND given_out='' LIMIT " + ::ToString(count) + " OFFSET " + ::ToString(offset) + " FOR UPDATE)", giveOutInfo, userId, result, session);
}

bool TPromoCodesMeta::GiveOutCodes(const TSet<TString>& ids, const TString& giveOutInfo, const TString& userId, TVector<TPromoCodeMeta>& result, NDrive::TEntitySession& session) const {
    return ids.empty() || GiveOutCodes("id IN (" + session.GetTransaction()->Quote(ids) + ") AND given_out = ''", giveOutInfo, userId, result, session);
}

bool TPromoCodesMeta::GiveOutCodes(const TString& idContition, const TString& giveOutInfo, const TString& userId, TVector<TPromoCodeMeta>& result, NDrive::TEntitySession& session) const {
    auto table = HistoryManager->GetDatabase().GetTable(TPromoCodeMeta::GetTableName());

    const TString update = "given_out=" + session.GetTransaction()->Quote(giveOutInfo);
    NStorage::TObjectRecordsSet<TPromoCodeMeta> records;
    if (!table->UpdateRow(idContition, update, session.GetTransaction(), &records)->IsSucceed()) {
        return false;
    }
    if (!HistoryManager->AddHistory(records.GetObjects(), userId, EObjectHistoryAction::UpdateData, session)) {
        return false;
    }
    result = records.GetObjects();
    return true;
}

bool TPromoCodesManager::GetUserActivationsHistory(const TString& userId, const TInstant startInstant, TVector<IPromoCodeMetaReport::TPtr>& codeReports, NDrive::TEntitySession& session) const {
    auto history = Usage->GetHistoryManager().GetEventsByHistoryUserId(userId, startInstant, session);
    if (!history) {
        return false;
    }
    TMap<TString, std::pair<TPromoCodeMeta, TString>> metaWithGenerators;
    {
        TSet<TString> ids;
        Transform(history->begin(), history->end(), std::inserter(ids, ids.begin()), [](const auto& ev) { return ev.GetId(); });
        auto metaInfos = Meta->GetObjects(ids, session);
        if (!metaInfos) {
            return false;
        }
        TSet<ui32> typeIds;
        Transform(metaInfos->begin(), metaInfos->end(), std::inserter(typeIds, typeIds.begin()), [](const auto& meta) { return meta.GetProfitType(); });
        auto profitTypes = Types->GetObjects(typeIds, session);
        if (!profitTypes) {
            return false;
        }
        TMap<ui32, TString> generators;
        Transform(profitTypes->begin(), profitTypes->end(), std::inserter(generators, generators.begin()), [](const auto& type) { return std::make_pair(type.GetId(), type.GetGenerator()); });
        for (const auto& meta : *metaInfos) {
            auto it = generators.find(meta.GetProfitType());
            if (it != generators.end() && it->second) {
                metaWithGenerators.emplace(meta.GetId(), std::make_pair(meta, it->second));
            }
        }
    }
    for (const auto& ev : *history) {
        auto it = metaWithGenerators.find(ev.GetId());
        if (it == metaWithGenerators.end()) {
            continue;
        }
        THolder<TPromoCodeMetaReport> report(new TPromoCodeMetaReport(it->second.first, it->second.second));
        report->SetUsageHistory({ ev });

        codeReports.push_back(std::move(report));
    }
    return true;
}
