#pragma once

#include "entities.h"

#include <drive/backend/promo_codes/common/entities.h>
#include <drive/backend/promo_codes/common/manager.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/roles/permissions.h>

class TProfitTypeConditionConstructor {
public:
    static TString BuildCondition(const TSet<ui32>& ids, NDrive::TEntitySession& session) {
        return "id IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const ui32 id) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("id", id);
        return trCondition;
    }

    static NStorage::TTableRecord BuildCondition(const TPromoProfitType& object) {
        return BuildCondition(object.GetId());
    }
};

class TPromoCodeTypes: public TDatabaseEntitiesManager<TPromoProfitType, TProfitTypeConditionConstructor> {
private:
    using TBase = TDatabaseEntitiesManager<TPromoProfitType, TProfitTypeConditionConstructor>;

public:
    TPromoCodeTypes(const IHistoryContext& context)
        : TBase(context)
    {
    }
};

class TIdConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
        return "id IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const TString& id) {
        NStorage::TTableRecord trCondition;
        if (!!id) {
            trCondition.Set("id", id);
        }
        return trCondition;
    }

    template <class T>
    static NStorage::TTableRecord BuildCondition(const T& object) {
        return BuildCondition(object.GetId());
    }
};

class TPromoCodesMeta: public TDatabaseEntitiesManager<TPromoCodeMeta, TIdConditionConstructor> {
private:
    using TBase = TDatabaseEntitiesManager<TPromoCodeMeta, TIdConditionConstructor>;

public:
    TPromoCodesMeta(const IHistoryContext& context)
        : TBase(context)
    {
    }

    bool GiveOutCodes(const TSet<TString>& ids, const TString& giveOutInfo, const TString& userId, TVector<TPromoCodeMeta>& result, NDrive::TEntitySession& session) const;
    bool GiveOutCodes(const TString& generator, const ui32 count, const TString& giveOutInfo, const TString& userId, TVector<TPromoCodeMeta>& result, NDrive::TEntitySession& session) const;

private:
    bool GiveOutCodes(const TString& idCondition, const TString& giveOutInfo, const TString& userId, TVector<TPromoCodeMeta>& result, NDrive::TEntitySession& session) const;
};

class TPromoCodesUsageHistoryManager: public TDatabaseHistoryManager<TPromoCodeUsage> {
private:
    using TBase = TDatabaseHistoryManager<TPromoCodeUsage>;
    using TObjectContainer = TPromoCodeUsage;

public:
    using TBase::TBase;

    TOptionalEvents GetEventsByInternalId(const TString& keyId, const TInstant since, NDrive::TEntitySession& session) const;
    TOptionalEvents GetEventsByHistoryUserId(const TString& keyId, const TInstant since, NDrive::TEntitySession& session) const;
};

class TPromoCodesUsage: public TDatabaseEntitiesManager<TPromoCodeUsage, TIdConditionConstructor, TPromoCodesUsageHistoryManager> {
private:
    using TBase = TDatabaseEntitiesManager<TPromoCodeUsage, TIdConditionConstructor, TPromoCodesUsageHistoryManager>;

public:
    TPromoCodesUsage(const IHistoryContext& context)
        : TBase(context)
    {
    }
};

class TPromoCodesManagerConfig: public TDBEntitiesManagerConfig {
private:
    R_READONLY(ui32, GenerationPackSize, 500);
    R_READONLY(ui32, CleanPackSize, 5000);

public:
    void Init(const TYandexConfig::Section* section) {
        TDBEntitiesManagerConfig::Init(section);
        GenerationPackSize = section->GetDirectives().Value("GenerationPackSize", GenerationPackSize);
        CleanPackSize = section->GetDirectives().Value("CleanPackSize", CleanPackSize);
    }

    void ToString(IOutputStream& os) const {
        TDBEntitiesManagerConfig::ToString(os);
        os << "GenerationPackSize: " << GenerationPackSize << Endl;
        os << "CleanPackSize: " << CleanPackSize << Endl;
    }
};

class TPromoCodesManager: public IPromoCodesManager, public IMessageProcessor {
private:
    const TPromoCodesManagerConfig Config;
    THolder<TPromoCodeTypes> Types;
    THolder<TPromoCodesMeta> Meta;
    THolder<TPromoCodesUsage> Usage;
    const NDrive::IServer& Server;

private:
    bool CleanOld(const TString& userId, NDrive::TEntitySession& session) const;

    struct TSelectOptions {
        NSQL::TQueryOptions Options;
        ui32 Limit;
        bool ActiveOnly;
        bool UseHistory;

        TSelectOptions(const TPromoCodesSearchContext& filter);
        void UpdateLimits(const ui32 offset, const ui32 limit, const ui32 originLimit = 0);

        template <class T>
        TSelectOptions& SetGenericCondition(const TString& field, T&& value) {
            Options.SetGenericCondition(field, std::forward<T>(value));
            return *this;
        }
    };

    bool GetPromoCodesImpl(const TSelectOptions& options, TUserPermissions::TPtr permissions, TVector<IPromoCodeMetaReport::TPtr>& result, NDrive::TEntitySession& session) const;

protected:
    virtual bool Process(IMessage* message) override;

public:
    virtual TString Name() const override {
        return "mp_promo_codes_manager";
    }

    const TPromoCodeTypes& GetTypes() const {
        return *Types;
    }

    const TPromoCodesMeta& GetMeta() const {
        return *Meta;
    }

    const TPromoCodesUsage& GetUsage() const {
        return *Usage;
    }

    TPromoCodesManager(const IHistoryContext& context, const TPromoCodesManagerConfig& config, const NDrive::IServer& server);
    ~TPromoCodesManager();

    virtual bool GetPromoCodes(const TPromoCodesSearchContext& filter, TUserPermissions::TPtr permissions, TVector<IPromoCodeMetaReport::TPtr>& result, NDrive::TEntitySession& session) const override;
    virtual bool GenerateCodes(const ui32 count, const IPromoCodeGenerator& generator, const TGeneratorContext& context, const TString& userId, TVector<TPromoCodeMeta>& codes, NDrive::TEntitySession& session) const override;
    virtual bool GiveOutCodes(const TSet<TString>& ids, const TString& givenOutInfo, TUserPermissions::TPtr permissions, TVector<TPromoCodeMeta>& codes, NDrive::TEntitySession& session) const override;
    virtual bool GiveOutCodes(const TString& generator, const ui32 count, const TString& givenOutInfo, TUserPermissions::TPtr permissions, TVector<TPromoCodeMeta>& codes, NDrive::TEntitySession& session) const override;
    virtual bool RemoveCodes(const TSet<TString>& ids, const bool force, const TString& userId, NDrive::TEntitySession& session) const override;
    virtual bool ApplyCode(const TApplyContext& context, const TString& objectId, TUserPermissions::TPtr permissions, NJson::TJsonValue& jsonReport, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const override;
    virtual bool GetPromoProfit(const TString& code, TAtomicSharedPtr<IPromoProfitBase>& result, NDrive::TEntitySession& session) const override;
    virtual TMaybe<TVector<TPromoCodeMeta>> GetMetaByCode(const TString& code, NDrive::TEntitySession& session) const override;
    virtual bool GetUserActivationsHistory(const TString& userId, const TInstant startInstant, TVector<IPromoCodeMetaReport::TPtr>& codeReports, NDrive::TEntitySession& session) const override;
};
