#include "radar_geohash.h"

#include <drive/library/cpp/geohash/geohash.h>
#include <rtline/library/geometry/polyline.h>

NStorage::TTableRecord TRadarGeohash::SerializeToTableRecord() const {
    NStorage::TTableRecord row;
    row.Set("tile_id", TileId);
    row.Set("tag_id", TagId);
    return row;
}

bool TRadarGeohash::Parse(const NStorage::TTableRecord& row) {
    return TBaseDecoder::DeserializeFromTableRecord(*this, row);
}

bool TRadarGeohash::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, TileId);
    READ_DECODER_VALUE(decoder, values, TagId);
    return true;
}

bool TRadarGeohashDB::Upsert(const TRadarGeohash& entity, NDrive::TEntitySession& session) const {
    auto table = Database->GetTable(GetTableName());
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::TTableRecord record = entity.SerializeToTableRecord();
    auto result = table->Upsert(record, transaction, record);
    if (!result) {
        session.SetErrorInfo("radar_geohash", "cannot upsert", EDriveSessionResult::InternalError);
        return false;
    }
    if (result->GetAffectedRows() != 1) {
        session.SetErrorInfo("radar_geohash", "expected 1 upserted rows, got " + ToString(result->GetAffectedRows()), EDriveSessionResult::InternalError);
        return false;
    }
    return true;
}

bool TRadarGeohashDB::Remove(const TString& tagId, NDrive::TEntitySession& session) const {
    auto table = Database->GetTable(GetTableName());
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::TTableRecord uniqueRecord;
    uniqueRecord.Set("tag_id", tagId);
    NStorage::TObjectRecordsSet<TRadarGeohash> removedData;
    if (!table->RemoveRow(uniqueRecord, transaction, &removedData)) {
        session.SetErrorInfo("TRadarGeohashDB::Remove", "unable to remove data");
        return false;
    }
    return true;
}

TRadarGeohashManager::TRadarGeohashManager(const IHistoryContext& context, const TDBEntitiesManagerConfig& config)
    : RadarGeohashDb(context)
    , Config(config)
{
}

TSet<TString> TRadarGeohashManager::GetPolygonTiles(const TVector<TGeoCoord>& searchArea, ui8 precision) const {
    if (searchArea.empty()) {
        // In this case we return full world tile.
        return {""};
    }
    return NDrive::GetPolygonGeohash(TPolyLine<TGeoCoord>(searchArea), precision);
}

bool TRadarGeohashManager::AddTagId(const TVector<TGeoCoord>& searchArea, const TString& tagId, ui8 precision, NDrive::TEntitySession& session) const {
    TSet<TString> tiles = GetPolygonTiles(searchArea, precision);
    for (auto&& tileId : tiles) {
        if (!RadarGeohashDb.Upsert({tileId, tagId}, session)) {
            session.AddErrorMessage("TRadarGeohashManager::SetTagId", "cannot set polygon tile id: " + tileId + "to tag id " + tagId);
            return false;
        }
    }
    return true;
}

TMaybe<TSet<TString>> TRadarGeohashManager::GetTagIds(const TGeoCoord& carLocation, ui8 precision, NDrive::TEntitySession& session) const {
    TString tileId = NDrive::GetCoordGeohash(carLocation, precision);
    TString search;
    TSet<TString> query = {""}; // Support tiles that covers full world.
    for (auto&& c : tileId) {
        search.push_back(c);
        query.insert(search);
    }
    auto fetchResult = RadarGeohashDb.FetchInfo(session, NSQL::TQueryOptions().SetGenericCondition("tile_id", std::move(query)));
    if (!fetchResult) {
        session.AddErrorMessage("TRadarGeohashManager::GetTagId", "cannot get tag id using tile id: " + tileId);
        return Nothing();
    }
    TSet<TString> result;
    for (auto&& [_, entity] : fetchResult) {
        result.insert(std::move(entity.GetTagId()));
    }
    return result;
}

bool TRadarGeohashManager::RemoveTagId(const TString& tagId, NDrive::TEntitySession& session) const {
    if (!RadarGeohashDb.Remove(tagId, session)) {
        session.AddErrorMessage("TRadarGeohashManager::RemoveTagId", "cannot remove polygon from tagId " + tagId);
        return false;
    }
    return true;
}
