#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/db_entities.h>
#include <drive/backend/database/transaction/tx.h>
#include <rtline/library/geometry/coord.h>

class TRadarGeohash {
    R_FIELD(TString, TileId);
    R_FIELD(TString, TagId);

public:
    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, TileId, -1);
        R_FIELD(i32, TagId, -1);
    
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            TileId = GetFieldDecodeIndex("tile_id", decoderBase);
            TagId = GetFieldDecodeIndex("tag_id", decoderBase);
        }
    };

    TRadarGeohash() = default;

    TRadarGeohash(const TString& tileId, const TString& tagId)
        : TileId(tileId)
        , TagId(tagId)
    {
    }

    bool Parse(const NStorage::TTableRecord& row);

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    NStorage::TTableRecord SerializeToTableRecord() const;
};

class TRadarGeohashDB: public TDBEntities<TRadarGeohash> {
private:
    using TBase = TDBEntities<TRadarGeohash>;

public:
    TRadarGeohashDB(const IHistoryContext& context)
        : TBase(context.GetDatabase())
    {
    }

    virtual TString GetTableName() const override {
        return "radar_geohash";
    }

    TString GetMainId(const TRadarGeohash& entity) const override {
        return entity.GetTileId() + "-" + entity.GetTagId();
    }

    bool Upsert(const TRadarGeohash& entity, NDrive::TEntitySession& session) const;
    bool Remove(const TString& tagId, NDrive::TEntitySession& session) const;
};

class TRadarGeohashManager {
private:
    const TRadarGeohashDB RadarGeohashDb;
    const TDBEntitiesManagerConfig Config;

public:
    TRadarGeohashManager(const IHistoryContext& context, const TDBEntitiesManagerConfig& config);
    // GetPolygonTiles visits all tiles that covers searchArea.
    TSet<TString> GetPolygonTiles(const TVector<TGeoCoord>& searchArea, ui8 precision) const;
    bool AddTagId(const TVector<TGeoCoord>& searchArea, const TString& tagId, ui8 precision, NDrive::TEntitySession& session) const;
    TMaybe<TSet<TString>> GetTagIds(const TGeoCoord& carLocation, ui8 precision, NDrive::TEntitySession& session) const;
    bool RemoveTagId(const TString& tagId, NDrive::TEntitySession& session) const;
};
