#include "config.h"

#include <rtline/util/algorithm/container.h>

#include <util/string/vector.h>

void TUserRegistrationManagerConfig::Init(const TYandexConfig::Section* section) {
    const auto& directives = section->GetDirectives();
    NotifierName = section->GetDirectives().Value<TString>("NotifierName", NotifierName);
    ChatId = section->GetDirectives().Value<TString>("ChatId", ChatId);
    BanChatId = section->GetDirectives().Value<TString>("BanChatId", BanChatId);
    MinRequiredAge = section->GetDirectives().Value<TDuration>("MinRequiredAge", MinRequiredAge);
    MinRequiredExperience = section->GetDirectives().Value<TDuration>("MinRequiredExperience", MinRequiredExperience);
    AutoBanThreshold = section->GetDirectives().Value<ui32>("AutoBanThreshold", AutoBanThreshold);
    IsNewChatEnabled = section->GetDirectives().Value<bool>("IsNewChatEnabled", IsNewChatEnabled);
    IsPrimaryDBUsed = section->GetDirectives().Value<bool>("IsPrimaryDBUsed", IsPrimaryDBUsed);
    IsAccountCheckEnabled = section->GetDirectives().Value<bool>("IsAccountCheckEnabled", IsAccountCheckEnabled);
    VerificationStatusTag = section->GetDirectives().Value<TString>("VerificationStatusTag", VerificationStatusTag);
    VerificationStatusField = section->GetDirectives().Value<TString>("VerificationStatusField", VerificationStatusField);
    ExpectedResolutionStatus = section->GetDirectives().Value<TString>("ExpectedResolutionStatus", ExpectedResolutionStatus);
    SensitiveDataHashesKey = directives.Value("SensitiveDataHashesKey", SensitiveDataHashesKey);
    if (!SensitiveDataHashesKey) {
        auto sensitiveDataHashesKeyPath = directives.Value<TString>("SensitiveDataHashesKeyPath");
        AssertCorrectConfig(!sensitiveDataHashesKeyPath.empty(), "no 'SensitiveDataHashesKey(Path)' token");
        SensitiveDataHashesKey = Strip(TFileInput(sensitiveDataHashesKeyPath).ReadAll());
    }

    const TYandexConfig::TSectionsMap children = section->GetAllChildren();
    {
        auto it = children.find("RequestConfig");
        if (it != children.end()) {
            RequestConfig.InitFromSection(it->second);
        }
    }

    {
        auto it = children.find("AccountChecker");
        if (it != children.end()) {
            AccountCheckerConfig.Init(it->second);
        }
    }

    {
        TString serviceRolesStr;
        serviceRolesStr = section->GetDirectives().Value<TString>("ServiceRoles", serviceRolesStr);
        if (serviceRolesStr) {
            ServiceRoles = MakeSet(SplitString(serviceRolesStr, ","));
        }
    }
}

void TUserRegistrationManagerConfig::ToString(IOutputStream& os) const {
    os << "NotifierName: " << NotifierName << Endl;
    os << "ChatId: " << ChatId << Endl;
    os << "BanChatId: " << BanChatId << Endl;
    os << "MinRequiredAge: " << MinRequiredAge << Endl;
    os << "MinRequiredExperience: " << MinRequiredExperience << Endl;
    os << "AutoBanThreshold: " << AutoBanThreshold << Endl;
    os << "SensitiveDataHashesKey: " << SensitiveDataHashesKey << Endl;

    os << "IsAccountCheckEnabled: " << IsAccountCheckEnabled << Endl;
    os << "IsNewChatEnabled: " << IsNewChatEnabled << Endl;
    os << "IsPrimaryDBUsed: " << IsPrimaryDBUsed << Endl;

    os << "VerificationStatusTag: " << VerificationStatusTag << Endl;
    os << "VerificationStatusField: " << VerificationStatusField << Endl;
    os << "ExpectedResolutionStatus: " << ExpectedResolutionStatus << Endl;

    os << "ServiceRoles: " << JoinStrings(ServiceRoles.begin(), ServiceRoles.end(), ",") << Endl;

    os << "<RequestConfig>" << Endl;
    RequestConfig.ToString(os);
    os << "</RequestConfig>" << Endl;

    os << "<AccountChecker>" << Endl;
    AccountCheckerConfig.ToString(os);
    os << "</AccountChecker>" << Endl;
}

TUserRegistrationManagerConfig TUserRegistrationManagerConfig::ParseFromString(const TString& configStr) {
    TUserRegistrationManagerConfig result;
    TAnyYandexConfig config;
    CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
    result.Init(config.GetRootSection());
    return result;
}

void TAccountCheckerConfig::Init(const TYandexConfig::Section* section) {
    Host = section->GetDirectives().Value<TString>("Host", Host);
    Route = section->GetDirectives().Value<TString>("Route", Route);
    SelfTvmId = section->GetDirectives().Value<ui32>("SelfTvmId", SelfTvmId);
    DestinationTvmId = section->GetDirectives().Value<ui32>("DestinationTvmId", DestinationTvmId);

    MinAccountAge = section->GetDirectives().Value<TDuration>("MinAccountAge", MinAccountAge);

    {
        TString noScoringCountries;
        noScoringCountries = section->GetDirectives().Value<TString>("NoScoringCountries", noScoringCountries);
        CreateSet(NoScoringCountries, noScoringCountries);
    }
    {
        TString fullScoringCountries;
        fullScoringCountries = section->GetDirectives().Value<TString>("FullScoringCountries", fullScoringCountries);
        CreateSet(FullScoringCountries, fullScoringCountries);
    }
    {
        TString highPrioritySuids;
        highPrioritySuids = section->GetDirectives().Value<TString>("HighPrioritySuids", highPrioritySuids);
        CreateSet(HighPrioritySuids, highPrioritySuids);
    }
    {
        TString lowPrioritySuids;
        lowPrioritySuids = section->GetDirectives().Value<TString>("LowPrioritySuids", lowPrioritySuids);
        CreateSet(LowPrioritySuids, lowPrioritySuids);
    }
}

void TAccountCheckerConfig::ToString(IOutputStream& os) const {
    os << "Host: " << Host << Endl;
    os << "Route: " << Route << Endl;
    os << "SelfTvmId: " << SelfTvmId << Endl;
    os << "DestinationTvmId: " << DestinationTvmId << Endl;
    os << "MinAccountAge: " << MinAccountAge << Endl;

    os << "NoScoringCountries: " << JoinStrings(NoScoringCountries.begin(), NoScoringCountries.end(), ",") << Endl;
    os << "FullScoringCountries: " << JoinStrings(FullScoringCountries.begin(), FullScoringCountries.end(), ",") << Endl;
    os << "HighPrioritySuids: " << JoinStrings(HighPrioritySuids.begin(), HighPrioritySuids.end(), ",") << Endl;
    os << "LowPrioritySuids: " << JoinStrings(LowPrioritySuids.begin(), LowPrioritySuids.end(), ",") << Endl;
}

TAccountCheckerConfig TAccountCheckerConfig::ParseFromString(const TString& configStr) {
    TAccountCheckerConfig result;
    TAnyYandexConfig config;
    CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
    result.Init(config.GetRootSection());
    return result;
}

void TAccountCheckerConfig::CreateSet(TSet<TString>& destination, const TString& rawString) {
    TVector<TString> tokens = SplitString(rawString, ",");
    for (auto&& token : tokens) {
        destination.insert(ToUpperUTF8(token));
    }
}
