#include "manager.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/chat_robots/state/robot_state.pb.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/user_origin.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/user_document_photos/manager.h>
#include <drive/backend/user_options/user_setting.h>

namespace NRegistrarUtil {
    bool CaseInsensitiveEqual(const TString& lhs, const TString& rhs) noexcept {
        try {
            return ToLowerUTF8(lhs) == ToLowerUTF8(rhs);
        } catch (const std::exception& e) {
            DEBUG_LOG << "CaseInsensitiveEqual exception: " << FormatExc(e) << Endl;
            return false;
        }
    }

    TString ToTitle(const TString& str) {
        if (str.empty()) {
            return str;
        }

        TStringBuf buf = str;

        size_t charLen;
        if (GetUTF8CharLen(charLen, (const unsigned char*)buf.begin(), (const unsigned char*)buf.end()) != RECODE_OK) {
            return str;
        }

        return ToUpperUTF8(buf.SubStr(0, charLen)) + ToLowerUTF8(buf.SubStr(charLen));
    }
};

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TUserRegistrationManager::TLicenseExpirationDelay& delay) {
    TInstant fromTmp, untilTmp;
    if (!NJson::ParseField(value, "expiration_from", fromTmp, true) ||
        !NJson::ParseField(value, "expiration_until", untilTmp, true) ||
        !NJson::ParseField(value, "apply_for_foreign", delay.ApplyForForeign, false))
    {
        return false;
    }
    delay.Range = TRange<TInstant>(fromTmp, untilTmp);

    if (value.Has("delay")) {
        TDuration delayTmp;
        if (!NJson::ParseField(value, "delay", delayTmp, true)) {
            return false;
        }
        delay.Delay = delayTmp;
        return true;
    } else if (value.Has("ignore_end")) {
        TInstant limitTmp;
        if (!NJson::ParseField(value, "ignore_end", limitTmp, true)) {
            return false;
        }
        delay.Delay = limitTmp;
        return true;
    }

    return false;
}

bool TUserRegistrationManager::TLicenseExpirationDelay::IsMatching(const TInstant& validToDate) const {
    return validToDate >= Range.From && validToDate <= Range.To;
}

bool TUserRegistrationManager::TLicenseExpirationDelay::ShouldIgnore(const TInstant& validToDate) const {
    if (const auto* duration = std::get_if<TDuration>(&Delay)) {
        if (Now() < validToDate + *duration) {
            return true;
        }
    } else if (const auto* limit = std::get_if<TInstant>(&Delay)) {
        if (Now() < *limit) {
            return true;
        }
    }
    return false;
}

TUserRegistrationManager::TUserRegistrationManager(const TUserRegistrationManagerConfig& config, const NDrive::IServer* server)
    : Config(config)
    , Server(server)
    , DriveAPI(*server->GetDriveAPI())
{
    Hasher.Reset(new TSensitiveDataHasher(Config.GetSensitiveDataHashesKey()));
    Scorer.Reset(
        new TAccountChecker(
            Config.GetAccountCheckerConfig(),
            server
        )
    );
    BlacklistExternal.Reset(new TBlacklistExternal(
        *DriveAPI.GetUsersData(),
        DriveAPI.GetTagsManager().GetUserTags()
    ));
}

bool TUserRegistrationManager::DeduceChatRobot(const TString& userId, TString& chatRobotId, TString& topic) const {
    TYangDocumentVerificationAssignment assignment;
    if (!Server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentVerificationAssignments().GetLastAssignmentForUser(userId, assignment)) {
        return false;
    }

    auto topicLink = GetOriginChat(assignment, TInstant::Zero());
    IChatRobot::ParseTopicLink(topicLink, chatRobotId, topic);

    return true;
}

bool TUserRegistrationManager::CleanupProblemTags(const TString& userId, const TString& operatorUserId, NDrive::TEntitySession& session) const {
    TVector<TDBTag> userTags;
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {}, userTags, session)) {
        INFO_LOG << "REGISTRAR UNEXPECTED: returning FALSE for user " << userId << " because RestoreTags failed" << Endl;
        return false;
    }

    for (auto&& tag : userTags) {
        auto tagPtr = tag.GetTagAs<TUserProblemTag>();
        if (!!tagPtr && (tagPtr->GetDropOnReentry(Server->GetDriveAPI()->GetTagsManager().GetTagsMeta()))) {
            if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(tag, operatorUserId, Server, session)) {
                INFO_LOG << "REGISTRAR UNEXPECTED: returning FALSE for user " << userId << " because RemoveTag failed" << Endl;
                return false;
            }
        }
    }

    return true;
}

bool TUserRegistrationManager::DoBackgroundCheckEntrance(TDriveUserData& userData, const TString& operatorUserId, const TUserPassportData* passportExt, const TUserDrivingLicenseData* drivingLicenseExt, TString chatId, TString topic) const {
    auto userId = userData.GetUserId();

    if (!chatId && !DeduceChatRobot(userId, chatId, topic)) {
        return false;
    }

    auto chatRobot = Server->GetChatRobot(chatId);
    auto robotPtr = dynamic_cast<const TSimpleChatBot*>(chatRobot.Get());
    if (!robotPtr) {
        return false;
    }

    auto tx = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    if (!CleanupRegistrationTags(userId, operatorUserId, true)) {
        return false;
    }

    TUserPassportData passport;
    TUserDrivingLicenseData drivingLicense;

    if (!passportExt || !drivingLicenseExt) {
        if (!GetPersonalData(userData, operatorUserId, passport, drivingLicense)) {
            return true;
        }
    } else {
        passport = *passportExt;
        drivingLicense = *drivingLicenseExt;
    }

    bool isManuallyApproved = IsManuallyApprovedUser(userData.GetUserId(), tx);

    if (!CheckDuplicates(userData, operatorUserId)) {
        return true;
    }

    if (!isManuallyApproved && !CheckDocumentsDataConsistency(userData, operatorUserId, passport, drivingLicense)) {
        return true;
    }
    if (!isManuallyApproved && !CheckDates(userData, operatorUserId, passport, drivingLicense)) {
        return true;
    }

    {
        auto session = Server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        if (!CleanupProblemTags(userId, operatorUserId, session)) {
            return false;
        }
        if (!session.Commit()) {
            return false;
        }
    }

    if (!isManuallyApproved && Config.GetIsAccountCheckEnabled() && !CheckAccountData(userData, operatorUserId, passport, drivingLicense, robotPtr, topic)) {
        return true;
    }

    {
        auto session = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (IsSubjectToBan(userData.GetUserId(), session)) {
            if (userData.GetStatus() == NDrive::UserStatusOnboarding) {
                return Reject(userData, operatorUserId, robotPtr, topic);
            }
            return true;
        }
    }

    if (!CheckDeviceSimilarityToBanned(userData, operatorUserId)) {
        return true;
    }

    return Approve(userData, operatorUserId, robotPtr, topic);
}

bool TUserRegistrationManager::HandleVerifiedAssignment(TYangDocumentVerificationAssignment& recentAssignment, const TString& operatorUserId, TInstant actuality, const TUserPassportData* passportExt, const TUserDrivingLicenseData* drivingLicenseExt) const {
    auto originChat = GetOriginChat(recentAssignment, actuality);
    TSet<TString> skipCheckRobots = StringSplitter(Server->GetSettings().GetValue<TString>("registrar.chat_whitelists.handle_assignment").GetOrElse({})).Split(',').SkipEmpty();
    if (skipCheckRobots.contains(originChat)) {
        return true;
    }
    TString chatId;
    TString topic;
    IChatRobot::ParseTopicLink(originChat, chatId, topic);

    auto chatRobot = Server->GetChatRobot(chatId);
    auto chatRobotPtr = dynamic_cast<const TSimpleChatBot*>(chatRobot.Get());
    if (!chatRobotPtr) {
        return false;
    }

    // Get user object
    TString userId = recentAssignment.GetUserId();
    TDriveUserData userData;
    {
        auto userFetchResult = DriveAPI.GetUsersData()->FetchInfo(userId);
        auto userPtr = userFetchResult.GetResultPtr(userId);
        if (!userPtr) {
            MaybeNotify("Пользователя с ID = " + userId + " не существует");
            return false;
        } else {
            userData = *userPtr;
        }
    }

    // Chat may be incomplete, in this case we shouldn't take any actions
    if (!chatRobotPtr->IsExistsForUser(userId, topic)) {
        if (userData.GetStatus() == NDrive::UserStatusBlocked) {
            auto session = chatRobotPtr->BuildChatEngineSession();
            if (!chatRobotPtr->EnsureChat(userId, topic, session) || !session.Commit()) {
                return false;
            }
        } else if (!chatRobotPtr->CreateCleanChat(userId, topic)) {
            ERROR_LOG << "unable to create chat for old user " << userId << Endl;
            return false;
        }
    }

    if (!chatRobotPtr->GetIsChatCompleted(userId, topic, TInstant::Zero())) {
        DEBUG_LOG << "skipping result with incomplete chat" << Endl;
        return false;
    }

    bool isOk = true;
    ui32 resubmitMask = 0;
    ui32 failureMask = 0;
    GetDocumentsResubmitMask(recentAssignment, resubmitMask, isOk, failureMask, actuality);
    if (resubmitMask & NUserDocument::InvalidResubmitMaskFlag) {
        INFO_LOG << "REGISTRAR UNEXPECTED: returning FALSE for " << recentAssignment.GetId() << " because invalid resubmit mask flag is set" << Endl;
        return false;
    }

    const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();

    // It's the new iteration. Clear all previous registration state tags.
    if (!CleanupRegistrationTags(userId, operatorUserId, false)) {
        return false;
    }

    TUserPassportData passport;
    TUserDrivingLicenseData drivingLicense;
    bool persdataReceived = false;

    if (!passportExt || !drivingLicenseExt) {
        persdataReceived = GetPersonalData(userData, operatorUserId, passport, drivingLicense);
    } else {
        persdataReceived = true;
    }
    if (passportExt) {
        passport = *passportExt;
    }
    if (drivingLicenseExt) {
        drivingLicense = *drivingLicenseExt;
    }
    if (passport.HasNamesForHash()) {
        userData.SetPassportNamesHash(GetDocumentNumberHash(passport.GetNamesForHash()));
    }
    if (passport.GetNumber()) {
        userData.SetPassportNumberHash(GetDocumentNumberHash(passport.GetNumber()));
    }
    if (drivingLicense.GetNumber()) {
        userData.SetDrivingLicenseNumberHash(GetDocumentNumberHash(drivingLicense.GetNumber()));
    }
    UpdateNames(userData, operatorUserId, passport, drivingLicense);

    // Ask to resubmit, if applicable
    if (!isOk) {
        if (!CheckTooManyResubmits(userData, operatorUserId, chatRobotPtr, topic)) {
            return true;
        }

        if (!CheckSelfieScreencapStatuses(userData, operatorUserId, recentAssignment, chatRobotPtr, topic)) {
            return true;
        }

        if (resubmitMask) {
            const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();
            auto session = tagsManager.BuildTx<NSQL::Writable>();
            auto tag = IJsonSerializableTag::BuildWithComment<TRegistrationUserTag>("documents_were_reasked", "Ждем новых документов или их проверки асессорами");
            if (!tagsManager.AddTag(tag, operatorUserId, userId, Server, session) || !session.Commit()) {
                MaybeNotify("Не могу добавить тег documents_were_reasked на пользователя " + userId + ": " + session.GetStringReport());
            }
        } else {
            return QueueForScreening(userData, operatorUserId, "documents_stuck", "Документы не ОК, перезапрос смысла не имеет: " + ToString(recentAssignment.GetIsFraud()) + ": " + ToString(failureMask));
        }


        if (resubmitMask) {
            auto session = chatRobotPtr->BuildChatEngineSession();

            for (auto&& fr : recentAssignment.GetFraudReasons()) {
                if (!chatRobotPtr->MaybeSendFraudReason(userData.GetUserId(), topic, fr, session)) {
                    MaybeNotify("Не могу сообщить о fraud_reason пользователю: " + userData.GetHRReport() + ": " + session.GetStringReport());
                    return true;
                }
            }
            if (!chatRobotPtr->AskToResubmit(userData.GetUserId(), topic, resubmitMask, session, operatorUserId) || !session.Commit()) {
                MaybeNotify("Не могу сделать перезапрос в роботе: " + userData.GetHRReport() + ": " + session.GetStringReport());
                return true;
            }

            auto pushSession = DriveAPI.BuildTx<NSQL::Writable>();
            auto comment = "Перезапрос документов";
            TString pushTagName = "reg_push_resubmit_one";
            if (resubmitMask & (resubmitMask - 1)) {
                pushTagName = "reg_push_resubmit_many";
            }


            if (!DriveAPI.GetUsersData()->SetChatToShow(userId, originChat, false, operatorUserId, Server, pushSession)) {
                MaybeNotify("Не могу выставить правильный редирект в чат: " + userData.GetHRReport() + ": " + pushSession.GetStringReport());
                return false;
            }
            if (userData.GetStatus() == NDrive::UserStatusActive) {
                // Send user to resubmit their docs after end of riding.
                auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>("blocked_for_resubmit", "Постпроверка нашла проблемы на фото");
                if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, pushSession) || !pushSession.Commit()) {
                    MaybeNotify("Не могу повесить тег 'blocked_for_resubmit': " + userData.GetHRReport() + ": " + pushSession.GetStringReport());
                    return false;
                }
                return true;
            }

            auto tag = MakeAtomicShared<TUserPushTag>(pushTagName);
            tag->SetComment(comment);
            auto optionalExternalUserId = TUserOriginTag::GetExternalUserId(userData.GetUserId(), *Server, pushSession);
            if (optionalExternalUserId) {
                tag->SetExternalUserId(*optionalExternalUserId);
            }
            if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, pushSession) || !pushSession.Commit()) {
                MaybeNotify("Не могу отправить пуш про перезапрос доков: " + userData.GetUserId() + ": " + pushSession.GetStringReport());
            }
        }
        return true;
    }

    if (recentAssignment.GetIsFraud() == TYangDocumentVerificationAssignment::EFraudStatus::MaybeFraud) {
        return QueueForScreening(userData, operatorUserId, "maybe_fraud_documents", "Асессор поставил MAYBE_FRAUD");
    }
    if (recentAssignment.GetIsFraud() == TYangDocumentVerificationAssignment::EFraudStatus::DefinitelyFraud) {
        return QueueForScreening(userData, operatorUserId, "definitely_fraud_documents", "Асессор поставил DEFINITELY_FRAUD");
    }

    if (!persdataReceived) {
        return true;
    }

    return DoBackgroundCheckEntrance(userData, operatorUserId, &passport, &drivingLicense, chatId, topic);
}

bool TUserRegistrationManager::CleanupRegistrationTags(const TString& userId, const TString& operatorUserId, bool ignoreTagFailures) const {
    TVector<TTaggedUser> taggedUsers;
    TVector<TString> tagNames;
    TSet<TString> noCleanupTags;
    auto td = DriveAPI.GetTagsManager().GetTagsMeta().GetTagsByType(TRegistrationUserTag::TypeName);
    for (auto&& i : td) {
        auto regTag = i->GetAs<TRegistrationUserTag::TDescription>();
        if (!regTag) {
            continue;
        }
        if (regTag->GetNoCleanup()) {
            noCleanupTags.emplace(regTag->GetName());
            continue;
        }
        tagNames.push_back(i->GetName());
    }
    const auto& userTagManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    TDBTags tags;
    {
        auto session = userTagManager.BuildSession(true);
        auto optionalTags = userTagManager.RestoreTags(TVector<TString>{userId}, tagNames, session);
        if (!optionalTags) {
            ERROR_LOG << "CleanupRegistrationTags: cannot RestoreTags for " << userId << ": " << session.GetStringReport() << Endl;
            return false;
        }
        tags = std::move(*optionalTags);
    }
    {
        for (auto&& tag : tags) {
            auto tagPtr = tag.GetTagAs<TRegistrationUserTag>();
            if (tagPtr && noCleanupTags.contains(tagPtr->GetName())) {
                continue;
            }
            if (tagPtr) {
                auto session = Server->GetDriveAPI()->GetTagsManager().GetUserTags().BuildSession();
                bool removed = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(tag, operatorUserId, Server, session) && session.Commit();
                if (!removed) {
                    ERROR_LOG << "UserRegistrationManager::CleanupRegistrationTags: could not remove tag: " << session.GetStringReport() << Endl;
                    session.ClearErrors();
                }
                if (!removed && !ignoreTagFailures) {
                    INFO_LOG << "REGISTRAR UNEXPECTED: returning FALSE for assignment because RemoveTag failed" << Endl;
                    return false;
                }
            }
        }
    }
    return true;
}

bool TUserRegistrationManager::CheckSelfieScreencapStatuses(TDriveUserData& userData, const TString& operatorUserId, TYangDocumentVerificationAssignment& recentAssignment, const TSimpleChatBot* chatRobotPtr, const TString& topic) const {
    if (!DriveAPI.HasDocumentPhotosManager()) {
        ERROR_LOG << "user photo manager undefined" << Endl;
        return false;
    }
    auto photosFetchResult = DriveAPI.GetDocumentPhotosManager().GetUserPhotosDB().FetchInfo(recentAssignment.GetPassportSelfieId());
    if (photosFetchResult.size() != 1) {
        return true;
    }
    auto selfieStatus = photosFetchResult.begin()->second.GetVerificationStatus();
    if (selfieStatus == NUserDocument::EVerificationStatus::VideoScreencap) {
        RejectWithTag(userData, operatorUserId, "blocked_fraud_selfie_screencap", "", chatRobotPtr, topic);
        return false;
    } else if (selfieStatus == NUserDocument::EVerificationStatus::VideoAnotherPerson) {
        RejectWithTag(userData, operatorUserId, "blocked_fraud_selfie_another_p", "", chatRobotPtr, topic);
        return false;
    }
    return true;
}

bool TUserRegistrationManager::CheckTooManyResubmits(TDriveUserData& userData, const TString& operatorUserId, const TSimpleChatBot* chatRobotPtr, const TString& topic) const {
    if (!DriveAPI.HasDocumentPhotosManager()) {
        ERROR_LOG << "user photo manager undefined" << Endl;
        return false;
    }
    auto assignments = DriveAPI.GetDocumentPhotosManager().GetDocumentVerificationAssignments().GetAllAssignmentsForUser(userData.GetUserId());

    size_t numRecentlyCreated = 0;
    bool hasDefFraud = false;

    for (auto&& assignment : assignments) {
        if (assignment.GetCreatedAt() > Now() - TDuration::Days(14)) {
            ++numRecentlyCreated;
            if (assignment.GetIsFraud() == TYangDocumentVerificationAssignment::EFraudStatus::DefinitelyFraud) {
                hasDefFraud = true;
            }
        }
    }

    if (hasDefFraud && numRecentlyCreated >= 3) {
        Reject(userData, operatorUserId, chatRobotPtr, topic);
        const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();
        auto session = DriveAPI.BuildTx<NSQL::Writable>();
        auto comment = "Получил definitely_fraud и много перезапросов";
        auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>("blocked_too_many_resubmits", comment);
        if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session) || !session.Commit()) {
            MaybeNotify("Не могу забанить за слишком много ресабмитов: " + userData.GetUserId() + ": " + session.GetStringReport());
        }
        return false;
    }

    return true;
}

bool TUserRegistrationManager::CheckMultAccount(TDriveUserData& userData, const TString& operatorUserId, const TString& twinId, const TString& defaultTagName, const TString& commentPrefix) const {
    const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();
    auto twinFetchResult = DriveAPI.GetUsersData()->FetchInfo(twinId);
    auto twinPtr = twinFetchResult.GetResultPtr(twinId);
    if (!twinPtr || twinPtr->GetStatus() != NDrive::UserStatusBlocked) {
        return true;
    }

    TVector<TDBTag> userTags;
    {
        auto session = tagsManager.BuildSession(true);
        if (!tagsManager.RestoreEntityTags(twinId, {}, userTags, session)) {
            NDrive::TEventLog::Log("CheckMultAccountError", NJson::TMapBuilder
                ("type", "RestoreEntityTags")
                ("user_id", userData.GetUserId())
                ("twin_id", twinId)
                ("errors", session.GetReport())
            );
            return false;
        }
    }

    if (GetTwinViolationScore(userTags) >= Config.GetAutoBanThreshold()) {
        TString dupTagName = defaultTagName;
        auto session = tagsManager.BuildSession(false);
        if (auto isDeleted = Server->GetDriveAPI()->GetUsersData()->IsDeletedByTags(twinPtr->GetUserId(), session)) {
            if (*isDeleted) {
                dupTagName = "blocked_by_security";
            }
        } else {
            MaybeNotify("Не могу проверить удалён ли похожий пользователь:" + twinId + " для : " + userData.GetUserId() + ": " + session.GetStringReport());
            return false;
        }
        MaybeNotify("Отказываем в регистрации пользователю " + userData.GetHRReport() + " за мульт с баном.");
        auto comment = commentPrefix + twinId + ". Теги:";
        bool isFirstTag = true;
        for (auto&& tag : userTags) {
            if (!tag.Is<TUserProblemTag>()) {
                continue;
            }
            comment += (isFirstTag ? " " : ", ") + tag->GetName();
            isFirstTag = false;
        }
        auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>(dupTagName, comment);
        if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session) || !session.Commit()) {
            MaybeNotify("Не могу забанить за дубликат: " + userData.GetUserId() + ": " + session.GetStringReport());
        }
        return false;
    }
    return true;
}

bool TUserRegistrationManager::CheckDeviceSimilarityToBanned(TDriveUserData& userData, const TString& operatorUserId) const {
    auto session = Server->GetDriveAPI()->BuildSession(true, false);
    TVector<TShortUserDevice> userDevices;
    Server->GetUserDevicesManager()->GetUserDevices(userData.GetUserId(), userDevices);

    TSet<TString> deviceIds;
    for (auto&& device : userDevices) {
        deviceIds.insert(device.GetDeviceId());
    }

    TSet<TString> twinIds;
    if (!Server->GetUserDevicesManager()->GetUsersByDeviceIds(deviceIds, twinIds, session, /*verifiedOnly=*/true)) {
        NDrive::TEventLog::Log("CheckDeviceSimilarityToBannedError", NJson::TMapBuilder
            ("type", "GetUsersByDeviceIds")
            ("user_id", userData.GetUserId())
        );
    }
    for (auto&& twinId : twinIds) {
        if (twinId == userData.GetUserId()) {
            continue;
        }
        if (!CheckMultAccount(userData, operatorUserId, twinId, "blocked_bad_deviceid", "Совпадает по DeviceID  с заблокированным аккаунтом: ")) {
            return false;
        }
    }

    return true;
}

bool TUserRegistrationManager::CheckAccountData(TDriveUserData& userData, const TString& operatorUserId, const TUserPassportData& passport, const TUserDrivingLicenseData& drivingLicense, const TSimpleChatBot* /*chatRobotPtr*/, const TString& /*topic*/) const {
    TSet<TString> countries;
    if (passport.GetBiographicalCountry()) {
        countries.insert(passport.GetBiographicalCountry());
    }
    if (passport.GetRegistrationCountry()) {
        countries.insert(passport.GetRegistrationCountry());
    }
    if (drivingLicense.GetCountry()) {
        countries.insert(drivingLicense.GetCountry());
    }
    if (drivingLicense.GetBackCountry()) {
        countries.insert(drivingLicense.GetBackCountry());
    }
    const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();

    ui64 uid = userData.GetPassportUid();
    if (!uid) {
        return false;
    }

    bool scoringPassage = Scorer->IsPassed(uid, countries, Config.GetNotifierName());
    auto session = DriveAPI.BuildTx<NSQL::Writable>();
    auto tag = IJsonSerializableTag::BuildWithComment<TSimpleUserTag>("yaccount_reg_date", ToString(Scorer->GetCachedCreationDate(uid).Seconds()));
    if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session, EUniquePolicy::Rewrite)) {
        MaybeNotify("Не могу навесить тег про возраст аккаунта: " + userData.GetUserId() + ": " + session.GetStringReport());
        return false;
    }
    if (!scoringPassage) {
        auto comment = "Не прошел проверку аккаунта по ЧЯ (обеление)";
        auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>("blocked_bb_check", comment);
        if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session) || !session.Commit()) {
            MaybeNotify("Не могу забанить за проверку по ЧЯ: " + userData.GetUserId() + ": " + session.GetStringReport());
        }
        return false;
    }

    return session.Commit();
}

bool TUserRegistrationManager::MoveTags(const TString& fromUserId, const TString& toUserId, const TString& operatorUserId, NDrive::TEntitySession& session) const {
    TVector<TDBTag> userTags;
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(fromUserId, {}, userTags, session)) {
        return false;
    }

    ITagsMeta::TTagDescriptionsByName tagDescriptions = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();

    TVector<TDBTag> tagsForRemove;
    const auto& tagsManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    for (auto&& tag : userTags) {
        auto descriptionIter = tagDescriptions.find(tag->GetName());
        if (descriptionIter == tagDescriptions.end() || !descriptionIter->second) {
            continue;
        }
        if (!tag.GetTagAs<ITemporaryActionTag>() && !tag.GetTagAs<TLandingUserTag>() && !descriptionIter->second->IsTransferredToDouble()) {
            continue;
        }
        auto tagCopy = tag.Clone(Server->GetDriveAPI()->GetTagsHistoryContext());
        if (!tagCopy) {
            continue;
        }
        if (!tagsManager.AddTag(tag.GetData(), operatorUserId, toUserId, Server, session)) {
            return false;
        }
        tagsForRemove.push_back(tag);
    }

    if (tagsForRemove.size() && !tagsManager.RemoveTagsSimple(tagsForRemove, operatorUserId, session, false)) {
        ERROR_LOG << "could not remove temporary action tags" << Endl;
        return false;
    }

    return true;
}

bool TUserRegistrationManager::MoveBonuses(const TString& fromUserId, const TString& toUserId, NDrive::TEntitySession& session) const {
    auto userAccounts = Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetUserAccounts(fromUserId, session);
    if (!userAccounts) {
        return false;
    }
    for (auto&& account : *userAccounts) {
        if (account->GetType() != NDrive::NBilling::EAccount::Bonus) {
            continue;
        }
        auto accountId = account->GetId();
        if (!Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().UnLinkAccount(fromUserId, accountId, toUserId, session)) {
            ERROR_LOG << "could not unlink account" << Endl;
            return false;
        }
        if (!Server->GetDriveAPI()->GetBillingManager().GetAccountsManager().LinkAccount(toUserId, accountId, toUserId, session)) {
            ERROR_LOG << "could not link account" << Endl;
            return false;
        }
    }
    return true;
}

bool TUserRegistrationManager::MoveRoles(const TString& fromUserId, const TString& toUserId, NDrive::TEntitySession& session) const {
    auto userRoles = Server->GetDriveAPI()->GetUsersData()->GetRoles().RestoreUserRoles(fromUserId, session);
    if (!userRoles) {
        return false;
    }

    TVector<TUserRole> grPool;
    TVector<TUserRole> userAccessPool;
    for (auto&& role : userRoles->GetRoles()) {
        if (!role.GetActive()) {
            continue;
        }
        if (role.GetRoleId().StartsWith("GR_default_user_")) {
            grPool.push_back(role);
        }
        if (role.GetRoleId().StartsWith("user_access_") || role.GetRoleId() == "default_user" || role.GetRoleId() == "user_access_tariff_fix_point") {
            userAccessPool.push_back(role);
        }
    }

    if (grPool.size()) {
        return AddRolesFromPool(toUserId, grPool, session);
    } else {
        return AddRolesFromPool(toUserId, userAccessPool, session);
    }
}

bool TUserRegistrationManager::MoveData(TDriveUserData& userData, const TDriveUserData* twinData, const TString& operatorUserId, NDrive::TEntitySession& session, NDataTransfer::TDataTransferTraits traits) const {
    if (!twinData) {
        return false;
    }
    if ((traits & NDataTransfer::Tags) && !MoveTags(twinData->GetUserId(), userData.GetUserId(), operatorUserId, session)) {
        MaybeNotify("Не могу перенести промо-скидки от " + twinData->GetUserId() + " к " + userData.GetUserId() + ": " + session.GetStringReport());
        return false;
    }
    if ((traits & NDataTransfer::Roles) && userData.GetApprovedAt() == TInstant::Zero() && !MoveRoles(twinData->GetUserId(), userData.GetUserId(), session)) {
        MaybeNotify("Не могу перенести роли от " + twinData->GetUserId() + " к " + userData.GetUserId() + ": " + session.GetStringReport());
        return false;
    }
    if ((traits & NDataTransfer::Bonuses) && userData.GetApprovedAt() == TInstant::Zero() && !MoveBonuses(twinData->GetUserId(), userData.GetUserId(), session)) {
        MaybeNotify("Не могу перенести бонусы от " + twinData->GetUserId() + " к " + userData.GetUserId() + ": " + session.GetStringReport());
        return false;
    }
    if ((traits & NDataTransfer::FirstRide) && !twinData->IsFirstRiding()) {
        userData.SetFirstRiding(false);
    }
    return true;
}

bool TUserRegistrationManager::AddRolesFromPool(const TString& userId, const TVector<TUserRole>& pool, NDrive::TEntitySession& session) const {
    for (auto&& role : pool) {
        TUserRole newRole = role;
        newRole.SetUserId(userId);
        if (!Server->GetDriveAPI()->GetUsersData()->GetRoles().Link(newRole, userId, session)) {
            return false;
        }
    }
    return true;
}

bool TUserRegistrationManager::CheckSamePersonProfiles(TDriveUserData& userData, const TString& operatorUserId, const TString& soughtHash, const TString& tagName) const {
    auto twinIds = DriveAPI.GetUsersData()->GetDocumentOwnersByHash(soughtHash);
    const auto& tagsManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    for (auto&& twinId : twinIds) {
        if (twinId == userData.GetUserId()) {
            continue;
        }

        if (DriveAPI.HasBillingManager()) {
            auto session = DriveAPI.BuildTx<NSQL::Writable>();
            auto debt = DriveAPI.GetBillingManager().GetDebt(twinId, session);
            if (!debt) {
                MaybeNotify("Не могу загрузить активные платежи: " + userData.GetUserId() + ": " + session.GetStringReport());
                return false;
            }
            if (*debt > 0) {
                auto comment = "Есть долг у другого аккаунта с такими документами. Номер документа совпадает с " + twinId;
                MaybeNotify("Отказываем в регистрации пользователю " + userData.GetHRReport() + " за мульт с долгом");
                auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>(tagName, comment);
                if (!BanUser(userData, operatorUserId, NBans::EReason::DuplicatePassport, session) || !tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session) || !session.Commit()) {
                    MaybeNotify("Не могу забанить за дубликат документа: " + userData.GetUserId() + ": " + session.GetStringReport());
                }
                return false;
            }
        }

        auto userFetchResult = DriveAPI.GetUsersData()->FetchInfo(twinId);
        auto twinPtr = userFetchResult.GetResultPtr(twinId);
        if (!twinPtr) {
            continue;
        }

        bool isStrongMatch = false;
        if (twinPtr->GetDrivingLicenseNumberHash() != "" && twinPtr->GetDrivingLicenseNumberHash() != EmptyFieldHash && userData.GetDrivingLicenseNumberHash() == twinPtr->GetDrivingLicenseNumberHash()) {
            isStrongMatch = true;
        } else if (twinPtr->GetPassportNumberHash() != "" && twinPtr->GetPassportNumberHash() != EmptyFieldHash && userData.GetPassportNumberHash() == twinPtr->GetPassportNumberHash()) {
            isStrongMatch = true;
        }
        if (!isStrongMatch) {
            continue;
        }

        if (CheckMultAccount(userData, operatorUserId, twinId, tagName, "Есть бан у другого аккаунта с такими документами. Номер документа совпадает с ")) {
            auto session = Server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (auto isDeleted = Server->GetDriveAPI()->GetUsersData()->IsDeletedByTags(twinId, session)) {
                if (*isDeleted) {
                    continue;
                }
            } else {
                MaybeNotify("Не могу проверить похожего пользователя:" + twinId + " для : " + userData.GetUserId() + ": " + session.GetStringReport());
                return false;
            }
            MaybeNotify("Блокируем пользователя " + twinPtr->GetHRReport() + " за то, что он зарегистрировал второго пользователя (номер доков совпал), и это " + userData.GetHRReport());
            auto comment = "Зарегистрирован более новый пользователь с такими же документами: " + userData.GetUserId();
            auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>(tagName, comment);
            if (!MoveData(userData, twinPtr, operatorUserId, session)) {
                continue;
            }

            TAtomicSharedPtr<TConnectionUserTag> connectionTag = new TConnectionUserTag();
            connectionTag->SetComment("Мультиакк");
            connectionTag->SetName("same_person");
            connectionTag->SetConnectedUserId(twinId);

            TAtomicSharedPtr<TConnectionUserTag> revConnectionTag = new TConnectionUserTag();
            revConnectionTag->SetComment("Мультиакк (более поздний)");
            revConnectionTag->SetName("same_person");
            revConnectionTag->SetConnectedUserId(userData.GetUserId());

            if (!GetUserConnections(userData.GetUserId(), "same_person", session).contains(twinId) && !tagsManager.AddTag(connectionTag, operatorUserId, userData.GetUserId(), Server, session)) {
                MaybeNotify("Не могу добавить connection_tag на пользователя: " + twinId + ": " + session.GetStringReport());
            }
            if (!GetUserConnections(twinId, "same_person", session).contains(userData.GetUserId()) && !tagsManager.AddTag(revConnectionTag, operatorUserId, twinId, Server, session)) {
                MaybeNotify("Не могу добавить обратный connection_tag на пользователя: " + userData.GetUserId() + ": " + session.GetStringReport());
            }
            if (!tagsManager.AddTag(tag, operatorUserId, twinId, Server, session) || !session.Commit()) {
                MaybeNotify("Не могу забанить за дубликат документа: " + twinId + ": " + session.GetStringReport());
            }
        } else {
            return false;
        }
    }
    return true;
}

TSet<TString> TUserRegistrationManager::GetUserConnections(const TString& userId, const TString& tagName, NDrive::TEntitySession& session) const {
    TVector<TDBTag> tags;
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({userId}, {tagName}, tags, session)) {
        return {};
    }
    TSet<TString> result;
    for (auto&& tag : tags) {
        auto impl = tag.GetTagAs<TConnectionUserTag>();
        if (!impl) {
            continue;
        }
        result.insert(impl->GetConnectedUserId());
    }
    return result;
}

ui32 TUserRegistrationManager::GetTwinViolationScore(const TVector<TDBTag>& tags) const {
    const auto& tagsMeta = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta();

    ui32 pointsSum = 0;
    for (auto&& tag : tags) {
        auto tagPtr = tag.GetTagAs<TUserProblemTag>();
        if (!!tagPtr && !tagPtr->GetIsNeutral(tagsMeta)) {
            pointsSum += tagPtr->GetPoints(tagsMeta);
        }
    }

    return pointsSum;
}

ui32 TUserRegistrationManager::GetTwinViolationScore(const TString& userId) const {
    TVector<TDBTag> userTags;
    auto session = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {}, userTags, session)) {
        return false;
    }

    return GetTwinViolationScore(userTags);
}

bool TUserRegistrationManager::CheckDuplicates(TDriveUserData& userData, const TString& operatorUserId) const {
    if (userData.GetPassportNumberHash() && !CheckSamePersonProfiles(userData, operatorUserId, userData.GetPassportNumberHash(), "blocked_duplicate_namber_pass")) {
        return false;
    }
    if (userData.GetDrivingLicenseNumberHash() && !CheckSamePersonProfiles(userData, operatorUserId, userData.GetDrivingLicenseNumberHash(), "blocked_duplicate_driver_license")) {
        return false;
    }
    return true;
}

bool TUserRegistrationManager::CheckDates(TDriveUserData& userData, const TString& operatorUserId, TUserPassportData& passportData, TUserDrivingLicenseData& drivingLicenseData) const {
    auto minAllowedBirthDate = (Now() - Config.GetMinRequiredAge()).ToString();
    auto minAllowedExperienceFrom = Now() - Config.GetMinRequiredExperience();
    if (passportData.GetBirthDate() > minAllowedBirthDate && passportData.GetBirthDate()) {
        QueueForScreening(userData, operatorUserId, "too_young", "Слишком молод");
        return false;
    }

    TInstant effectiveExperienceStartDate;
    if (drivingLicenseData.GetExperienceFrom() != TInstant::Zero()) {
        effectiveExperienceStartDate = drivingLicenseData.GetExperienceFrom();
    } else if (drivingLicenseData.GetCategoriesBValidFromDate() != TInstant::Zero()) {
        effectiveExperienceStartDate = drivingLicenseData.GetCategoriesBValidFromDate();
    } else if (drivingLicenseData.GetIssueDate() != TInstant::Zero()) {
        effectiveExperienceStartDate = drivingLicenseData.GetIssueDate();
    } else {
        QueueForScreening(userData, operatorUserId, "too_unexperienced", "Невозможно установить дату начала водительского стажа");
        return false;
    }

    if (effectiveExperienceStartDate > minAllowedExperienceFrom) {
        QueueForScreening(userData, operatorUserId, "too_unexperienced", "Нет нужного кол-ва лет стажа");
        return false;
    }

    if (drivingLicenseData.GetCategoriesBValidToDate() && drivingLicenseData.GetCategoriesBValidToDate() < Now()) {
        bool shouldIgnore = false;

        NJson::TJsonValue delaysJson;
        TString rangesStr = Server->GetSettings().GetValueDef<TString>("documents_expiration.driving_license.ignore_ranges", "");

        if (rangesStr &&
            NJson::ReadJsonFastTree(rangesStr, &delaysJson) &&
            delaysJson["ignore_ranges"].IsArray())
        {
            for (auto&& delayJson : delaysJson["ignore_ranges"].GetArray()) {
                TLicenseExpirationDelay delay;
                if (!NJson::TryFromJson(delayJson, delay)) {
                    ERROR_LOG << "Failed to parse delay " << delayJson.GetStringRobust() << Endl;
                    break;
                }

                if (!delay.IsMatching(drivingLicenseData.GetCategoriesBValidToDate()) ||
                    (drivingLicenseData.IsForeign() && !delay.ApplyForForeign))
                {
                    continue;
                }

                if (delay.ShouldIgnore(drivingLicenseData.GetCategoriesBValidToDate())) {
                    shouldIgnore = true;
                    break;
                }
            }
        } else {
            INFO_LOG << "TUserRegistrationManager::CheckDates: expiration delays unavailable: " << rangesStr << Endl;
        }

        if (!shouldIgnore) {
            QueueForScreening(userData, operatorUserId, "driving_license_expired", "ВУ истекло");
            return false;
        }
    }

    auto effectiveBirthDate = passportData.GetBirthDate() ? passportData.GetBirthDate() : drivingLicenseData.GetBirthDate();
    if (effectiveBirthDate) {
        TString minAllowedBirthDate = Now().ToString();
        struct tm tmNow;
        Now().LocalTime(&tmNow);
        minAllowedBirthDate = "19" + ToString(tmNow.tm_year - 67) + minAllowedBirthDate.substr(4);
        if (minAllowedBirthDate > effectiveBirthDate) {
            const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();
            auto session = DriveAPI.BuildTx<NSQL::Writable>();
            auto comment = "Слишком стар";
            auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>("blocked_too_old", comment);
            auto characterTag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>("user_character_too_old", comment);
            if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session) || !tagsManager.AddTag(characterTag, operatorUserId, userData.GetUserId(), Server, session) || !session.Commit()) {
                MaybeNotify("Не могу забанить за too_old: " + userData.GetUserId());
            }
            return false;
        }
    }

    return true;
}

bool TUserRegistrationManager::Approve(TDriveUserData& userData, const TString& operatorUserId, const TSimpleChatBot* chatRobotPtr, const TString& topic) const {
    if (!CleanupRegistrationTags(userData.GetUserId(), operatorUserId, true)) {
        return false;
    }

    bool needNotify = (userData.GetStatus() == NDrive::UserStatusOnboarding || userData.GetStatus() == NDrive::UserStatusScreening);
    bool needChatMessage = (userData.GetStatus() != NDrive::UserStatusActive);
    TString pushType;
    TString nextChatItemId;
    bool isFirstApprove = false;
    if (userData.GetApprovedAt() == TInstant::Zero()) {
        pushType = "reg_push_approved";
        nextChatItemId = "finish_ok";
        isFirstApprove = true;
    } else {
        pushType = "reg_push_unblock_generic";
        nextChatItemId = "finish_ok_again";
    }
    userData.UpdateStatus(NDrive::UserStatusActive);

    MaybeNotify("Впускаем в сервис: " + userData.GetHRReport());

    // Send messages to new chat
    if (needChatMessage && !chatRobotPtr->ApproveFinal(userData.GetUserId(), topic, nextChatItemId)) {
        MaybeNotify("Не могу отправить пользователю " + userData.GetHRReport() + " сообщения о конце регистрации");
    }

    auto session = DriveAPI.BuildTx<NSQL::Writable>();

    if (!CleanupProblemTags(userData.GetUserId(), operatorUserId, session)) {
        return false;
    }

    // No need to force show the chat anymore
    if (!DriveAPI.GetUsersData()->DropChatShow(userData.GetUserId(), operatorUserId, session)) {
        MaybeNotify("Не убрать у пользователя " + userData.GetHRReport() + " тег на показ чата" + ": " + session.GetStringReport());
        return false;
    }

    // Send approval push
    if (needNotify) {
        const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();
        auto tag = MakeAtomicShared<TUserPushTag>(pushType);
        tag->SetComment("from registration robot");
        auto optionalExternalUserId = TUserOriginTag::GetExternalUserId(userData.GetUserId(), *Server, session);
        if (optionalExternalUserId) {
            tag->SetExternalUserId(*optionalExternalUserId);
        }
        if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session)) {
            MaybeNotify("Не могу добавить тег reg_push_approved на пользователя " + userData.GetHRReport() + ": " + session.GetStringReport());
            return false;
        }
    }

    // Change user status to active
    if (!DriveAPI.GetUsersData()->UpdateUser(userData, operatorUserId, session) || !session.Commit()) {
        MaybeNotify("Не могу обновить данные пользователя " + userData.GetUserId() + " в таблице \"user\"" + ": " + session.GetStringReport());
        return false;
    }

    if (isFirstApprove) {
        if (!Server || !Server->GetUserDevicesManager() || !Server->GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventType::SuccessRegistration, userData.GetUserId(), userData.GetApprovedAt())) {
            TUnistatSignalsCache::SignalAdd("frontend", "register-success-registration-fail", 1);
        }
    }
    return true;
}

bool TUserRegistrationManager::GetPersonalData(TDriveUserData& userData, const TString& operatorUserId, TUserPassportData& passportData, TUserDrivingLicenseData& drivingLicenseData) const {
    if (!userData.GetPassportDatasyncRevision()) {
        QueueForScreening(userData, operatorUserId, "no_documents_revision", "Нет ревизии паспорта");
        return false;
    }
    if (!userData.GetDrivingLicenseDatasyncRevision()) {
        QueueForScreening(userData, operatorUserId, "no_documents_revision", "Нет ревизии ВУ");
        return false;
    }
    const auto& uid = userData.GetUid();
    if (!uid) {
        return false;
    }

    {
        if (!DriveAPI.GetPrivateDataClient().GetPassportSync(userData, userData.GetPassportDatasyncRevision(), passportData)) {
            QueueForScreening(userData, operatorUserId, "no_documents_revision", "Не могу получить данные паспорта из DataSync");
            return false;
        }

        if (!DriveAPI.GetPrivateDataClient().GetDrivingLicenseSync(userData, userData.GetDrivingLicenseDatasyncRevision(), drivingLicenseData)) {
            QueueForScreening(userData, operatorUserId, "no_documents_revision", "Не могу получить данные ВУ из DataSync");
            return false;
        }
    }

    return true;
}

bool TUserRegistrationManager::UpdateNames(TDriveUserData& userData, const TString& operatorUserId, const TUserPassportData& passportData, const TUserDrivingLicenseData& drivingLicenseData) const {
    bool isNameSet = false;
    if (passportData.GetFirstName()) {
        userData.SetFirstName(NRegistrarUtil::ToTitle(passportData.GetFirstName()));
        userData.SetLastName(NRegistrarUtil::ToTitle(passportData.GetLastName()));
        userData.SetPName(NRegistrarUtil::ToTitle(passportData.GetMiddleName()));
        isNameSet = true;
    } else if (drivingLicenseData.GetFirstName()) {
        userData.SetFirstName(NRegistrarUtil::ToTitle(drivingLicenseData.GetFirstName()));
        userData.SetLastName(NRegistrarUtil::ToTitle(drivingLicenseData.GetLastName()));
        userData.SetPName(NRegistrarUtil::ToTitle(drivingLicenseData.GetMiddleName()));
        isNameSet = true;
    }
    if (isNameSet) {
        auto session = DriveAPI.BuildTx<NSQL::Writable>();
        return DriveAPI.GetUsersData()->UpdateUser(userData, operatorUserId, session) && session.Commit();
    }
    return false;
}

bool TUserRegistrationManager::CheckDocumentsDataConsistency(TDriveUserData& userData, const TString& operatorUserId, const TUserPassportData& passportData, const TUserDrivingLicenseData& drivingLicenseData) const {
    if ((passportData.GetBiographicalCountry() == "RUS") ^ (drivingLicenseData.GetCountry() == "RUS")) {
        return true;
    }

    if (!NRegistrarUtil::CaseInsensitiveEqual(passportData.GetFirstName(), drivingLicenseData.GetFirstName())) {
        QueueForScreening(userData, operatorUserId, "documents_data_inconsistent", "Имена в паспорте и в правах не совпадают");
        return false;
    }
    if (!NRegistrarUtil::CaseInsensitiveEqual(passportData.GetLastName(), drivingLicenseData.GetLastName())) {
        QueueForScreening(userData, operatorUserId, "documents_data_inconsistent", "Фамилии в паспорте и в правах не совпадают");
        return false;
    }
    if (!NRegistrarUtil::CaseInsensitiveEqual(passportData.GetBirthDate(), drivingLicenseData.GetBirthDate())) {
        QueueForScreening(userData, operatorUserId, "documents_data_inconsistent", "Даты рождения в паспорте и в правах не совпадают");
        return false;
    }

    return true;
}

bool TUserRegistrationManager::Reject(TDriveUserData& userData, const TString& operatorUserId, const TSimpleChatBot* chatRobotPtr, const TString& topic) const {
    MaybeNotify("Отказываем в регистрации: " + userData.GetHRReport());

    // Send messages to new chat
    {
        auto session = chatRobotPtr->BuildChatEngineSession();
        bool isWriteSucceeded = chatRobotPtr->SendRejectionMessages(userData.GetUserId(), topic, session) && session.Commit();
        if (!isWriteSucceeded) {
            return false;
        }
    }

    auto session = DriveAPI.BuildTx<NSQL::Writable>();

    // Send rejection push
    if (userData.GetStatus() == NDrive::UserStatusOnboarding) {
        const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();
        auto tag = MakeAtomicShared<TUserPushTag>("reg_push_rejected");
        tag->SetComment("from registration robot");
        auto optionalExternalUserId = TUserOriginTag::GetExternalUserId(userData.GetUserId(), *Server, session);
        if (optionalExternalUserId) {
            tag->SetExternalUserId(*optionalExternalUserId);
        }
        if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session)) {
            MaybeNotify("Не могу добавить тег reg_push_rejected на пользователя " + userData.GetHRReport() + ": " + session.GetStringReport());
            return false;
        }
    }

    if (!DriveAPI.GetUsersData()->SetChatToShow(userData.GetUserId(), chatRobotPtr->GetFullChatId(topic), false, operatorUserId, Server, session)) {
        MaybeNotify("Не могу выставить правильный редирект в чат: " + userData.GetHRReport() + ": " + session.GetStringReport());
        return false;
    }

    // Update user status
    userData.UpdateStatus(NDrive::UserStatusRejected);
    if (!DriveAPI.GetUsersData()->UpdateUser(userData, operatorUserId, session) || !session.Commit()) {
        return false;
    }

    return true;
}

bool TUserRegistrationManager::RejectWithTag(TDriveUserData& userData, const TString& operatorUserId, const TString& tagName, const TString& comment, const TSimpleChatBot* chatRobotPtr, const TString& topic) const {
    if (!Reject(userData, operatorUserId, chatRobotPtr, topic)) {
        return false;
    }
    const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();
    auto session = DriveAPI.BuildTx<NSQL::Writable>();
    auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>(tagName, comment);
    if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session) || !session.Commit()) {
        MaybeNotify("Не могу сделать reject с тегом " + tagName + " и комментом " + comment + " пользователя " + userData.GetHRReport() + ": " + session.GetStringReport());
        return false;
    }
    return true;
}

bool TUserRegistrationManager::QueueForScreening(TDriveUserData& userData, const TString& operatorUserId, const TString& tagId, const TString& comment) const {
    userData.UpdateStatus(NDrive::UserStatusScreening);
    MaybeNotify("Отправляем в скрининг: " + userData.GetHRReport() + " по причине " + tagId + " с комментарием " + comment);
    auto session = DriveAPI.BuildTx<NSQL::Writable>();

    if (tagId != "") {
        const auto& tagsManager = DriveAPI.GetTagsManager().GetUserTags();
        auto tag = IJsonSerializableTag::BuildWithComment<TRegistrationUserTag>(tagId, comment);
        if (!tagsManager.AddTag(tag, operatorUserId, userData.GetUserId(), Server, session)) {
            MaybeNotify("Не могу добавить тег " + tagId + " на пользователя " + userData.GetHRReport() + ": " + session.GetStringReport());
            return false;
        }
    }
    if (!DriveAPI.GetUsersData()->UpdateUser(userData, operatorUserId, session) || !session.Commit()) {
        MaybeNotify("cannot UpdateUser " + userData.GetUserId()  + ": " + session.GetStringReport());
        return false;
    }

    return true;
}

void TUserRegistrationManager::MaybeNotify(const TString& message) const {
    DEBUG_LOG << "Notification: " << message << Endl;
    if (Config.GetNotifierName()) {
        NDrive::INotifier::Notify(Server->GetNotifier(Config.GetNotifierName()), message);
    }
    NDrive::TEventLog::Log("RegistrationManagerNotification", NJson::ToJson(NJson::JsonString(message)));
}

void TUserRegistrationManager::GetDocumentsResubmitMask(const TYangDocumentVerificationAssignment& assignment, ui32& resubmitMask, bool& isOk, ui32& failureMask, const TInstant actuality) const {
    resubmitMask = 0;
    isOk = true;

    TSet<TString> photoIds;
    photoIds.insert(assignment.GetLicenseBackId());
    photoIds.insert(assignment.GetLicenseFrontId());
    photoIds.insert(assignment.GetPassportBiographicalId());
    photoIds.insert(assignment.GetPassportRegistrationId());
    photoIds.insert(assignment.GetPassportSelfieId());
    TUserDocumentPhotosDB::TFetchResult fetchResult;
    if (DriveAPI.HasDocumentPhotosManager()) {
        fetchResult = DriveAPI.GetDocumentPhotosManager().GetUserPhotosDB().FetchInfo(photoIds, actuality);
    }

    for (auto&& photoId : photoIds) {
        auto documentPtr = fetchResult.GetResultPtr(photoId);
        if (documentPtr == nullptr) {
            resubmitMask |= NUserDocument::InvalidResubmitMaskFlag;
            isOk = false;
            MaybeNotify("Неконсистентность в задании " + assignment.GetId() + ". Нет такого фото: " + photoId);
        } else if (documentPtr->GetVerificationStatus() != NUserDocument::EVerificationStatus::Ok) {
            isOk = false;
            resubmitMask |= documentPtr->GetType() * documentPtr->IsResubmitNeeded();
            failureMask |= documentPtr->GetType();
        }
    }
}

bool TUserRegistrationManager::BanUser(const TString& userId, const TString& operatorId, const NBans::EReason preliminaryReason, NDrive::TEntitySession& session) const {
    auto userFetchResult = DriveAPI.GetUsersData()->FetchInfo(userId, session);
    auto userPtr = userFetchResult.GetResultPtr(userId);
    if (!userPtr) {
        session.SetErrorInfo("user", "not_exist", EDriveSessionResult::InternalError);
        return false;
    }
    auto userData = *userPtr;
    return BanUser(userData, operatorId, preliminaryReason, session);
}

bool TUserRegistrationManager::GetBanChatFromTag(TString& banChatId, const TDBTag& tag, const ITagsMeta::TTagDescriptionsByName& descriptions) const {
    auto description =  descriptions.find(tag->GetName());
    if (description != descriptions.end()) {
        auto problemTagDescription = std::dynamic_pointer_cast<const TUserProblemTag::TDescription>(description->second);
        if (problemTagDescription && !problemTagDescription->GetBanChat().empty()) {
            banChatId = problemTagDescription->GetBanChat();
            return true;
        }
    }
    return false;
}

bool TUserRegistrationManager::UserTagsContain(const TVector<TDBTag>& tags, const TString& soughtTag, const ITagsMeta::TTagDescriptionsByName& descriptions, TString& banChatId) const {
    for (auto&& tag : tags) {
        if (tag->GetName() == soughtTag) {
            GetBanChatFromTag(banChatId, tag, descriptions);
            return true;
        }
    }
    return false;
}

bool TUserRegistrationManager::BanUser(TDriveUserData& userData, const TString& operatorId, const NBans::EReason preliminaryReason, NDrive::TEntitySession& session) const {
    auto userId = userData.GetUserId();

    TVector<TDBTag> userTags;
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {}, userTags, session)) {
        session.SetErrorInfo("tags", "get_user_tags", EDriveSessionResult::InternalError);
        return false;
    }
    auto banChatId = Config.GetBanChatId();
    auto tagDescriptions = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User, {TUserProblemTag::TypeName});

    NBans::EReason reason = preliminaryReason;
    if (
        UserTagsContain(userTags, "blocked_speed_asshole", tagDescriptions, banChatId) ||
        UserTagsContain(userTags, "blocked_speed_asshole_3day", tagDescriptions, banChatId) ||
        UserTagsContain(userTags, "blocked_speed_asshole_10day", tagDescriptions, banChatId)  ||
        UserTagsContain(userTags, "blocked_speed_asshole_1week", tagDescriptions, banChatId) ||
        UserTagsContain(userTags, "blocked_speed_asshole_forever", tagDescriptions, banChatId)
    ) {
        reason = NBans::EReason::SpeedAsshole;
    } else if (UserTagsContain(userTags, "blocked_duplicate_driver_license", tagDescriptions, banChatId)) {
        reason = NBans::EReason::DuplicateLicense;
    } else if (UserTagsContain(userTags, "blocked_duplicate_namber_pass", tagDescriptions, banChatId)) {
        reason = NBans::EReason::DuplicatePassport;
    } else if (UserTagsContain(userTags, "blocked_old_license", tagDescriptions, banChatId)) {
        reason = NBans::EReason::OldLicense;
    } else if (UserTagsContain(userTags, "blocked_for_resubmit", tagDescriptions, banChatId)) {
        reason = NBans::EReason::Resubmit;
    }

    auto chatRobot = Server->GetChatRobot(banChatId);
    auto chatRobotPtr = dynamic_cast<const TRegistrationChatBot*>(chatRobot.Get());
    if (!chatRobotPtr) {
        return false;
    }

    if (
        (userData.GetStatus() != NDrive::UserStatusBlocked || reason == NBans::EReason::OldLicense) &&
        chatRobotPtr->IsExistsForUser(userId, "") &&
        userData.GetStatus() != NDrive::UserStatusRejected &&
        reason != NBans::EReason::Resubmit
    ) {
        if (!chatRobotPtr->SendBanMessages(userId, "", session, operatorId)) {
            session.SetErrorInfo("new_chat", "send_messages", EDriveSessionResult::InternalError);
            return false;
        }
    }

    // Update user status
    userData.UpdateStatus(NDrive::UserStatusBlocked);

    if (!DriveAPI.GetUsersData()->UpdateUser(userData, operatorId, session)) {
        return false;
    }

    auto problemTags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, MakeVector(NContainer::Keys(tagDescriptions)), session);
    if (!problemTags) {
        return false;
    }
    TVector<TDBTag> updatedTags;
    for (auto&& tag : *problemTags) {
        auto description = tagDescriptions.find(tag->GetName());
        if (description != tagDescriptions.end()) {
            auto problemTagDescription = std::dynamic_pointer_cast<const TUserProblemTag::TDescription>(description->second);
            if (problemTagDescription && problemTagDescription->GetBanDuration()) {
                tag->SetSLAInstant(TInstant::Now() + problemTagDescription->GetBanDuration());
                updatedTags.push_back(std::move(tag));
            }
        }
    }
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagsData(updatedTags, userId, session)) {
        return false;
    }
    return true;
}

bool TUserRegistrationManager::UnbanUser(const TString& userId, const TString& operatorId, NDrive::TEntitySession& session) const {
    auto userFetchResult = DriveAPI.GetUsersData()->FetchInfo(userId, session);
    auto userPtr = userFetchResult.GetResultPtr(userId);
    if (!userPtr) {
        session.SetErrorInfo("user", "not_exist", EDriveSessionResult::InternalError);
        return false;
    }
    if (userPtr->GetStatus() != NDrive::UserStatusBlocked && userPtr->GetStatus() != NDrive::UserStatusActive) {
        session.SetErrorInfo("user", "not_blocked", EDriveSessionResult::DataCorrupted);
        return false;
    }

    auto user = *userPtr;
    if (auto unbanStatus = GetUnbanStatus(user, session)) {
        user.UpdateStatus(*unbanStatus);
        if (!DriveAPI.GetUsersData()->UpdateUser(user, operatorId, session)) {
            return false;
        }
        return true;
    }
    return false;
}

TString TUserRegistrationManager::GetDisabledStatus(const TString& userId) const {
    auto userRoles = DriveAPI.GetUsersData()->GetRoles().GetCachedUserRoles(userId);
    if (userRoles->GetRoles().empty()) {
        return NDrive::UserStatusOnboarding;
    }
    for (auto&& role : userRoles->GetRoles()) {
        if (Config.GetServiceRoles().contains(role.GetRoleId())) {
            return NDrive::UserStatusPassive;
        }
    }
    return NDrive::UserStatusOnboarding;
}

bool TUserRegistrationManager::IsManuallyApprovedUser(const TString& userId, NDrive::TEntitySession& session) const {
    auto manuallyApprovedTags = SplitString(Server->GetSettings().GetValueDef<TString>("user_registration.manually_approved_tags", "user_registered_manually,user_manual_check_ОК"), ",");
    auto optionalTags = Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TVector<TString>{ userId }, manuallyApprovedTags, session);
    return Yensured(optionalTags)->size() > 0;
}

TMaybe<TString> TUserRegistrationManager::GetUnbanStatus(const TDriveUserData& userData, NDrive::TEntitySession& session) const {
    if (IsManuallyApprovedUser(userData.GetUserId(), session)) {
        return NDrive::UserStatusActive;
    }

    TString disabledStatus = GetDisabledStatus(userData.GetUserId());

    TMaybe<TString> resolutionByTag;
    TUserSetting userSetting(userData.GetUserId());
    auto resolutionStatus = userSetting.GetValue(Config.GetVerificationStatusTag(), Config.GetVerificationStatusField(), session);
    if (resolutionStatus) {
        resolutionByTag = *resolutionStatus;
    } else {
        if (resolutionStatus.GetError() != TUserSettingStatus::NotFound) {
            return Nothing();
        }
        auto precheckResults = userSetting.GetValue(Config.GetVerificationStatusTag(), "precheck_results", session);
        if (!precheckResults && precheckResults.GetError() != TUserSettingStatus::NotFound) {
            return Nothing();
        }
        if (precheckResults) {
            NJson::TJsonValue jsonPrecheckResults;
            TString currentPrecheckResult;
            if (NJson::ReadJsonFastTree(*precheckResults, &jsonPrecheckResults) && jsonPrecheckResults.IsArray()) {
                ui64 precheckTs = 0, currentPrecheckTs = 0;
                for (auto&& precheckInfoJson : jsonPrecheckResults.GetArray()) {
                    if (NJson::ParseField(precheckInfoJson, "ts", currentPrecheckTs, true)
                        && currentPrecheckTs > precheckTs
                        && NJson::ParseField(precheckInfoJson, "action", currentPrecheckResult, true))
                    {
                        precheckTs = currentPrecheckTs;
                        resolutionByTag = currentPrecheckResult;
                    }
                }
            }
        }
    }
    if (resolutionByTag) {
        if (*resolutionByTag == Config.GetExpectedResolutionStatus()) {
            return NDrive::UserStatusActive;
        } else {
            NDrive::TEventLog::Log("GetUnbanStatus", NJson::TMapBuilder
                ("user_id", userData.GetUserId())
                ("tag_resolution", *resolutionByTag)
            );
            return disabledStatus;
        }
    }

    Y_ENSURE_BT(DriveAPI.HasDocumentPhotosManager());
    auto yas = DriveAPI.GetDocumentPhotosManager().GetDocumentVerificationAssignments().GetAllAssignmentsForUser(userData.GetUserId());
    if (yas.size() == 0) {
        NDrive::TEventLog::Log("GetUnbanStatus", NJson::TMapBuilder
            ("user_id", userData.GetUserId())
            ("latest_assignment", "not found")
        );
        return disabledStatus;
    }
    auto latestYa = yas[0];
    for (size_t i = 1; i < yas.size(); ++i) {
        if (yas[i].GetCreatedAt() > latestYa.GetCreatedAt() && yas[i].GetLicenseBackId() && yas[i].GetLicenseFrontId() && yas[i].GetPassportBiographicalId()) {
            latestYa = yas[i];
        }
    }
    if (latestYa.GetIsFraud() == TYangDocumentVerificationAssignment::EFraudStatus::NotFraud) {
        ui32 resubmitMask, failureMask;
        bool isOk;
        GetDocumentsResubmitMask(latestYa, resubmitMask, isOk, failureMask, Now());
        if (!isOk || resubmitMask) {
            NDrive::TEventLog::Log("GetUnbanStatus", NJson::TMapBuilder
                ("user_id", userData.GetUserId())
                ("is_ok", ToString(isOk))
                ("resubmit_mask", ToString(resubmitMask))
                ("latest_assignment", latestYa.GetId())
            );
            return disabledStatus;
        }
        return NDrive::UserStatusActive;
    }
    NDrive::TEventLog::Log("GetUnbanStatus", NJson::TMapBuilder
        ("user_id", userData.GetUserId())
        ("latest_assignment", latestYa.GetId())
        ("latest_assignment_fraud", latestYa.GetIsFraud())
    );
    return disabledStatus;
}

ui32 TUserRegistrationManager::GetUserViolationScore(const TVector<TDBTag>& tags) const {
    const auto& tagsMeta = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta();

    ui32 pointsSum = 0;
    for (auto&& tag : tags) {
        auto tagPtr = tag.GetTagAs<TUserProblemTag>();
        if (!!tagPtr) {
            pointsSum += tagPtr->GetPoints(tagsMeta);
        }
    }

    return pointsSum;
}

ui32 TUserRegistrationManager::GetUserViolationScore(const TString& userId, NDrive::TEntitySession& session) const {
    TVector<TDBTag> userTags;
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {}, userTags, session)) {
        return false;
    }
    return GetUserViolationScore(userTags);
}

bool TUserRegistrationManager::HasInstantBanTags(const TVector<TDBTag>& tags) const {
    for (auto&& tag : tags) {
        auto tagPtr = tag.GetTagAs<TUserProblemTag>();
        if (!!tagPtr && tagPtr->GetInstantBan(Server->GetDriveAPI()->GetTagsManager().GetTagsMeta())) {
            return true;
        }
    }
    return false;
}

bool TUserRegistrationManager::HasInstantBanTags(const TString& userId, NDrive::TEntitySession& session) const {
    TVector<TDBTag> userTags;
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {}, userTags, session)) {
        return false;
    }
    return HasInstantBanTags(userTags);
}

bool TUserRegistrationManager::IsSubjectToBan(const TString& userId, NDrive::TEntitySession& session) const {
    TVector<TDBTag> performedTags;
    if (!DriveAPI.GetTagsManager().GetDeviceTags().RestorePerformerTags(TVector<TString>({ userId }), performedTags, session)) {
        ERROR_LOG << "can't restore performed tags for user " << userId << Endl;
        return false;
    }

    TVector<TDBTag> userTags;
    if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(userId, {}, userTags, session)) {
        return false;
    }

    bool suspendBan = performedTags.size() > 0;
    if (suspendBan && HasInstantBanTags(userTags)) {
        suspendBan = false;
    }

    return (!suspendBan && GetUserViolationScore(userTags) >= Config.GetAutoBanThreshold());
}

bool TUserRegistrationManager::IsSubjectToFinishBan(const TString& userId, NDrive::TEntitySession& session) const {
    auto fetchResult = DriveAPI.GetUsersData()->FetchInfo(userId, session);
    auto userPtr = fetchResult.GetResultPtr(userId);
    if (!userPtr) {
        return false;
    }
    if (userPtr->GetStatus() != NDrive::UserStatusBlocked) {
        return false;
    }

    return (GetUserViolationScore(userId, session) < Config.GetAutoBanThreshold());
}

ui32 TUserRegistrationManager::GetBanThreshold() const {
    return Config.GetAutoBanThreshold();
}

bool TUserRegistrationManager::GetTwinsByDocuments(const TSet<TString>& originalUsers, TSet<TString>& resultSet) const {
    auto userData = Yensured(DriveAPI.GetUsersData());
    auto session = userData->BuildSession(true);
    auto usersFetchResult = userData->FetchInfo(originalUsers, session);
    for (auto&& userIt : usersFetchResult) {
        TVector<TString> hashes;
        if (userIt.second.GetPassportNumberHash() && userIt.second.GetPassportNumberHash() != EmptyFieldHash) {
            hashes.push_back(userIt.second.GetPassportNumberHash());
        }
        if (userIt.second.GetDrivingLicenseNumberHash() && userIt.second.GetDrivingLicenseNumberHash() != EmptyFieldHash) {
            hashes.push_back(userIt.second.GetDrivingLicenseNumberHash());
        }
        for (auto&& hash : hashes) {
            auto optionalTwinIds = DriveAPI.GetUsersData()->GetDocumentOwnersByHash(hash, session);
            if (!optionalTwinIds) {
                session.Check();
                return false;
            }
            const auto& twinIds = *optionalTwinIds;
            for (auto&& twinId : twinIds) {
                resultSet.emplace(twinId);
            }
        }
    }
    return true;
}

bool TUserRegistrationManager::GetTwinsByUserFields(const TSet<TString>& originalUsers, TSet<TString>& resultSet) const {
    auto userData = DriveAPI.GetUsersData();
    auto session = userData->BuildSession(true);
    auto usersFetchResult = userData->FetchInfo(originalUsers, session);
    for (auto&& [id, user] : usersFetchResult) {
        auto filledFields = user.GetFilledFields();
        if (!filledFields) {
            continue;
        }
        ui32 count = 0;
        NSQL::TQueryOptions queryOptions;
        if (filledFields & ToUnderlying(TDriveUserData::EField::FirstName)) {
            queryOptions.AddGenericCondition("first_name", user.GetFirstName());
            count += 1;
        }
        if (filledFields & ToUnderlying(TDriveUserData::EField::LastName)) {
            queryOptions.AddGenericCondition("last_name", user.GetLastName());
            count += 1;
        }
        if (filledFields & ToUnderlying(TDriveUserData::EField::PName)) {
            queryOptions.AddGenericCondition("patronymic_name", user.GetPName());
            count += 1;
        }
        if (filledFields & ToUnderlying(TDriveUserData::EField::Phone)) {
            queryOptions.AddGenericCondition("phone", user.GetPhone());
            count += 1;
        }
        if (count < 3) {
            ERROR_LOG << "GetTwinsByUserFields: skip " << id << ": weak mask " << filledFields << Endl;
            continue;
        }
        auto dupFetchResult = userData->FetchInfo(session, queryOptions);
        for (auto&& [dupId, dup] : dupFetchResult) {
            if (dupId == id) {
                continue;
            }
            resultSet.insert(dupId);
        }
    }

    return true;
}

bool TUserRegistrationManager::ActualizeBlacklistBans(const TString& operatorUserId, const TVector<TString>& tagNames, const TString& dupsTagName, const TBlacklistOptions& options) const {
    TSet<TString> banSeedUsers;
    TSet<TString> alreadyBannedByBlacklist;
    if (!GetUsersWithTag(tagNames, banSeedUsers) || !GetUsersWithTag({ dupsTagName }, alreadyBannedByBlacklist)) {
        return false;
    }
    if (options.SetComment) {
        bool result = true;
        for (auto&& userId : banSeedUsers) {
            result &= ActualizeBlacklistBans(operatorUserId, { userId }, alreadyBannedByBlacklist, dupsTagName, options);
        }
        return result;
    } else {
        return ActualizeBlacklistBans(operatorUserId, banSeedUsers, alreadyBannedByBlacklist, dupsTagName, options);
    }
}

bool TUserRegistrationManager::ActualizeBlacklistBans(const TString& operatorUserId, const TSet<TString>& banSeedUsers, const TSet<TString>& alreadyBannedByBlacklist, const TString& dupsTagName, const TBlacklistOptions& options) const {
    TSet<TShortUserDevice> bannedUserDevices;
    {
        auto session = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!Server->GetUserDevicesManager()->GetUserDevices(banSeedUsers, bannedUserDevices, session)) {
            return false;
        }
    }
    TSet<TString> bannedDeviceIds;
    for (auto&& device : bannedUserDevices) {
        bannedDeviceIds.insert(device.GetDeviceId());
    }

    auto traits = options.Traits;
    TSet<TString> connectedUsers;
    {
        auto session = Server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (traits & NUserConnections::ESimilarityTraits::Device && !Server->GetUserDevicesManager()->GetUsersByDeviceIds(bannedDeviceIds, connectedUsers, session, options.VerifiedDeviceIdOnly)) {
            return false;
        }
    }
    if (traits & NUserConnections::ESimilarityTraits::DocumentHashes && !GetTwinsByDocuments(banSeedUsers, connectedUsers)) {
        return false;
    }
    if (traits & NUserConnections::ESimilarityTraits::UserFields && !GetTwinsByUserFields(banSeedUsers, connectedUsers)) {
        return false;
    }

    const auto& tagsManager = Server->GetDriveAPI()->GetTagsManager().GetUserTags();
    for (auto&& userId : connectedUsers) {
        if (!options.Direct) {
            break;
        }
        if (banSeedUsers.contains(userId)) {
            // A seed user, banned by default. No need to do anything.
            continue;
        }
        if (alreadyBannedByBlacklist.contains(userId)) {
            // A twin user. Already banned.
            continue;
        }
        auto session = Server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        auto tag = Server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(dupsTagName);
        if (!tag) {
            continue;
        }
        if (options.SetComment) {
            tag->SetComment("duplicated with " + JoinSeq(",", banSeedUsers));
        }
        if (!tagsManager.AddTag(tag, operatorUserId, userId, Server, session) || !session.Commit()) {
            MaybeNotify("Не могу добавить тег " + dupsTagName + " на пользователя " + userId);
        } else {
            MaybeNotify("Забанен из-за похожести по кого-то блеклисту, device: " + userId);
        }
    }

    for (auto&& userId : alreadyBannedByBlacklist) {
        if (!options.Reverse) {
            break;
        }
        if (banSeedUsers.contains(userId)) {
            // A seed user. No need to do anything.
            continue;
        }
        if (connectedUsers.contains(userId)) {
            // Should still remain a banned user.
            continue;
        }

        auto session = tagsManager.BuildSession();
        auto optionalTags = tagsManager.RestoreTags(TVector<TString>{ userId }, { dupsTagName }, session);
        if (!optionalTags) {
            ERROR_LOG << "cannot RestoreTags: " << session.GetStringReport() << Endl;
            return false;
        }
        for (auto&& tag : *optionalTags) {
            if (!Server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(tag, operatorUserId, Server, session, true) || !session.Commit()) {
                MaybeNotify("Не могу разбанить по блеклисту: снять тег " + dupsTagName + " с пользователя " + userId + ": " + session.GetStringReport());
            } else {
                MaybeNotify("Разбан по блеклисту: сняли тег " + dupsTagName + " с пользователя " + userId);
            }
        }
    }

    return true;
}

bool TUserRegistrationManager::GetUsersWithTag(const TVector<TString>& tagNames, TSet<TString>& userIds) const {
    TVector<TDBTag> dbTags;
    auto session = DriveAPI.BuildTx<NSQL::ReadOnly>();
    if (!DriveAPI.GetTagsManager().GetUserTags().RestoreTags({}, tagNames, dbTags, session)) {
        return false;
    }

    userIds.clear();
    for (auto&& tag : dbTags) {
        userIds.insert(tag.GetObjectId());
    }

    return true;
}

TBlacklistExternal* TUserRegistrationManager::GetBlacklistExternal() const {
    return BlacklistExternal.Get();
}

TString TUserRegistrationManager::GetDocumentNumberHash(const TString& number) const {
    return Hasher->GetHash(ToUpperUTF8(number));
}

bool TUserRegistrationManager::ReaskDocumentPhotos(const TString& userId, const ui32 resubmitMask, NDrive::TEntitySession& tx, const TString& operatorId, const TString& chatId, const TVector<TDocumentResubmitOverride>& resubmitOverride) const {
    auto userFR = DriveAPI.GetUsersData()->FetchInfo(userId, tx);
    if (userFR.empty()) {
        tx.SetErrorInfo("UserRegistrationManager::ReaskDocumentPhotos", "cannot get user info: " + userId, EDriveSessionResult::InternalError);
        return false;
    }
    const TString& reaskChatId = chatId ? chatId : Config.GetChatId();
    auto userPtr = userFR.GetResultPtr(userId);
    if (userPtr->GetStatus() == NDrive::UserStatusActive) {
        auto tag = IJsonSerializableTag::BuildWithComment<TUserProblemTag>("blocked_for_resubmit", "Тег для введения в чат по завершению аренды");
        {
            if (!DriveAPI.GetTagsManager().GetUserTags().AddTag(tag, operatorId, userId, Server, tx)) {
                MaybeNotify("Не могу повесить тег 'blocked_for_resubmit': " + userId);
                return false;
            }
            if (!DriveAPI.GetUsersData()->SetChatToShow(userId, reaskChatId, false, operatorId, Server, tx)) {
                MaybeNotify("Не могу выставить правильный редирект в чат: " + userId);
                return false;
            }
        }
    }

    auto chatRobot = Server->GetChatRobot(reaskChatId);
    auto chatRobotPtr = dynamic_cast<const TRegistrationChatBot*>(chatRobot.Get());
    if (!chatRobotPtr) {
        tx.SetErrorInfo("UserRegistrationManager::ReaskDocumentPhotos", "cannot get chat robot: " + reaskChatId, EDriveSessionResult::InternalError);
        return false;
    }

    if (Config.GetIsPrimaryDBUsed()) {
        return chatRobotPtr->AskToResubmit(userId, "", resubmitMask, tx, operatorId, true, resubmitOverride);
    }
    auto session = chatRobotPtr->BuildChatEngineSession();
    if (!chatRobotPtr->AskToResubmit(userId, "", resubmitMask, session, operatorId, true, resubmitOverride)) {
        tx.MergeErrorMessages(session.GetMessages(), "chat_session");
        tx.SetErrorInfo("UserRegistrationManager::ReaskDocumentPhotos", "cannot AskToResubmit", EDriveSessionResult::InternalError);
        return false;
    }
    if (!session.Commit()) {
        tx.MergeErrorMessages(session.GetMessages(), "chat_session");
        tx.SetErrorInfo("UserRegistrationManager::ReaskDocumentPhotos", "cannot commit chat session", EDriveSessionResult::InternalError);
        return false;
    }
    return true;
}

TString TUserRegistrationManager::GetNotifierName() const {
    return Config.GetNotifierName();
}

TUserRegistrationManagerConfig TUserRegistrationManager::GetConfig() const {
    return Config;
}

TString TUserRegistrationManager::GetOriginChat(const TYangDocumentVerificationAssignment& assignment, const TInstant actuality) const {
    TSet<TString> photoIds;
    photoIds.insert(assignment.GetLicenseBackId());
    photoIds.insert(assignment.GetLicenseFrontId());
    photoIds.insert(assignment.GetPassportBiographicalId());
    photoIds.insert(assignment.GetPassportRegistrationId());
    photoIds.insert(assignment.GetPassportSelfieId());
    TUserDocumentPhotosDB::TFetchResult fetchResult;
    if (DriveAPI.HasDocumentPhotosManager()) {
        fetchResult = DriveAPI.GetDocumentPhotosManager().GetUserPhotosDB().FetchInfo(photoIds, actuality);
    }

    TInstant lastSubmittedAt = TInstant::Zero();
    TString result = "registration";
    for (auto&& it : fetchResult) {
        if (it.second.GetSubmittedAt() > lastSubmittedAt) {
            lastSubmittedAt = it.second.GetSubmittedAt();
            result = it.second.GetOriginChat();
        }
    }

    return result;
}
