#pragma once

#include "account_checker.h"
#include "config.h"
#include "hasher.h"
#include "ifaces.h"

#include <drive/backend/chat_robots/registration/bot.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/users/blacklist_external.h>

namespace NRegistrarUtil {
    TString ToTitle(const TString& str);
};

class TUserRegistrationManager : public IUserRegistrationManager {
public:
    struct TLicenseExpirationDelay {
    public:
        TRange<TInstant> Range;
        std::variant<TDuration, TInstant> Delay;
        bool ApplyForForeign = true;

    public:
        bool IsMatching(const TInstant& validToDate) const;
        bool ShouldIgnore(const TInstant& validToDate) const;
    };

private:
    const TUserRegistrationManagerConfig Config;  // for sending reasks
    const NDrive::IServer* Server;
    const TDriveAPI& DriveAPI;

    THolder<TSensitiveDataHasher> Hasher;
    THolder<TAccountChecker> Scorer;
    THolder<TBlacklistExternal> BlacklistExternal;

private:
    bool QueueForScreening(TDriveUserData& user, const TString& operatorUserId, const TString& tagId, const TString& comment) const;

    void MaybeNotify(const TString& message) const;

    void GetDocumentsResubmitMask(const TYangDocumentVerificationAssignment& assignment, ui32& resubmitMask, bool& isOk, ui32& failureMask, const TInstant actuality) const;

    bool CheckDocumentsDataConsistency(TDriveUserData& userData, const TString& operatorUserId, const TUserPassportData& passportData, const TUserDrivingLicenseData& drivingLicenseData) const;
    bool CheckDates(TDriveUserData& userData, const TString& operatorUserId, TUserPassportData& passportData, TUserDrivingLicenseData& drivingLicenseData) const;
    bool CheckDuplicates(TDriveUserData& userData, const TString& operatorUserId) const;
    bool CheckAccountData(TDriveUserData& userData, const TString& operatorUserId, const TUserPassportData& passport, const TUserDrivingLicenseData& drivingLicense, const TSimpleChatBot* chatRobotPtr, const TString& topic) const;
    bool CheckSamePersonProfiles(TDriveUserData& userData, const TString& operatorUserId, const TString& soughtHash, const TString& tagName) const;
    bool CheckTooManyResubmits(TDriveUserData& userData, const TString& operatorUserId, const TSimpleChatBot* chatRobotPtr, const TString& topic) const;
    bool CheckSelfieScreencapStatuses(TDriveUserData& userData, const TString& operatorUserId, TYangDocumentVerificationAssignment& recentAssignment, const TSimpleChatBot* chatRobotPtr, const TString& topic) const;
    bool CheckDeviceSimilarityToBanned(TDriveUserData& userData, const TString& operatorUserId) const;

    bool GetPersonalData(TDriveUserData& userData, const TString& operatorUserId, TUserPassportData& passportData, TUserDrivingLicenseData& drivingLicenseData) const;
    TMaybe<TString> GetUnbanStatus(const TDriveUserData& userData, NDrive::TEntitySession& session) const;
    TString GetDisabledStatus(const TString& userId) const;

    bool UpdateNames(TDriveUserData& userData, const TString& operatorUserId, const TUserPassportData& passportData, const TUserDrivingLicenseData& drivingLicenseData) const;

    ui32 GetUserViolationScore(const TString& userId, NDrive::TEntitySession& session) const;
    ui32 GetUserViolationScore(const TVector<TDBTag>& tags) const;
    ui32 GetTwinViolationScore(const TString& userId) const;
    ui32 GetTwinViolationScore(const TVector<TDBTag>& tags) const;

    bool HasInstantBanTags(const TString& userId, NDrive::TEntitySession& session) const;
    bool HasInstantBanTags(const TVector<TDBTag>& tags) const;

    bool CheckMultAccount(TDriveUserData& userData, const TString& operatorUserId, const TString& twinId, const TString& tagName, const TString& commentPrefix) const;

    bool GetBanChatFromTag(TString& banChatId, const TDBTag& tag, const ITagsMeta::TTagDescriptionsByName& descriptions) const;

    bool UserTagsContain(const TVector<TDBTag>& tags, const TString& soughtTag, const ITagsMeta::TTagDescriptionsByName& descriptions, TString& banChatId) const;
    TSet<TString> GetUserConnections(const TString& userId, const TString& tagName, NDrive::TEntitySession& session) const;

    bool GetUsersWithTag(const TVector<TString>& tagNames, TSet<TString>& userIds) const;

    bool IsManuallyApprovedUser(const TString& userId, NDrive::TEntitySession& session) const;
    bool CleanupRegistrationTags(const TString& userId, const TString& operatorUserId, bool ignoreTagFailures = false) const;
    bool CleanupProblemTags(const TString& userId, const TString& operatorUserId, NDrive::TEntitySession& session) const;
    bool MoveTags(const TString& fromUserId, const TString& toUserId, const TString& operatorUserId, NDrive::TEntitySession& session) const;
    bool MoveBonuses(const TString& fromUserId, const TString& toUserId, NDrive::TEntitySession& session) const;
    bool MoveRoles(const TString& fromUserId, const TString& toUserId, NDrive::TEntitySession& session) const;
    bool AddRolesFromPool(const TString& userId, const TVector<TUserRole>& pool, NDrive::TEntitySession& session) const;
    TString GetOriginChat(const TYangDocumentVerificationAssignment& ya, const TInstant actuality) const;

public:
    TUserRegistrationManager(const TUserRegistrationManagerConfig& config, const NDrive::IServer* server);

    virtual bool HandleVerifiedAssignment(TYangDocumentVerificationAssignment& recentAssignment, const TString& operatorUserId, TInstant actuality, const TUserPassportData* passportExt = nullptr, const TUserDrivingLicenseData* drivingLicenseExt = nullptr) const override;
    virtual bool DoBackgroundCheckEntrance(TDriveUserData& userData, const TString& operatorUserId, const TUserPassportData* passportExt = nullptr, const TUserDrivingLicenseData* drivingLicenseExt = nullptr, TString chatId = "", TString topic = "") const override;

    virtual bool Approve(TDriveUserData& userData, const TString& operatorUserId, const TSimpleChatBot* chatRobotPtr, const TString& topic) const override;

    virtual bool BanUser(const TString& userId, const TString& operatorId, const NBans::EReason reason, NDrive::TEntitySession& session) const override;
    virtual bool BanUser(TDriveUserData& userData, const TString& operatorId, const NBans::EReason reason, NDrive::TEntitySession& session) const;
    virtual bool UnbanUser(const TString& userId, const TString& operatorId, NDrive::TEntitySession& session) const override;
    virtual bool IsSubjectToBan(const TString& userId, NDrive::TEntitySession& session) const override;
    virtual bool IsSubjectToFinishBan(const TString& userId, NDrive::TEntitySession& session) const override;
    virtual ui32 GetBanThreshold() const override;

    virtual bool Reject(TDriveUserData& userId, const TString& operatorUserId, const TSimpleChatBot* chatRobotPtr, const TString& topic) const;
    virtual bool RejectWithTag(TDriveUserData& userData, const TString& operatorUserId, const TString& tagName, const TString& comment, const TSimpleChatBot* chatRobotPtr, const TString& topic) const;
    bool ActualizeBlacklistBans(const TString& operatorUserId, const TVector<TString>& tagNames, const TString& dupsTagName, const TBlacklistOptions& options) const override;
    bool ActualizeBlacklistBans(const TString& operatorUserId, const TSet<TString>& seed, const TSet<TString>& alreadyBanned, const TString& dupsTagName, const TBlacklistOptions& options) const;

    virtual bool MoveData(TDriveUserData& userData, const TDriveUserData* twinData, const TString& operatorUserId, NDrive::TEntitySession& session, NDataTransfer::TDataTransferTraits traits = NDataTransfer::TransferAll) const override;

    virtual bool ReaskDocumentPhotos(const TString& userId, const ui32 resubmitMask, NDrive::TEntitySession& session, const TString& operatorId, const TString& chatId = "", const TVector<TDocumentResubmitOverride>& resubmitOverride = {}) const override;

    virtual TString GetNotifierName() const override;
    virtual TUserRegistrationManagerConfig GetConfig() const override;

    virtual TBlacklistExternal* GetBlacklistExternal() const override;
    virtual TString GetDocumentNumberHash(const TString& number) const override;
    bool GetTwinsByDocuments(const TSet<TString>& originalUsers, TSet<TString>& resultSet) const;
    bool GetTwinsByUserFields(const TSet<TString>& originalUsers, TSet<TString>& resultSet) const;

    TMaybe<TSet<TString>> GetTwinsByDocuments(const TSet<TString>& originalUsers, NDrive::TEntitySession& session) const;

    virtual bool DeduceChatRobot(const TString& userId, TString& robotId, TString& topic) const override;
};
