#include "abstract.h"

#include <drive/backend/logging/logging.h>
#include <drive/library/cpp/openssl/rsa.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/exception.h>

#include <library/cpp/regex/pcre/regexp.h>
#include <library/cpp/svnversion/svnversion.h>

#include <util/string/builder.h>

namespace NDrive {
    TString GetVersion() {
        if (TStringBuf tag = GetTag()) {
            return TString(tag);
        } else {
            return TStringBuilder() << GetBranch() << "/r" << GetProgramSvnRevision();
        }
    }

    const TString Version = GetVersion();
}

IServerReportBuilder::IServerReportBuilder(IReplyContext::TPtr context, const TString& processor)
    : HandlerName(processor)
    , Context(context)
{
    if (!!Context) {
        const TCgiParameters& cgi = Context->GetCgiParameters();
        ReqId = cgi.Get("reqid");
        ReportEventLog = cgi.Has("dump", "eventlog");
        DumpEventLog |= ReportEventLog;
    }
}

IServerReportBuilder::IServerReportBuilder(const TCtx& ctx)
    : IServerReportBuilder(ctx.Context, ctx.Handler)
{
    AccessControlAllowOrigin = ctx.AccessControlAllowOrigin;
}

void IServerReportBuilder::AddEvent(TInstant timestamp, NJson::TJsonValue&& e) {
    if (ShouldRecordEventLog()) {
        auto g = MakeThreadSafeGuard();
        DoAddEvent(timestamp, std::move(e));
    }
}

IServerReportBuilder::TEventsGuard IServerReportBuilder::BuildEventGuard(const TString& info) {
    return TEventsGuard(this, info);
}

void IServerReportBuilder::Finish(HttpCodes code) {
    Finish(TCodedException(code));
}

void IServerReportBuilder::Finish(const TCodedException& e) {
    bool expected = false;
    if (Finished.compare_exchange_strong(expected, true)) {
        auto g = MakeThreadSafeGuard();
        DoFinish(e);
    }
}

void IServerReportBuilder::Finish(HttpCodes code, const TString& contentType, const TBuffer& report) {
    bool expected = false;
    if (Finished.compare_exchange_strong(expected, true)) {
        auto g = MakeThreadSafeGuard();
        DoFinish(code, contentType, report, /*writeEventLog=*/true);
    }
}

void IServerReportBuilder::DoFinish(int code, const TString& contentType, const TBuffer& report, bool writeEventLog) {
    TString accessControlAllowOrigin;
    if (AccessControlAllowOrigin) {
        auto origin = ToString(Context->GetRequestData().HeaderInOrEmpty("Origin"));
        if (AccessControlAllowOrigin->Match(origin.c_str())) {
            accessControlAllowOrigin = origin;
        }
    } else {
        accessControlAllowOrigin = "*";
    }
    if (accessControlAllowOrigin) {
        Context->AddReplyInfo("Access-Control-Allow-Origin", accessControlAllowOrigin);
    }
    const TString& reqid = GetReqId();
    if (ReportDebugInfo && reqid) {
        Context->AddReplyInfo("X-Req-Id", reqid);
    }
    if (ReportDebugInfo && Terminated) {
        Context->AddReplyInfo("X-Terminated", "timeout");
    }
    if (ReportDebugInfo && NDrive::Version) {
        Context->AddReplyInfo("X-Version", NDrive::Version);
    }
    Context->AddReplyInfo("Content-Type", contentType);
    if (SecretVersion) {
        TString encrypted;
        if (SecretKey && NOpenssl::AESEncrypt(SecretKey, report, encrypted)) {
            TBuffer r(encrypted.data(), encrypted.size());
            Context->AddReplyInfo("SecretVersion", SecretVersion);
            Context->MakeSimpleReply(r, code);
        } else {
            Context->AddReplyInfo("SecretVersion", "0");
            Context->MakeSimpleReply(report, code);
        }
    } else {
        Context->MakeSimpleReply(report, code);
    }

    NDrive::TUnistatSignals::OnReply(GetHandlerName(), static_cast<HttpCodes>(code), *Context);
    if (writeEventLog) {
        NDrive::TEventLog::Log(NDrive::TEventLog::ResponseInfo, *Context, code, NJson::TMapBuilder
            ("Content-Length", report.size())
            ("Content-Type", contentType)
        );
    }
}

bool IServerReportBuilder::ShouldRecordEventLog() const {
    return DumpEventLog || ReportEventLog;
}

IServerReportBuilder::TEventsGuard::TEventsGuard(IServerReportBuilder& report, const TString& info)
    : TEventsGuard(&report, info)
{
}

IServerReportBuilder::TEventsGuard::TEventsGuard(IServerReportBuilder* report, const TString& info)
    : TEventGuardTraits(info)
    , Report(report)
{
    if (Report && Report->ShouldRecordEventLog()) {
        Report->AddEvent(Start, BuildStartEvent());
    }
}

void IServerReportBuilder::TEventsGuard::AddEvent(NJson::TJsonValue&& e) {
    if (Report) {
        if (!e.IsMap()) {
            NJson::TJsonValue ee = NJson::JSON_MAP;
            ee["event"] = std::move(e);
            e = std::move(ee);
        }
        e["source"] = Name;
        Report->AddEvent(std::move(e));
    }
}

IServerReportBuilder::TEventsGuard::~TEventsGuard() {
    if (Report) {
        if (Report->ShouldRecordEventLog()) {
            const TInstant finish = Now();
            Report->AddEvent(finish, BuildFinishEvent(finish));
        }
    }
}

IServerReportBuilder::TGuard::TGuard(IServerReportBuilder::TPtr report, const int code)
    : Report(report)
    , ContextGuard(report ? report->GetContext() : nullptr)
    , EventLoggerGuard(report)
    , ErrorsInfo(code)
{
    if (Report && Report->ShouldRecordEventLog()) {
        EvLogRequest = MakeHolder<TEventsGuard>(Report.Get(), "FullRequest");
    }
}

IServerReportBuilder::TGuard::~TGuard() {
    try {
        Flush();
    } catch (const std::exception& e) {
        auto reqid = Report ? Report->GetReqId() : "unknown_reqid";
        ALERT_LOG << "cannot flush " << reqid << ": " << FormatExc(e) << Endl;
    }
}

IServerReportBuilder::TGuard& IServerReportBuilder::TGuard::AddEvent(NJson::TJsonValue&& ev) {
    if (Report) {
        Report->AddEvent(std::move(ev));
    }
    return *this;
}

IServerReportBuilder::TEventsGuard IServerReportBuilder::TGuard::BuildEventGuard(const TString& info) {
    return TEventsGuard(Report.Get(), info);
}

IServerReportBuilder::TPtr IServerReportBuilder::TGuard::Release() {
    IServerReportBuilder::TPtr report;
    bool expected = false;
    if (Flushed.compare_exchange_strong(expected, true)) {
        if (EvLogRequest) {
            EvLogRequest->Release();
        }
        report = std::move(Report);
    }
    Y_ASSERT(!Report);
    return report;
}

void IServerReportBuilder::TGuard::Flush() {
    bool expected = false;
    if (Flushed.compare_exchange_strong(expected, true) && Report) {
        EvLogRequest.Destroy();
        Report->Finish(ErrorsInfo);
        Report.Drop();
    }
}
