#pragma once

#include <drive/backend/logging/events.h>
#include <drive/backend/logging/evlog.h>

#include <drive/library/cpp/searchserver/context/replier.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/exception.h>

#include <util/datetime/base.h>
#include <util/generic/cast.h>
#include <util/system/spinlock.h>

class TRegExMatch;

class IServerReportBuilder: public NThreading::IEventLogger {
public:
    struct TCtx {
        IReplyContext::TPtr Context;
        const TRegExMatch* AccessControlAllowOrigin;
        TString Handler;

        explicit operator bool() const {
            return !!Context;
        }
    };
    using TPtr = TAtomicSharedPtr<IServerReportBuilder>;

private:
    R_FIELD(bool, DumpEventLog, false);
    R_FIELD(bool, ReportDebugInfo, false);
    R_FIELD(bool, Terminated, false);
    R_FIELD(TString, SecretVersion);
    R_FIELD(TString, SecretKey);
    R_READONLY(bool, ReportEventLog, false);
    R_READONLY(TString, HandlerName);
    R_READONLY(TString, ReqId);

private:
    TAdaptiveLock Lock;
    std::atomic<bool> Finished = false;

protected:
    IReplyContext::TPtr Context;
    const TRegExMatch* AccessControlAllowOrigin = nullptr;

protected:
    virtual void DoAddEvent(TInstant timestamp, NJson::TJsonValue&& e) = 0;
    virtual void DoFinish(const TCodedException& e) = 0;
    virtual void DoFinish(int code, const TString& contentType, const TBuffer& report, bool writeEventLog);

    auto MakeThreadSafeGuard() const {
        return Guard(Lock);
    }

public:
    IServerReportBuilder(IReplyContext::TPtr context, const TString& processor);
    IServerReportBuilder(const TCtx& ctx);
    virtual ~IServerReportBuilder() = default;

    const IReplyContext* GetContext() const {
        return Context.Get();
    }
    IReplyContext* GetContext() {
        return Context.Get();
    }

    using IEventLogger::AddEvent;
    void AddEvent(TInstant timestamp, NJson::TJsonValue&& e) override;

    void Finish(HttpCodes code);
    void Finish(const TCodedException& e);
    void Finish(HttpCodes code, const TString& contentType, const TBuffer& report);

    bool ShouldRecordEventLog() const override;

public:
    class TEventsGuard: public NThreading::TEventGuardTraits {
    private:
        IServerReportBuilder* Report;

    public:
        void Release() {
            Report = nullptr;
        }

        TEventsGuard(IServerReportBuilder& report, const TString& info);
        TEventsGuard(IServerReportBuilder* report, const TString& info);
        ~TEventsGuard();

        void AddEvent(NJson::TJsonValue&& e);
    };

    class TGuard: public TNonCopyable {
    private:
        std::atomic<bool> Flushed = false;
        IServerReportBuilder::TPtr Report;
        NDrive::TEventLog::TContextGuard ContextGuard;
        NThreading::TEventLoggerGuard EventLoggerGuard;
        TCodedException ErrorsInfo = TCodedException(HTTP_INTERNAL_SERVER_ERROR);
        THolder<TEventsGuard> EvLogRequest;

    public:
        TGuard(IServerReportBuilder::TPtr report, const int code);
        ~TGuard();

        TGuard& AddEvent(NJson::TJsonValue&& ev);
        TEventsGuard BuildEventGuard(const TString& info);

        IServerReportBuilder::TPtr Release();

        IServerReportBuilder::TPtr GetReport() {
            return Report;
        }

        template <class T>
        T& MutableReportAs() {
            return *VerifyDynamicCast<T*>(Report.Get());
        }

        int GetCode() const {
            return ErrorsInfo.GetCode();
        }

        void SetCode(const TCodedException& e) {
            ErrorsInfo = e;
        }

        void SetCode(const int code) {
            ErrorsInfo.SetCode(code);
        }

        const TCodedException& GetErrors() const {
            return ErrorsInfo;
        }

        TCodedException& MutableErrors() {
            return ErrorsInfo;
        }

        const TString& GetReqId() const {
            return Report ? Report->GetReqId() : Default<TString>();
        }

    private:
        void Flush();
    };

public:
    TEventsGuard BuildEventGuard(const TString& info);
};

using TEventsGuard = IServerReportBuilder::TEventsGuard;
