#include "json.h"

#include <drive/backend/logging/events.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>

#include <rtline/library/json/cast.h>

#include <util/digest/fnv.h>
#include <util/stream/buffer.h>
#include <util/string/type.h>
#include <util/system/hostname.h>

namespace {
    constexpr TStringBuf DefaultReportLable = "";
    constexpr TStringBuf EventLogField = "__event_log";
    constexpr TStringBuf HostField = "__host";

    const TString JsonContentType = "application/json";
}

TJsonReport::TJsonReport::TGuard::TGuard(IServerReportBuilder::TPtr report, int code)
    : TGuardBase(report, code)
    , JsonReport(MutableReportAs<TJsonReport>())
{
}

void TJsonReport::AddReportElement(TStringBuf name, NJson::TJsonValue&& elem, const bool allowUndefined) {
    if (!allowUndefined && !elem.IsDefined()) {
        return;
    }
    auto guard = MakeThreadSafeGuard();
    Report.SetValueByPath(name, std::move(elem));
}

void TJsonReport::AddReportElementString(TStringBuf name, TString&& elem) {
    Y_ASSERT(NJson::IsJson(elem));
    auto guard = MakeThreadSafeGuard();
    StrReports.emplace(name, std::move(elem));
}

void TJsonReport::SetExternalReportString(TString&& report) {
    Y_ASSERT(NJson::IsJson(report));
    auto guard = MakeThreadSafeGuard();
    StrReports.emplace(DefaultReportLable, std::move(report));
}

void TJsonReport::SetExternalReport(NJson::TJsonValue&& report) {
    auto guard = MakeThreadSafeGuard();
    Y_ASSERT(Report.IsNull() || (Report.IsMap() && report.IsMap()));
    if (Report.IsNull()) {
        Report = std::move(report);
    } else {
        for (auto&& [key, value] : report.GetMapSafe()) {
            Report.InsertValue(key, std::move(value));
        }
    }
}

bool TJsonReport::PrintReport(IOutputStream& so) const {
    auto defaultReport = StrReports.find(DefaultReportLable);
    if (defaultReport != StrReports.end()) {
        Y_ASSERT(NJson::IsJson(defaultReport->second));
        so << defaultReport->second;
        return true;
    }

    const NJson::TJsonValue::TMapType* mapInfo = nullptr;
    if (Report.IsDefined()) {
        if (!Report.GetMapPointer(&mapInfo)) {
            return false;
        }
    }

    NJson::TJsonWriter writer(&so, false, ShouldSortKeys());
    writer.OpenMap();
    if (mapInfo) {
        for (auto&& [key, value] : *mapInfo) {
            writer.Write(key, value);
        }
    }
    for (auto&& [key, value] : StrReports) {
        Y_ASSERT(NJson::IsJson(value));
        writer.UnsafeWrite(key, value);
    }
    writer.CloseMap();
    return true;
}

void TJsonReport::DoAddEvent(TInstant timestamp, NJson::TJsonValue&& e) {
    if (FirstEvent == TInstant::Zero()) {
        FirstEvent = timestamp;
    }
    if (!e.IsMap()) {
        NJson::TJsonValue ee = NJson::JSON_MAP;
        ee.InsertValue("event", std::move(e));
        e = std::move(ee);
    }
    e.InsertValue("_ts", timestamp.MicroSeconds());
    if (GetReportEventLog() && LastEvent != TInstant::Zero()) {
        e.InsertValue("delta", 0.001 * (timestamp - LastEvent).MicroSeconds());
    }
    if (GetReportEventLog()) {
        e.InsertValue("elapsed", 0.001 * (timestamp - FirstEvent).MicroSeconds());
    }
    EventLog.AppendValue(std::move(e));
    LastEvent = timestamp;
}

void TJsonReport::DoFinish(const TCodedException& e) {
    bool dumpEventLog = GetDumpEventLog();
    bool reportDebugInfo = GetReportDebugInfo();
    bool reportEventLog = reportDebugInfo && GetReportEventLog();
    if (reportEventLog) {
        Report.InsertValue(EventLogField, std::move(EventLog));
        Report.InsertValue(HostField, HostName());
    }
    if (e.GetCode() != HTTP_OK && e.HasReport()) {
        Report.InsertValue("error_details", e.GetReport(reportDebugInfo));
    }

    const auto& reqid = GetReqId();
    if (e.GetCode() != HTTP_OK && !reportDebugInfo) {
        Report.InsertValue("reqid", reqid);
        NDrive::TEventLog::Log("DebugInfo", e.GetReport(/*debug=*/true));
    }

    TBuffer report;
    if (!Report.IsNull() || StrReports.size()) {
        TBufferOutput output(report);
        if (!PrintReport(output)) {
            if (ShouldSortKeys()) {
                NJson::WriteJson(&output, &Report, /*formatOutput=*/false, /*sortKeys=*/true);
            } else {
                output << Report.GetStringRobust();
            }
        }
        output.Write('\n');
    }
    TStringBuf serialized = { report.data(), report.size() };
    Y_ASSERT(NJson::IsJson(serialized));
    DoFinish(e.GetCode(), JsonContentType, report, /*writeEventLog=*/false);
    {
        const auto reqidHash = FnvHash<ui32>(reqid);
        if (report.Size() <= 128 * 1024) {
            NDrive::TEventLog::Log(NDrive::TEventLog::Response, *Context, e.GetCode(), Report, serialized);
        } else if (report.Size() <= 1024 * 1024 && (reqidHash & 1)) {
            NDrive::TEventLog::Log(NDrive::TEventLog::CompressedResponse, *Context, e.GetCode(), Report, serialized);
        } else {
            NJson::TJsonValue data;
            data[EventLogField] = std::move(Report[EventLogField]);
            data["size"] = report.Size();
            NDrive::TEventLog::Log(NDrive::TEventLog::ResponseInfo, *Context, e.GetCode(), data);
        }
    }
    if (dumpEventLog && !reportEventLog) {
        NDrive::TEventLog::Log("EventLog", EventLog, e.GetCode());
    }
}

bool TJsonReport::ShouldSortKeys() const {
    return SortKeys || GetReportEventLog();
}
