#include "config.h"

#include <drive/backend/database/drive/proto/drive.pb.h>

#include <rtline/util/algorithm/container.h>

#include <util/string/join.h>

bool TUserPermissionsFeatures::DeserializeFromProto(const NDrive::NProto::TUserPermissionsFeatures& info) {
    IsPlusUser = info.GetIsPlusUser();
    IsYandexUser = info.GetIsYandexUser();
    ClientIP = info.GetClientIP();
    TVMTicket = info.GetTVMTicket();
    IsFallbackUser = info.GetIsFallbackUser();
    Uid = info.GetUid();
    DefaultEmails = { info.GetDefaultEmails().begin(), info.GetDefaultEmails().end() };
    if (info.HasUserLocation()) {
        UserLocation.ConstructInPlace();
        UserLocation->Deserialize(info.GetUserLocation());
    }
    if (info.HasMobilePay()) {
        MobilePay = info.GetMobilePay();
    }
    return true;
}

void TUserPermissionsFeatures::SerializeToProto(NDrive::NProto::TUserPermissionsFeatures& info) const {
    info.SetIsPlusUser(IsPlusUser);
    info.SetIsYandexUser(IsYandexUser);
    if (ClientIP) {
        info.SetClientIP(ClientIP);
    }
    if (TVMTicket) {
        info.SetTVMTicket(TVMTicket);
    }
    if (Uid) {
        info.SetUid(Uid);
    }
    info.SetIsFallbackUser(IsFallbackUser);
    for (const auto& mail : DefaultEmails) {
        info.AddDefaultEmails(mail);
    }
    if (UserLocation) {
        *info.MutableUserLocation() = UserLocation->Serialize();
    }
    if (MobilePay) {
        info.SetMobilePay(*MobilePay);
    }
}

TSet<TString> TRolesConfig::GetAdditionalRoles(const TUserPermissionsFeatures& userFeatures) const {
    TSet<TString> result;
    if (userFeatures.GetIsPlusUser()) {
        result.insert(PlusRoles.begin(), PlusRoles.end());
    }
    if (userFeatures.GetIsYandexUser()) {
        result.insert(YandexRoles.begin(), YandexRoles.end());
    }
    return result;
}

TSet<TString> TRolesConfig::GetStatusSpecificRoles(const TString& status) const {
    if (CustomRoles.contains(status)) {
        return CustomRoles.at(status);
    }
    return {};
}

void TRolesConfig::Init(const TYandexConfig::Section* section) {
    TVector<TString> necessaryRoles;
    if (section && section->GetDirectives().FillArray("NecessaryRoles", necessaryRoles)) {
        NecessaryRoles = MakeSet(necessaryRoles);
    } else {
        NecessaryRoles.emplace("GR_default_user_base");
    }
    TVector<TString> plusRoles;
    if (section && section->GetDirectives().FillArray("PlusRoles", plusRoles)) {
        PlusRoles = MakeSet(plusRoles);
    } else {
        PlusRoles.emplace("discount_plus");
    }
    TVector<TString> yandexRoles;
    if (section && section->GetDirectives().FillArray("YandexRoles", yandexRoles)) {
        YandexRoles = MakeSet(yandexRoles);
    } else {
        YandexRoles.emplace("yandex_user");
    }
    TVector<TString> blockedRoles;
    if (section && section->GetDirectives().FillArray("BlockedRoles", blockedRoles)) {
        BlockedRoles = MakeSet(blockedRoles);
    } else {
        BlockedRoles.emplace("default_user");
    }
    TVector<TString> defaultRoles;
    if (section && section->GetDirectives().FillArray("DefaultRoles", defaultRoles)) {
        DefaultRoles = MakeSet(defaultRoles);
    } else {
        DefaultRoles.emplace("default_user");
        DefaultRoles.emplace("user_access_base");
    }
    TVector<TString> baseRoles;
    if (section && section->GetDirectives().FillArray("BaseRoles", baseRoles)) {
        BaseRoles = MakeSet(baseRoles);
    } else {
        BaseRoles.emplace("onboarding_role");
    }
    TVector<TString> hiddenMTRoles;
    if (section && section->GetDirectives().FillArray("HiddenMTRoles", hiddenMTRoles)) {
        HiddenMTRoles = MakeSet(hiddenMTRoles);
    } else {
        HiddenMTRoles.emplace("hidden_manual_transmission");
    }
    TVector<TString> chatRoles;
    if (section && section->GetDirectives().FillArray("ChatRoles", chatRoles)) {
        ChatRoles = MakeSet(chatRoles);
    } else {
        ChatRoles.emplace("role_enable_chat");
    }
    TVector<TString> fastRegisteredRoles;
    if (section && section->GetDirectives().FillArray("FastRegisteredRoles", fastRegisteredRoles)) {
        FastRegisteredRoles = MakeSet(fastRegisteredRoles);
    } else {
        FastRegisteredRoles.emplace("default_user");
        FastRegisteredRoles.emplace("fast_registered_access");
    }
    TVector<TString> firstRidingRoles;
    if (section && section->GetDirectives().FillArray("FirstRidingRoles", firstRidingRoles)) {
        FirstRidingRoles = MakeSet(firstRidingRoles);
    } else {
        FirstRidingRoles.emplace("first_riding_role");
    }
    TVector<TString> firstPlusRidingRoles;
    if (section && section->GetDirectives().FillArray("FirstPlusRidingRoles", firstPlusRidingRoles)) {
        FirstPlusRidingRoles = MakeSet(firstPlusRidingRoles);
    }

    for (const auto& status : NDrive::AcceptableUserStatuses) {
        if (status.empty()) {
            continue;
        }
        TString directiveName = status;
        directiveName.front() = AsciiToUpper(directiveName.front());

        {
            directiveName += "SpecificRoles";
            TVector<TString> statusRoles;
            if (section && section->GetDirectives().FillArray(directiveName, statusRoles)) {
                CustomRoles[status] = MakeSet(statusRoles);
            }
        }
    }
}

void TRolesConfig::ToString(IOutputStream& os) const {
    os << "PlusRoles: " << JoinSeq(",", PlusRoles) << Endl;
    os << "YandexRoles: " << JoinSeq(",", YandexRoles) << Endl;
    os << "DefaultRoles: " << JoinSeq(",", DefaultRoles) << Endl;
    os << "BlockedRoles: " << JoinSeq(",", BlockedRoles) << Endl;
    os << "BaseRoles: " << JoinSeq(",", BaseRoles) << Endl;
    os << "HiddenMTRoles: " << JoinSeq(",", HiddenMTRoles) << Endl;
    os << "NecessaryRoles: " << JoinSeq(",", NecessaryRoles) << Endl;
    os << "FirstRidingRoles: " << JoinSeq(",", FirstRidingRoles) << Endl;
    os << "FirstPlusRidingRoles: " << JoinSeq(",", FirstPlusRidingRoles) << Endl;

    for (const auto& status : NDrive::AcceptableUserStatuses) {
        if (status.empty() || !CustomRoles.contains(status)) {
            continue;
        }
        TString directiveName = status;
        directiveName.front() = AsciiToUpper(directiveName.front());

        {
            directiveName += "SpecificRoles";
            os << directiveName << ": " << JoinSeq(",", CustomRoles.at(status)) << Endl;
        }
    }
}
