#include "manager.h"

#include <drive/backend/abstract/user_position_context.h>
#include <drive/backend/actions/disable.h>
#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/interfaces/account.h>
#include <drive/backend/billing/accounts_manager.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/experiments/context.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/offers/actions/abstract.h>

#include <library/cpp/threading/named_lock/named_lock.h>

#include <util/random/random.h>


class TUserRolesPool {
private:
    R_FIELD(TVector<TUserRole>, OwnRoles);
    R_FIELD(TSet<TString>, AdditionalRoles);
    R_READONLY(TVector<TUserRoleInfo>, RolesForUsage);
    R_READONLY(TSet<TString>, RoleIdsForUsage);

private:
    TSet<TString> DisabledRoles;
    TSet<TString> RecursiveDisabledRoles;
    TSet<TString> OwnedRoleIds;
    TString UserId;

public:
    TUserRolesPool(const TString& userId)
        : UserId(userId)
    {
    }

    TUserRolesPool& AddAdditional(const TSet<TString>& additional) {
        AdditionalRoles.insert(additional.begin(), additional.end());
        return *this;
    }

    TUserRolesPool& AddBanned(const TString& roleId) {
        DisabledRoles.insert(roleId);
        return *this;
    }
    TUserRolesPool& AddBannedRecursively(const TString& roleId) {
        RecursiveDisabledRoles.insert(roleId);
        return AddBanned(roleId);
    }
    TUserRolesPool& AddBanned(const TUserRole& role) {
        return AddBanned(role.GetRoleId());
    }

    TUserRolesPool& AddOwn(TConstArrayRef<TUserRole> roles, TInstant now) {
        for (auto&& i : roles) {
            if (!i.IsActive(now)) {
                AddBanned(i);
                continue;
            }
            if (OwnedRoleIds.insert(i.GetRoleId()).second) {
                OwnRoles.push_back(i);
            }
        }
        return *this;
    }

    const TSet<TString>& GetRecursiveDisabledRoles() const {
        return RecursiveDisabledRoles;
    }

    const TUserRolesPool& RoleIdsToSet(TSet<TString>& ids) const {
        ids.insert(AdditionalRoles.begin(), AdditionalRoles.end());
        for (auto&& i : OwnRoles) {
            ids.emplace(i.GetRoleId());
        }
        return *this;
    }

    TUserRolesPool& BuildRoles(const TMap<TString, TDriveRoleHeader>& fResult) {
        TSet<ui32> usedGroups;
        for (auto&& i : OwnRoles) {
            auto itEntity = fResult.find(i.GetRoleId());
            const TDriveRoleHeader* entity = (itEntity == fResult.end()) ? nullptr : &itEntity->second;
            if (entity && entity->GetGroup()) {
                usedGroups.emplace(entity->GetGroup());
            }
            RolesForUsage.emplace_back(TUserRoleInfo(i, entity));
            RoleIdsForUsage.emplace(i.GetRoleId());
        }
        for (auto&& i : AdditionalRoles) {
            if (DisabledRoles.contains(i) || OwnedRoleIds.contains(i)) {
                continue;
            }
            auto itEntity = fResult.find(i);
            const TDriveRoleHeader* entity = (itEntity == fResult.end()) ? nullptr : &itEntity->second;
            if (!entity || !usedGroups.contains(entity->GetGroup())) {
                if (entity) {
                    RolesForUsage.emplace_back(TUserRoleInfo(TUserRole(UserId, i), entity));
                }
                RoleIdsForUsage.emplace(i);
            }
        }
        return *this;
    }

    TUserRolesPool() = default;
};

TRolesManager::TRolesManager(const ITagsHistoryContext& context, TUsersDB& userDB, const TPropositionsManagerConfig& propositionsConfig, const THistoryConfig& historyConfig)
    : TDatabaseSessionConstructor(context.GetDatabase())
    , UserDB(userDB)
    , RolesDB(context, historyConfig)
    , RolesInfoDB(context, historyConfig)
    , ActionsDB(context, historyConfig, propositionsConfig)
    , SnapshotPropositions(context, propositionsConfig)
    , PermissionsCache(16 * 1024)
{
}

TRolesManager::~TRolesManager() {
    if (IsActive()) {
        Stop();
    }
}

bool TRolesManager::IsActive() const {
    return
        RolesDB.IsActive() ||
        ActionsDB.IsActive();
}

void TRolesManager::Start() {
    Y_ENSURE_BT(RolesDB.Start());
    Y_ENSURE_BT(ActionsDB.Start());
}

void TRolesManager::Stop() {
    if (!ActionsDB.Stop()) {
        ERROR_LOG << "cannot stop ActionsDB manager" << Endl;
    }
    if (!RolesDB.Stop()) {
        ERROR_LOG << "cannot stop RolesDB manager" << Endl;
    }
}

void RemoveDisabledActions(TVector<TDBAction>& actions) {
    TSet<TString> disabledActions;
    for (auto&& action : actions) {
        if (action->GetType() == TDisableAction::Type()) {
            if (const auto disableAction = action.GetAs<TDisableAction>()) {
                disabledActions.insert(disableAction->GetActions().begin(), disableAction->GetActions().end());
            } else {
                auto evlog = NDrive::GetThreadEventLogger();
                if (evlog) {
                    evlog->AddEvent(NJson::TMapBuilder
                        ("event", "CannotRemoveDisabledAction")
                        ("action_id", action->GetName())
                    );
                } else {
                    ERROR_LOG << "Cannot cast " << action->GetName() << " to disable_action" << Endl;
                }
            }
        }
    }
    const auto pred = [&disabledActions](const TDBAction& item) {
        return !item || disabledActions.contains(item->GetName());
    };
    actions.erase(std::remove_if(actions.begin(), actions.end(), pred), actions.end());
}

void AddActions(TVector<TDBAction>& result, TVector<TDBAction> additionalActions, const TString& userId, const bool removeDisabledActions = false) {
    const auto pred = [&userId](const TDBAction& item) -> bool {
        return !item || !item->GetEnabled() || (!!userId && !item->GetExperiment().CheckMatching(userId));
    };
    additionalActions.erase(std::remove_if(additionalActions.begin(), additionalActions.end(), pred), additionalActions.end());
    result.insert(result.end(), additionalActions.begin(), additionalActions.end());
    if (removeDisabledActions) {
        RemoveDisabledActions(result);
    }
}

bool TRolesManager::GetActions(const TSet<TString>& roles, const TInstant reqActuality, const TString& userId, const TInstant ts, TVector<TDBAction>& result, const TSet<TString>& disabledRoles, const bool removeDisabledActions) const {
    result.clear();
    TSet<TString> availableActionsNow = RolesInfoDB.GetActions(roles, reqActuality, ts, disabledRoles);

    TVector<TDBAction> additionalActions;
    if (!ActionsDB.GetCustomObjectsFromCache(additionalActions, availableActionsNow, reqActuality)) {
        return false;
    }

    AddActions(result, additionalActions, userId, removeDisabledActions);
    return true;
}

TOptionalAdditionalActions TRolesManager::GetUserAdditionalActions(const TString& userId, const IDriveTagsManager& tagsManager, bool getPotential) const {
    auto user = tagsManager.GetUserTags().GetCachedObject(userId);
    if (!user) {
        return {};
    }
    return GetAdditionalActions(*user, tagsManager, getPotential);
}

TOptionalAdditionalActions TRolesManager::GetUserAdditionalActions(const TString& userId, const IDriveTagsManager& tagsManager, bool getPotential, NDrive::TEntitySession& session) const {
    auto user = tagsManager.GetUserTags().RestoreObject(userId, session);
    if (!user) {
        return {};
    }
    return GetAdditionalActions(*user, tagsManager, getPotential);
}

TSet<TString> GetAccountIds(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts) {
    TSet<TString> accountIds;
    for (const auto& account : accounts) {
        if (!account) {
            continue;
        }

        accountIds.emplace(ToString(account->GetId()));
        if (account->GetParent()) {
            accountIds.emplace(ToString(account->GetParent()->GetId()));
        }
    }
    return accountIds;
}

TOptionalAdditionalActions TRolesManager::GetAccountsAdditionalActions(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts, TMap<TString, TTaggedObject>&& objects, const IDriveTagsManager& tagsManager, bool getPotential) const {
    auto actions = GetAdditionalActions(std::move(objects), tagsManager, getPotential);
    if (!actions) {
        return {};
    }
    TVector<TAdditionalAction> result;
    for (auto&& action : *actions) {
        auto accountStr = action.GetTag().GetObjectId();
        ui64 accountId = 0;
        if (!TryFromString(accountStr, accountId)) {
            continue;
        }
        TSet<TString> availableDescriptions;
        for (auto&& account : accounts) {
            if (!account) {
                continue;
            }
            if (account->GetId() == accountId || account->GetParent() && account->GetParent()->GetId() == accountId) {
                availableDescriptions.emplace(account->GetUniqueName());
            }
        }
        if (availableDescriptions.size()) {
            auto mutableAction = action.GetAction()->Clone();
            if (mutableAction) {
                if (auto builder = dynamic_cast<IOfferBuilderAction*>(mutableAction.Get())) {
                    builder->MutableChargableAccounts().insert(availableDescriptions.begin(), availableDescriptions.end());
                    builder->MutableOrganizationId() = accountId;
                }
                if (auto corrector = dynamic_cast<IOfferCorrectorAction*>(mutableAction.Get())) {
                    corrector->MutableChargableAccounts().insert(availableDescriptions.begin(), availableDescriptions.end());
                }
            }
            result.emplace_back(action.GetTag(), std::move(mutableAction));
        }
    }

    return result;
}

TOptionalAdditionalActions TRolesManager::GetAccountsAdditionalActions(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts, const IDriveTagsManager& tagsManager, bool getPotential) const {
    auto accountIds = GetAccountIds(accounts);
    TMap<TString, TTaggedObject> objects;
    for (const auto& id : accountIds) {
        auto object = tagsManager.GetAccountTags().GetCachedObject(id);
        if (!object) {
            return {};
        }
        objects.emplace(object->GetId(), *object);
    }
    return GetAccountsAdditionalActions(accounts, std::move(objects), tagsManager, getPotential);
}

TOptionalAdditionalActions TRolesManager::GetAccountsAdditionalActions(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts, const IDriveTagsManager& tagsManager, bool getPotential, NDrive::TEntitySession& session) const {
    auto accountIds = GetAccountIds(accounts);
    TMap<TString, TTaggedObject> objects;
    if (!tagsManager.GetAccountTags().RestoreObjects(accountIds, objects, session)) {
        return {};
    }
    return GetAccountsAdditionalActions(accounts, std::move(objects), tagsManager, getPotential);
}

TOptionalAdditionalActions TRolesManager::GetAdditionalActions(TMap<TString, TTaggedObject>&& objects, const IDriveTagsManager& tagsManager, bool getPotential) const {
    TVector<TAdditionalAction> result;
    for (const auto& [_, object] : objects) {
        auto actions = GetAdditionalActions(object, tagsManager, getPotential);
        if (!actions) {
            return {};
        }
        for (auto&& action : *actions) {
            result.push_back(std::move(action));
        }
    }
    return TAdditionalActions(std::move(result));
}

TOptionalAdditionalActions TRolesManager::GetAdditionalActions(const TTaggedObject& object, const IDriveTagsManager& tagsManager, bool getPotential) const {
    TVector<TAdditionalAction> result;
    if (object) {
        for (auto&& t : object.GetTags()) {
            const IUserActionTag* taTag = t.GetTagAs<IUserActionTag>();
            if (taTag) {
                auto actions = taTag->GetActions(t, tagsManager, *this, getPotential);
                {
                    for (auto&& action : actions) {
                        result.emplace_back(t, std::move(action));
                    }
                }
            }
        }
    }
    return TAdditionalActions(std::move(result));
}

TUserPermissions::TMutablePtr TRolesManager::BuildUsersPermissions(
    const TDriveUserData& userData,
    const TDriveUserRoles& userRoles,
    const TTaggedUser& userTags,
    const IDriveTagsManager& tagsManager,
    const TRolesConfig& rolesFeatures,
    const TUserPermissionsFeatures& userFeatures,
    TInstant reqActuality,
    IReplyContext::TPtr context,
    const TAreasDB* areasInfo,
    bool addCustomActions,
    const NDrive::NBilling::TAccountsManager* accountsManager
) const {
    TUserPermissions::TMutablePtr result;
    const TInstant nowTs = ModelingNow();
    const TString& userId = userData.GetUserId();
    NDrive::TExperimentContext experimentContext(context);

    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "BuildUsersPermissions")
            ("actuality", NJson::ToJson(reqActuality))
            ("now", NJson::ToJson(nowTs))
            ("user_id", userId)
        );
    }

    TMap<TString, TUserRolesPool> rolesByUser;
    do {
        TUserRolesPool& pool = rolesByUser[userId];
        pool.AddAdditional(rolesFeatures.GetChatRoles());

        {
            for (auto&& role : userRoles.GetRoles()) {
                if (!role.IsActive(nowTs)) {
                    pool.AddBanned(role);
                    continue;
                }
                if (role.IsForced()) {
                    pool.AddOwn({ role }, nowTs);
                    continue;
                }
            }
        }
        for (auto&& role : experimentContext.GetDisabledRoles()) {
            pool.AddBanned(role);
        }
        for (auto&& role : experimentContext.GetRecursiveDisabledRoles()) {
            pool.AddBannedRecursively(role);
        }
        pool.AddAdditional(experimentContext.GetEnabledRoles());

        if (
            userData.GetStatus() == NDrive::UserStatusBlocked ||
            userData.GetStatus() == NDrive::UserStatusDeleted ||
            userData.GetStatus() == NDrive::UserStatusRejected
        ) {
            pool.AddAdditional(rolesFeatures.GetBlockedRoles());
            continue;
        }

        if (!userData.IsStaffAccount() && !userData.IsMTAllowed()) {
            pool.AddAdditional(rolesFeatures.GetHiddenMTRoles());
        }

        {
            auto specificRoles =
                rolesFeatures.GetStatusSpecificRoles(userData.GetStatus());
            pool.AddAdditional(std::move(specificRoles));
        }

        if (!userData.IsStaffAccount() && (
            userData.GetStatus() == NDrive::UserStatusOnboarding ||
            userData.GetStatus() == NDrive::UserStatusScreening ||
            NDrive::IsPreOnboarding(userData.GetStatus())
        )) {
            pool.AddAdditional(rolesFeatures.GetBaseRoles());
            pool.AddAdditional(rolesFeatures.GetAdditionalRoles(userFeatures));
            continue;
        }

        if (!userRoles.GetRoles().empty()) {
            pool.AddOwn(userRoles.GetRoles(), nowTs);
        } else {
            if (userData.GetStatus() == NDrive::UserStatusActive) {
                pool.AddAdditional(rolesFeatures.GetDefaultRoles());
            }
        }

        if (userData.GetStatus() == NDrive::UserStatusActive) {
            pool.AddAdditional(rolesFeatures.GetNecessaryRoles());
            pool.AddAdditional(rolesFeatures.GetAdditionalRoles(userFeatures));
        }
        if (userData.GetStatus() == NDrive::UserStatusFastRegistered) {
            pool.AddAdditional(rolesFeatures.GetFastRegisteredRoles());
            pool.AddAdditional(rolesFeatures.GetAdditionalRoles(userFeatures));
        }
        if (userData.GetStatus() == NDrive::UserStatusFastRegistered && userData.IsFirstRiding()) {
            pool.AddAdditional({
                "fast_registered_active_access"
            });
        }
        if (userData.GetStatus() == NDrive::UserStatusActive && userData.IsFirstRiding()) {
            pool.AddAdditional(rolesFeatures.GetFirstRidingRoles());
        }
        if (userData.GetStatus() == NDrive::UserStatusActive && userData.IsFirstRiding() && userFeatures.GetIsPlusUser()) {
            pool.AddAdditional(rolesFeatures.GetFirstPlusRidingRoles());
        }
    } while (false);

    TSet<TString> roleIds;
    for (auto&& i : rolesByUser) {
        i.second.RoleIdsToSet(roleIds);
    }

    if (evlog) {
        evlog->AddEvent("GetRoleHeaders");
    }
    TMap<TString, TDriveRoleHeader> gRoleHeaders;
    R_ENSURE(RolesDB.GetCustomObjectsMap(roleIds, gRoleHeaders, reqActuality), HTTP_INTERNAL_SERVER_ERROR, "cannot get roles");
    for (auto&& [userId, roles] : rolesByUser) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "BuildRoles")
                ("user_id", userId)
            );
        }
        roles.BuildRoles(gRoleHeaders);
    }

    {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "BuildPermissions")
                ("user_id", userId)
            );
        }
        auto it = rolesByUser.find(userId);
        R_ENSURE(it != rolesByUser.end(), HTTP_INTERNAL_SERVER_ERROR, "cannot find rolesByUser");
        const TSet<TString>& userRoleIds = it->second.GetRoleIdsForUsage();
        const TVector<TUserRoleInfo>& userRolesUsage = it->second.GetRolesForUsage();
        const TSet<TString>& disabledRoles = it->second.GetRecursiveDisabledRoles();

        TVector<TDBAction> actualActions;
        TVector<TDBAction> enableActions;
        if (evlog) {
            evlog->AddEvent("GetActions");
        }
        R_ENSURE(GetActions(userRoleIds, reqActuality, userId, nowTs, actualActions, disabledRoles), HTTP_INTERNAL_SERVER_ERROR, "cannot get actual actions");
        R_ENSURE(GetActions(userRoleIds, reqActuality, userId, TInstant::Max(), enableActions, disabledRoles), HTTP_INTERNAL_SERVER_ERROR, "cannot get enabled actions");

        TVector<NDrive::NBilling::IBillingAccount::TPtr> accounts;
        if (accountsManager) {
            auto userAccounts = accountsManager->GetSortedUserAccounts(userId);
            for (auto&& account : userAccounts) {
                if (account->IsActual(Now())) {
                    accounts.emplace_back(std::move(account));
                }
            }
        }

        if (addCustomActions) {
            if (evlog) {
                evlog->AddEvent("GetUserAdditionalActions");
            }
            auto additionalUserActions = GetAdditionalActions(userTags, tagsManager, false);
            R_ENSURE(additionalUserActions, HTTP_INTERNAL_SERVER_ERROR, "cannot GetAdditionalActions for " << userId);
            auto additionalAccountActions = GetAccountsAdditionalActions(accounts, tagsManager, false);
            R_ENSURE(additionalAccountActions, HTTP_INTERNAL_SERVER_ERROR, "cannot GetAccountsAdditionalActions for " << userId);
            TVector<TDBAction> additionalActions;
            for (auto&& action : *additionalUserActions) {
                additionalActions.push_back(action.GetAction());
            }
            for (auto&& action : *additionalAccountActions) {
                additionalActions.push_back(action.GetAction());
            }
            AddActions(actualActions, std::move(additionalActions), userId);
        }

        if (evlog) {
            evlog->AddEvent("FilterActions");
        }
        TSet<TString> geoTagsForActions;
        for (auto&& actAction : actualActions) {
            if (!actAction) {
                continue;
            }
            geoTagsForActions.insert(actAction->GetUserAreaTagsFilter().GetCleanAreaTags().begin(), actAction->GetUserAreaTagsFilter().GetCleanAreaTags().end());
        }
        if (userFeatures.HasUserLocation() && areasInfo) {
            TSet<TString> geoActualTags = areasInfo->CheckGeoTags(userFeatures.GetUserLocationRef(), geoTagsForActions, reqActuality);
            const auto pred = [&geoActualTags](const TDBAction& item) -> bool {
                return !item || !item->GetUserAreaTagsFilter().FilterWeak(&geoActualTags);
            };
            actualActions.erase(std::remove_if(actualActions.begin(), actualActions.end(), pred), actualActions.end());
        } else {
            const auto pred = [](const TDBAction& item) -> bool {
                return !item || !item->GetUserAreaTagsFilter().FilterWeak(nullptr);
            };
            actualActions.erase(std::remove_if(actualActions.begin(), actualActions.end(), pred), actualActions.end());
        }
        if (context) {
            const auto pred = [&context](const TDBAction& item) {
                return !item || !item->CheckRequest(context);
            };
            actualActions.erase(std::remove_if(actualActions.begin(), actualActions.end(), pred), actualActions.end());
        }
        {
            const auto pred = [&](const TDBAction& item) {
                return !item || !item->CheckUser(userTags);
            };
            actualActions.erase(std::remove_if(actualActions.begin(), actualActions.end(), pred), actualActions.end());
        }
        {
            // disabled actions are not filtered yet, let's filter them now
            RemoveDisabledActions(actualActions);
        }

        if (evlog) {
            evlog->AddEvent("FillTagActionsAndEvolutions");
        }
        TMap<TTagAction::ETagAction, TSet<TTagDescription::TConstPtr>> tagsByActions;
        TMap<TString, TMap<TString, TTagEvolutionAction>> evolutions;
        auto tagActionMatchCache = ActionsDB.GetTagActionMatchCache(reqActuality);
        for (auto&& action : actualActions) {
            if (const TTagAction* tagAction = action.GetAs<TTagAction>()) {
                auto actionTags = ActionsDB.GetActionTags(tagAction->GetName(), tagActionMatchCache.Get());
                for (auto&& i : tagAction->GetTagActions()) {
                    auto& mapLocal = tagsByActions[i];
                    for (auto&& tDescription : actionTags) {
                        mapLocal.emplace(tDescription);
                    }
                }
            }
            if (const TTagEvolutionAction* tagAction = action.GetAs<TTagEvolutionAction>()) {
                TSet<std::pair<TString, TString>> evolveTags = ActionsDB.GetEvolutionTags(tagAction->GetName(), tagActionMatchCache.Get());
                for (auto&& i : evolveTags) {
                    if (tagAction->GetTwoSideEvolution()) {
                        evolutions[i.second][i.first] = tagAction->BuildReversed();
                    }
                    evolutions[i.first][i.second] = *tagAction;
                }
            }
        }
        result = TUserPermissions::Create(
            userData,
            userFeatures,
            actualActions,
            enableActions,
            userRolesUsage,
            tagsByActions,
            evolutions,
            std::move(accounts)
        );
    }
    return result;
}

TUserPermissions::TPtr TRolesManager::GetUserPermissions(
    const TString& userId,
    const IDriveTagsManager& tagsManager,
    const TRolesConfig& rolesFeatures,
    const TUserPermissionsFeatures& userFeatures,
    TInstant reqActuality,
    IReplyContext::TPtr context,
    const TAreasDB* areasInfo,
    bool addCustomActions,
    bool useCache,
    const NDrive::NBilling::TAccountsManager* accountsManager,
    bool forceFetchPermissions
) const {
    if (useCache) {
        auto eventLogger = NDrive::GetThreadEventLogger();
        auto lifetimeKey = TString{"user_permissions.cache_lifetime"};
        auto lifetime = NDrive::HasServer()
            ? NDrive::GetServer().GetSettings().GetValue<TDuration>(lifetimeKey)
            : Nothing();
        auto now = Now();
        auto defaultLifetime = TDuration::Seconds(42);
        auto rnd = TDuration::Seconds(RandomNumber<ui64>(lifetime.GetOrElse(defaultLifetime).Seconds() / 3));
        auto threshold = now - lifetime.GetOrElse(defaultLifetime) + rnd;
        auto cachedPermissions = PermissionsCache.find(userId).GetOrElse(nullptr);

        NNamedLock::TNamedLockPtr lock;
        bool cacheHit = cachedPermissions && cachedPermissions->GetTimestamp() > threshold;
        if (!cacheHit) {
            auto eg = NDrive::BuildEventGuard("GetUserPermissionsLock");
            if (lock = NNamedLock::AcquireLock(userId, reqActuality)) {
                cachedPermissions = PermissionsCache.find(userId).GetOrElse(nullptr);
                cacheHit = cachedPermissions && cachedPermissions->GetTimestamp() > threshold;
                if (cacheHit) {
                    lock.Reset();
                }
            }
        }

        if (cacheHit) {
            if (eventLogger) {
                eventLogger->AddEvent(NJson::TMapBuilder
                    ("event", "PermissionsCacheHit")
                    ("user_id", userId)
                    ("timestamp", NJson::ToJson(cachedPermissions->GetTimestamp()))
                );
            }
            return cachedPermissions;
        }

        if (eventLogger) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "PermissionsCacheMiss")
                ("user_id", userId)
            );
        }

        auto permissions = GetUserPermissions(userId, tagsManager, rolesFeatures, userFeatures, reqActuality, context, areasInfo, addCustomActions, /*useCache=*/false, accountsManager);
        if (permissions) {
            PermissionsCache.update(userId, permissions);
        }
        return permissions;
    } else {
        auto eg = NDrive::BuildEventGuard("GetUserPermissions");
        if (!forceFetchPermissions) {
            auto user = UserDB.GetCachedObject(userId);
            if (!user) {
                if (eg) {
                    eg->AddEvent(NJson::TMapBuilder
                        ("event", "UserNotFound")
                        ("user_id", userId)
                    );
                }
                return nullptr;
            }

            auto userRoles = UserDB.GetRoles().GetCachedUserRoles(userId);
            auto userTags = tagsManager.GetUserTags().GetCachedObject(userId);
            return BuildUsersPermissions(
                *user,
                *userRoles,
                *userTags,
                tagsManager,
                rolesFeatures,
                userFeatures,
                reqActuality,
                context,
                areasInfo,
                addCustomActions,
                accountsManager
            );
        } else {
            auto tx = BuildTx<NSQL::ReadOnly>();
            auto user = UserDB.Fetch(userId, tx);
            if (!user) {
                if (eg) {
                    eg->AddEvent(NJson::TMapBuilder
                        ("event", "UserNotFound")
                        ("user_id", userId)
                    );
                }
                return nullptr;
            }

            auto userRoles = UserDB.GetRoles().RestoreUserRoles(userId, tx);
            auto userTags = tagsManager.GetUserTags().RestoreObject(userId, tx);
            return BuildUsersPermissions(
                *user,
                *userRoles,
                *userTags,
                tagsManager,
                rolesFeatures,
                userFeatures,
                reqActuality,
                context,
                areasInfo,
                addCustomActions,
                accountsManager
            );
        }
    }
}

bool TRolesManager::ApplySnapshot(const TRoleSnapshot& snapshot, const TString& userId, NDrive::TEntitySession& session) const {
    if (!RolesDB.Upsert(snapshot.GetRole(), userId, session)) {
        return false;
    }
    RolesInfoDB.GetRoleActions().ApplySnapshot(snapshot.GetRole().GetRoleId(), snapshot.GetRoleActions().GetSlaves(), userId, session);
    RolesInfoDB.GetRoleRoles().ApplySnapshot(snapshot.GetRole().GetRoleId(), snapshot.GetRoleRoles().GetSlaves(), userId, session);
    return true;
}

bool TRolesManager::GetRoleSnapshots(const TSet<TString>& roleIds, TInstant actuality, TVector<TRoleSnapshot>& result) const {
    result.clear();
    TMap<TString, TDriveRoleRoles> gRoles;
    TMap<TString, TDriveRoleActions> gActions;
    if (!RolesInfoDB.GetRoleRoles().GetCustomObjectsMap(roleIds, gRoles, actuality)) {
        return false;
    }
    if (!RolesInfoDB.GetRoleActions().GetCustomObjectsMap(roleIds, gActions, actuality)) {
        return false;
    }
    TVector<TDBAction> actions;
    TSet<TString> actionKeys;
    for (auto&& i : gActions) {
        for (auto&& j : i.second.GetSlaves()) {
            actionKeys.emplace(j.GetSlaveObjectId());
        }
    }
    TSet<TString> roleKeys;
    for (auto&& i : gRoles) {
        for (auto&& j : i.second.GetSlaves()) {
            roleKeys.emplace(j.GetSlaveObjectId());
        }
    }
    if (!ActionsDB.GetCustomObjectsFromCache(actions, actionKeys, actuality)) {
        return false;
    }
    TMap<TString, TDBAction> actionsMap;
    for (auto&& i : actions) {
        actionsMap.emplace(i->GetName(), i);
    }
    roleKeys.insert(roleIds.begin(), roleIds.end());
    TMap<TString, TDriveRoleHeader> gRolesInfo;
    if (!RolesDB.GetCustomObjectsMap(roleKeys, gRolesInfo, actuality)) {
        return false;
    }

    for (auto&& i : roleIds) {
        TRoleSnapshot snapshot;
        snapshot.SetActions(actionsMap);
        snapshot.SetRoles(gRolesInfo);
        {
            auto itRoleInfo = gRolesInfo.find(i);
            if (itRoleInfo == gRolesInfo.end()) {
                continue;
            }
            snapshot.SetRole(itRoleInfo->second);
        }
        {
            auto itRoleInfo = gRoles.find(i);
            if (itRoleInfo != gRoles.end()) {
                snapshot.SetRoleRoles(itRoleInfo->second);
            }
        }
        {
            auto itActionInfo = gActions.find(i);
            if (itActionInfo != gActions.end()) {
                snapshot.SetRoleActions(itActionInfo->second);
            }
        }
        result.emplace_back(std::move(snapshot));
    }


    return true;
}
