#pragma once

#include "permissions.h"
#include "roles.h"

#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/threading/concurrent_cache.h>

namespace NDrive::NBilling {
    class IBillingAccount;
}

class TRoleSnapshotPropositions: public TPropositionsManager<TRoleSnapshot> {
public:
    TRoleSnapshotPropositions(const ITagsHistoryContext& context, const TPropositionsManagerConfig& propositionsConfig)
        : TPropositionsManager<TRoleSnapshot>(context, "role_snapshot_propositions", propositionsConfig)
    {
    }
};

class TAdditionalAction {
private:
    TDBTag AdditionalActionTag;
    TDBAction AdditionalAction;

public:
    TAdditionalAction(const TDBTag& additionalActionTag, TDBAction additionalAction)
        : AdditionalActionTag(additionalActionTag)
        , AdditionalAction(std::move(additionalAction))
    {
    }

    const TDBTag& GetTag() const {
        return AdditionalActionTag;
    }

    const TDBAction& GetAction() const {
        return AdditionalAction;
    }
};

class TAdditionalActions {
private:
    using TActions = TVector<TAdditionalAction>;

private:
    TActions Actions;

public:
    TAdditionalActions(TActions&& actions)
        : Actions(std::move(actions))
    {
    }

    TActions::const_iterator begin() const {
        return Actions.begin();
    }
    TActions::const_iterator end() const {
        return Actions.end();
    }
    TActions::iterator begin() {
        return Actions.begin();
    }
    TActions::iterator end() {
        return Actions.end();
    }

    template <class T>
    static TVector<TUserAction::TConstPtr> Filter(TAdditionalActions&& actions) {
        TVector<TUserAction::TConstPtr> result;
        for (auto&& action : actions) {
            auto impl = std::dynamic_pointer_cast<const T>(action.GetAction().Impl());
            if (impl) {
                result.push_back(std::move(impl));
            }
        }
        return result;
    }

    void Join(TAdditionalActions&& other) {
        for (auto&& action : other.Actions) {
            Actions.push_back(std::move(action));
        }
    }
};

using TOptionalAdditionalActions = TMaybe<TAdditionalActions>;

namespace NDrive::NBilling {
    class TAccountsManager;
}

class TRolesManager: public TDatabaseSessionConstructor {
private:
    using TBase = TDatabaseSessionConstructor;

private:
    TUsersDB& UserDB;
    mutable TRolesDB RolesDB;
    mutable TRoleInfoDB RolesInfoDB;
    mutable TActionsDB ActionsDB;
    mutable TRoleSnapshotPropositions SnapshotPropositions;

    mutable NUtil::TConcurrentCache<TString, TUserPermissions::TPtr> PermissionsCache;

    using TBase::Database;

public:
    TRolesManager(const ITagsHistoryContext& context, TUsersDB& userDB, const TPropositionsManagerConfig& actionPropositions, const THistoryConfig& historyConfig);
    ~TRolesManager();

    bool IsActive() const;
    void Start();
    void Stop();

    bool GetActions(const TSet<TString>& roles, const TInstant reqActuality, const TString& userId, const TInstant ts, TVector<TDBAction>& result, const TSet<TString>& disabledRoles = {}, const bool removeDisabledActions = false) const;

    TActionsDB& GetActionsDB() const {
        return ActionsDB;
    }

    TRolesDB& GetRolesDB() const {
        return RolesDB;
    }

    TRoleInfoDB& GetRolesInfoDB() const {
        return RolesInfoDB;
    }

    TRoleSnapshotPropositions& GetSnapshotPropositions() const {
        return SnapshotPropositions;
    }

    TVector<TDBAction> GetActions(TInstant reqActuality = TInstant::Zero()) const {
        TVector<TDBAction> result;
        Y_ENSURE_BT(ActionsDB.GetAllObjectsFromCache(result, reqActuality));
        return result;
    }

    TMap<TString, TDBAction> GetActions(const TSet<TString>& ids, TInstant actuality = TInstant::Zero()) const {
        TVector<TDBAction> actions;
        Y_ENSURE_BT(ActionsDB.GetCustomObjectsFromCache(actions, ids, actuality));
        TMap<TString, TDBAction> result;
        for (auto&& i : actions) {
            result.emplace(i->GetName(), i);
        }
        return result;
    }

    template <class T>
    TMap<TString, T> GetActionsAs(const TSet<TString>& ids, const TInstant reqActuality = TInstant::Zero()) const {
        TMap<TString, T> result;
        TVector<TDBAction> actions;
        Y_ENSURE_BT(ActionsDB.GetCustomObjectsFromCache(actions, ids, reqActuality));
        for (auto&& action : actions) {
            const T* actionT = action.GetAs<T>();
            if (actionT) {
                result.emplace(actionT->GetName(), *actionT);
            }
        }
        return result;
    }

    TMaybe<TDBAction> GetAction(const TString& name, TInstant actuality = TInstant::Zero()) const {
        TVector<TDBAction> actions;
        Y_ENSURE_BT(ActionsDB.GetCustomObjectsFromCache(actions, NContainer::Scalar(name), actuality));
        if (actions.empty()) {
            return {};
        }
        return actions.front();
    }

    TVector<TDriveRoleHeader> GetRoles(TInstant reqActuality = TInstant::Zero()) const {
        TVector<TDriveRoleHeader> result;
        Y_ENSURE_BT(RolesDB.GetAllObjectsFromCache(result, reqActuality));
        return result;
    }

    TVector<TDriveRoleHeader> GetRoles(const TSet<TString>& roleIds, TInstant reqActuality = TInstant::Zero()) const {
        return RolesDB.GetCachedObjectsVector(roleIds, reqActuality);
    }

    TVector<TDriveRoleHeader> GetRolesWithAction(const TString actionId, TInstant reqActuality = TInstant::Zero()) const {
        return GetRolesWithActions({ actionId }, reqActuality);
    }

    TVector<TDriveRoleHeader> GetRolesWithActions(const TSet<TString>& actionIds, TInstant reqActuality = TInstant::Zero()) const {
        TVector<TDriveRoleHeader> result;
        TSet<TString> roleIds;
        TVector<TDriveRoleActions> actionsByRole;
        Y_ENSURE_BT(GetRolesInfoDB().GetRoleActions().GetAllObjectsFromCache(actionsByRole, reqActuality));
        for (auto&& i : actionsByRole) {
            if (i.HasSlaves(actionIds)) {
                roleIds.emplace(i.GetRoleId());
            }
        }

        Y_ENSURE_BT(GetRolesDB().GetCustomObjectsFromCache(result, roleIds, reqActuality));
        return result;
    }

    TSet<TString> GetUsersWithTagAction(const TString& tagName, TTagAction::ETagAction tagAction, TInstant reqActuality = TInstant::Zero()) const {
        auto allActions = GetActions(reqActuality);
        TSet<TString> actionIds;
        for (auto&& action : allActions) {
            auto impl = action.GetAs<TTagAction>();
            if (!impl) {
                continue;
            }
            if (impl->Match(tagName) && impl->GetTagActions().contains(tagAction)) {
                actionIds.emplace(impl->GetName());
            }
        }

        auto roleHeaders = GetRolesWithActions(actionIds, reqActuality);
        TVector<TString> roleIds(roleHeaders.size());
        for (size_t i = 0; i < roleHeaders.size(); ++i) {
            roleIds[i] = roleHeaders[i].GetName();
        }

        TSet<TString> userIds;
        if (!UserDB.GetRoles().GetUsersWithAtLeastOneRoles(MakeSet(roleIds), userIds, true)) {
            ERROR_LOG << "GetUsersWithAtLeastOneRoles failed! returning empty set" << Endl;
            return {};
        }

        return userIds;
    }

    TSet<TString> GetPotentialTagAssignees(const TString& tagName, TInstant reqActuality = TInstant::Zero()) const {
        return GetUsersWithTagAction(tagName, TTagAction::ETagAction::AssigneeToPerform, reqActuality);
    }

    TSet<TString> GetPotentialTagPerformers(const TString& tagName, TInstant reqActuality = TInstant::Zero()) const {
        return GetUsersWithTagAction(tagName, TTagAction::ETagAction::Perform, reqActuality);
    }

    bool RemoveAction(const TString actionId, const TString& userId, NDrive::TEntitySession& session) const {
        return RemoveActions({actionId}, userId, session);
    }

    bool RemoveActions(const TVector<TString>& actions, const TString& userId, NDrive::TEntitySession& session) const {
        return ActionsDB.RemoveActions(actions, userId, session);
    }

    bool DeprecateActions(const TVector<TString>& actions, const TString& userId, NDrive::TEntitySession& session) const {
        return ActionsDB.DeprecateActions(actions, userId, session);
    }

    bool RemoveRole(const TString& roleName, const TString& userId, NDrive::TEntitySession& session) const {
        return RolesDB.RemoveRole(roleName, userId, session);
    }

    bool RemoveRoles(const TVector<TString>& roleNames, const TString& userId, NDrive::TEntitySession& session) const {
        for (auto&& i : roleNames) {
            if (!RemoveRole(i, userId, session)) {
                return false;
            }
        }
        return true;
    }

    bool UpsertRole(const TDriveRoleHeader& role, const TString& userId, NDrive::TEntitySession& session) const {
        return RolesDB.Upsert(role, userId, session);
    }

    bool UpsertRoleForUser(const TUserRole& userRole, const TString& userActor, NDrive::TEntitySession& session) const {
        return UserDB.GetRoles().Link(userRole, userActor, session);
    }

    bool RemoveRoleForUser(const TString& userId, const TString& roleName, const TString& userActor, NDrive::TEntitySession& session) const {
        return UserDB.GetRoles().Unlink(userId, roleName, userActor, session);
    }

    bool RemoveAllRolesForUser(const TString& userId, const TString& userActor, NDrive::TEntitySession& session) const {
        auto g = UserDB.GetRoles().RestoreUserRoles(userId, session);
        if (!g) {
            return false;
        }
        for (auto&& i : g->GetRoles()) {
            if (!UserDB.GetRoles().Unlink(userId, i.GetRoleId(), userActor, session)) {
                return false;
            }
        }
        return true;
    }

    template <class TContainer>
    bool RemoveRolesForUser(const TString& userId, const TContainer& roleNames, const TString& userActor, NDrive::TEntitySession& session) const {
        for (auto&& i : roleNames) {
            if (!RemoveRoleForUser(userId, i, userActor, session)) {
                return false;
            }
        }
        return true;
    }

    TOptionalAdditionalActions GetUserAdditionalActions(const TString& userId, const IDriveTagsManager& tagsManager, bool getPotential) const;
    TOptionalAdditionalActions GetUserAdditionalActions(const TString& userId, const IDriveTagsManager& tagsManager, bool getPotential, NDrive::TEntitySession& session) const;
    TOptionalAdditionalActions GetAccountsAdditionalActions(const TVector<TAtomicSharedPtr<NDrive::NBilling::IBillingAccount>>& accounts, const IDriveTagsManager& tagsManager, bool getPotential, NDrive::TEntitySession& session) const;
    TOptionalAdditionalActions GetAccountsAdditionalActions(const TVector<TAtomicSharedPtr<NDrive::NBilling::IBillingAccount>>& accounts, const IDriveTagsManager& tagsManager, bool getPotential) const;

    TUserPermissions::TMutablePtr BuildUsersPermissions(
        const TDriveUserData& userData,
        const TDriveUserRoles& userRoles,
        const TTaggedUser& userTags,
        const IDriveTagsManager& tagsManager,
        const TRolesConfig& rolesFeatures,
        const TUserPermissionsFeatures& userFeatures = {},
        TInstant actuality = TInstant::Zero(),
        IReplyContext::TPtr context = nullptr,
        const TAreasDB* areasInfo = nullptr,
        bool addAdditionalActions = true,
        const NDrive::NBilling::TAccountsManager* accountsManager = nullptr
    ) const;
    TUserPermissions::TPtr GetUserPermissions(
        const TString& userId,
        const IDriveTagsManager& tagsManager,
        const TRolesConfig& rolesFeatures,
        const TUserPermissionsFeatures& userFeatures = {},
        TInstant actuality = TInstant::Zero(),
        IReplyContext::TPtr context = nullptr,
        const TAreasDB* areasInfo = nullptr,
        bool addCustomActions = true,
        bool useCache = false,
        const NDrive::NBilling::TAccountsManager* accountsManager = nullptr,
        bool forceFetchPermissions = false
    ) const;

    bool ApplySnapshot(const TRoleSnapshot& snapshot, const TString& userId, NDrive::TEntitySession& session) const;
    bool GetRoleSnapshots(const TSet<TString>& roleIds, TInstant actuality, TVector<TRoleSnapshot>& result) const;

private:
    TOptionalAdditionalActions GetAccountsAdditionalActions(const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts, TMap<TString, TTaggedObject>&& objects, const IDriveTagsManager& tagsManager, bool getPotential) const;
    TOptionalAdditionalActions GetAdditionalActions(const TTaggedObject& user, const IDriveTagsManager& tagsManager, bool getPotential) const;
    TOptionalAdditionalActions GetAdditionalActions(TMap<TString, TTaggedObject>&& objects, const IDriveTagsManager& tagsManager, bool getPotential) const;
};
